/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter;

import java.io.File;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

/**
 * {@link AsyncConverter}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
/**
 * {@link AsyncExecutor}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
public class AsyncExecutor implements ICounter {

    final protected static int MAX_REQUEST_QEUE_SIZE = 16384;

    final protected static long CLEANUP_TIMEOUT_MILLIS = 600000;

    /**
     * {@link AsyncIdentifier}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.8.0
     */
    protected class AsyncIdentifier {

        protected AsyncIdentifier(@NonNull final String asyncHash) {
            super();

            m_asyncHash = asyncHash;
            m_startTimeMillis = System.currentTimeMillis();
        }

        /* (non-Javadoc)
         * @see java.lang.Object#hashCode()
         */
        @Override
        public int hashCode() {
            return (31 + ((m_asyncHash == null) ? 0 : m_asyncHash.hashCode()));
        }

        /* (non-Javadoc)
         * @see java.lang.Object#equals(java.lang.Object)
         */
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }

            if ((obj == null) || getClass() != obj.getClass()) {
                return false;
            }

            AsyncIdentifier other = (AsyncIdentifier) obj;

            if (m_asyncHash == null) {
                if (other.m_asyncHash != null) {
                    return false;
                }
            } else if (!m_asyncHash.equals(other.m_asyncHash)) {
                return false;
            }

            return true;
        }

        // - API ---------------------------------------------------------------

        /**
         * @return
         */
        protected String getAsyncHash() {
            return m_asyncHash;
        }

        /**
         * @return
         */
        protected long getStartTimeMillis() {
            return m_startTimeMillis;
        }

        // - Members -----------------------------------------------------------

        private String m_asyncHash = null;

        private long m_startTimeMillis = 0;
    }

    /**
     * {@link AsyncRunnable}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.8.0
     */
    protected class AsyncRunnable implements Runnable {

        /**
         * Initializes a new {@link AsyncRunnable}.
         * @param jobType
         * @param jobProperties
         */
        protected AsyncRunnable(@NonNull String jobType, @NonNull HashMap<String, Object> jobProperties) {
            m_jobType = jobType;
            m_jobProperties = new HashMap<>(jobProperties.size());
            m_jobProperties.putAll(jobProperties);

            final File inputFile = (File) jobProperties.get(Properties.PROP_INPUT_FILE);

            try (final InputStream inputStm = (InputStream) jobProperties.get(Properties.PROP_INPUT_STREAM)) {
                if (StringUtils.isNotEmpty(m_jobType) && ((null != inputFile) || (null != inputStm))) {
                    final File tempInputFile = ManagerBasics.createTempFile("oxasync", null);

                    if (null != tempInputFile) {
                        if (null != inputFile) {
                            FileUtils.copyFile(inputFile, tempInputFile);
                            m_jobProperties.put(Properties.PROP_INPUT_FILE, tempInputFile);
                            m_asyncId = implCalculateAsyncHash(null);
                        } else if (null != inputStm) {
                            FileUtils.copyInputStreamToFile(inputStm, tempInputFile);

                            // replace the original input content file temp. input file
                            m_jobProperties.remove(Properties.PROP_INPUT_STREAM);
                            m_jobProperties.put(Properties.PROP_INPUT_FILE, tempInputFile);
                            m_asyncId = implCalculateAsyncHash(null);
                        }
                    }
                }
            } catch (final Exception e) {
                ManagerBasics.logExcpImpl(m_manager, e);
            }

            if (!isValid()) {
                // valid input as well, if a remote cache hash input source is set
                final String remoteCacheHash = (String) jobProperties.get(Properties.PROP_REMOTE_CACHE_HASH);

                if (StringUtils.isNotEmpty(remoteCacheHash)) {
                    m_asyncId = implCalculateAsyncHash(remoteCacheHash);
                }
            }

            if (isValid()) {
                // ensure, that we don't run into an async recursion,
                // but also ensure, that the remote side still treats
                // our request as async request
                m_jobProperties.remove(Properties.PROP_ASYNC);
                m_jobProperties.put(Properties.PROP_REMOTE_ASYNC, Boolean.TRUE);

                final JobPriority jobPriority = (JobPriority) m_jobProperties.get(Properties.PROP_PRIORITY);

                if (null == jobPriority) {
                    m_jobProperties.put(Properties.PROP_PRIORITY, JobPriority.BACKGROUND);
                }
            } else {
                clear();
            }
        }

        /* (non-Javadoc)
         * @see java.lang.Runnable#run()
         */
        @Override
        public void run() {
            // reduce parent queue count by one
            decrement();

            if (isValid()) {
                logDebug("Documentconverter started asynchronous conversion", m_jobProperties);

                final HashMap<String, Object> resultProperties = new HashMap<>(8);
                IOUtils.closeQuietly(m_documentConverter.convert(m_jobType, m_jobProperties, resultProperties));

                // increment run counter by one
                m_ranCounter.incrementAndGet();

                logDebug("Documentconverter finished asynchronous conversion", m_jobProperties);
            }

            clear();
        }

        /**
         * @return
         */
        protected HashMap<String, Object> getJobProperties() {
            return m_jobProperties;
        }

        /**
         * @return
         */
        protected AsyncIdentifier getAsyncIdentifier() {
            return m_asyncId;
        }

        /**
         * @return
         */
        protected synchronized boolean isValid() {
            return (null != m_asyncId);
        }

        /**
         *
         */
        protected synchronized void clear() {
            if (null != m_jobProperties) {
                FileUtils.deleteQuietly((File) m_jobProperties.remove(Properties.PROP_INPUT_FILE));
            }

            m_jobType = null;
            m_jobProperties = null;
            m_asyncId = null;
        }

        /**
         * @param remoteCashHash
         * @return
         */
        private AsyncIdentifier implCalculateAsyncHash(final String remoteCashHash) {
            StringBuilder asyncHashBuilder = new StringBuilder();
            AsyncIdentifier ret = null;

            if (StringUtils.isNotEmpty(remoteCashHash)) {
                asyncHashBuilder.append(remoteCashHash).append('-');
            } else {
                final File inputFile = (File) m_jobProperties.get(Properties.PROP_INPUT_FILE);
                final String locale = (String) m_jobProperties.get(Properties.PROP_LOCALE);
                final StringBuilder fileHashBuilder = ManagerBasics.getFileHashBuilder(inputFile, locale);

                if (null != fileHashBuilder) {
                    asyncHashBuilder.append(m_jobType).append('-').
                        append(ManagerBasics.getFileHashBuilder(inputFile, locale).toString());
                } else {
                    asyncHashBuilder = null;
                }
            }

             if (null != asyncHashBuilder) {
                 asyncHashBuilder.append(((JobPriority) m_jobProperties.get(Properties.PROP_PRIORITY)).toString());
                 ret = new AsyncIdentifier(asyncHashBuilder.toString());
             }

            return ret;
        }

        // - Members -------------------------------------------------------

        protected String m_jobType = null;
        protected HashMap<String, Object> m_jobProperties = null;
        protected AsyncIdentifier m_asyncId = null;
    }

    /**
     * Initializes a new {@link AsyncExecutor}.
     * @param manager
     * @param maxThreadCount
     */
    public AsyncExecutor(@NonNull IDocumentConverter converter, @NonNull ManagerBasics manager, int maxThreadCount) {
        super();

        if (maxThreadCount > 0) {
            m_documentConverter = converter;
            m_manager = manager;
            m_requestExecutor = new ThreadPoolExecutor(maxThreadCount, maxThreadCount, 0, TimeUnit.MILLISECONDS, m_requestQueue);

            m_requestExecutor.setRejectedExecutionHandler(new RejectedExecutionHandler() {

                @Override
                public void rejectedExecution(Runnable runnable, ThreadPoolExecutor executor) {
                    if (null != runnable) {
                        ((AsyncRunnable) runnable).clear();
                    }
                }
            });

            // initialize timer to clean up run set
            final Runnable cleanupRunnable = new Runnable() {

                @Override
                public void run() {
                    final long curTimeMillis = System.currentTimeMillis();

                    synchronized (m_runSet) {
                        final Iterator<AsyncIdentifier> asyncIdIter = m_runSet.iterator();

                        while (asyncIdIter.hasNext()) {
                            final AsyncIdentifier curAsyncId = asyncIdIter.next();

                            if (null != curAsyncId) {
                                if ((curTimeMillis - curAsyncId.getStartTimeMillis()) > CLEANUP_TIMEOUT_MILLIS) {
                                    asyncIdIter.remove();
                                } else {
                                    // since the items are ordered by time, we
                                    // can stop the search when the first item
                                    // did not reach its timeout
                                    break;
                                }
                            }
                        }
                    }
                }
            };

            m_timerExecutor.scheduleAtFixedRate(cleanupRunnable, CLEANUP_TIMEOUT_MILLIS, CLEANUP_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS);
        }
    }

    // - ICounter --------------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.ICounter#increment()
     */
    @Override
    public void increment() {
        // ok
    }

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.ICounter#decrement()
     */
    @Override
    public void decrement() {
        // ok
    }

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.ICounter#getCount()
     */
    @Override
    public long getCount() {
        return m_requestQueue.size();
    }

    // - public API ------------------------------------------------------------

    public void terminate() {
        AsyncExecutor.AsyncRunnable curRunnable = null;

        isTerminated = true;

        while((curRunnable = (AsyncExecutor.AsyncRunnable) m_requestQueue.poll()) != null) {
            // decrement queue count by one
            decrement();
            curRunnable.clear();
        }
    }

    /**
     * @param jobType
     * @param jobProperties
     * @param resultProperties
     */
    public void triggerExecution(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        if (!isTerminated && (null != m_requestExecutor) && (null != jobType) && (null != jobProperties) && (null != resultProperties)) {
            final AsyncExecutor.AsyncRunnable asyncRunnable = new AsyncExecutor.AsyncRunnable(jobType, jobProperties);

            if (asyncRunnable.isValid()) {
                if (needsRunning(asyncRunnable)) {
                    logDebug("Documentconverter scheduled asynchronous job", asyncRunnable.getJobProperties());

                    // increment queue count by one
                    increment();
                    m_requestExecutor.execute(asyncRunnable);

                } else {
                    logDebug("Documentconverter already scheduled same asynchronous job => omitting request", asyncRunnable.getJobProperties());

                    m_omitCounter.incrementAndGet();
                    asyncRunnable.clear();
                }
            } else {
                if (ManagerBasics.isLogWarnImpl(m_manager)) {
                    ManagerBasics.logImpl(m_manager, LogType.LOGTYPE_WARN, "Documentconverter is not able to execute async. job without file input, stream input or cached hash value set => please set source input property accordingly", null);
                }
            }
        }
    }

    /**
     * @return
     */
    public LogData[] getLogData() {
        return new LogData[] {
            new LogData("async_jobs_remaining", Long.toString(getCount())),
            new LogData("async_jobs_ran", m_ranCounter.toString()),
            new LogData("async_job_omitted", m_omitCounter.toString())
        };
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param asyncRunnable
     * @return
     */
    protected boolean needsRunning(@NonNull final AsyncRunnable asyncRunnable) {
        final AsyncIdentifier asyncId = asyncRunnable.getAsyncIdentifier();
        boolean ret = true;

        synchronized (m_runSet) {
            if (m_runSet.remove(asyncId)) {
                ret = false;
            }

            // put entry at end of the set in every case
            m_runSet.add(asyncId);
        }


        return ret;
    }

    /**
     * @param text
     * @param jobProperties
     */
    protected void logDebug(@NonNull final String text, @NonNull final HashMap<String, Object> jobProperties) {
        if (ManagerBasics.isLogDebugImpl(m_manager)) {
            ManagerBasics.logImpl(m_manager, LogType.LOGTYPE_DEBUG, text, jobProperties, getLogData());
        }
    }

    // - Members -----------------------------------------------------------

    protected IDocumentConverter m_documentConverter = null;
    protected ManagerBasics m_manager = null;
    protected BlockingQueue<Runnable> m_requestQueue = new ArrayBlockingQueue<>(MAX_REQUEST_QEUE_SIZE);
    protected LinkedHashSet<AsyncIdentifier> m_runSet = new LinkedHashSet<>();
    protected ThreadPoolExecutor m_requestExecutor = null;
    protected ScheduledExecutorService m_timerExecutor = Executors.newScheduledThreadPool(1);
    protected boolean isTerminated = false;
    protected AtomicLong m_ranCounter = new AtomicLong(0);
    protected AtomicLong m_omitCounter = new AtomicLong(0);
}
