/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.impl;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.json.JSONObject;

import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.ContentTypeField;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMResourceAccessException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.ConflictResolution;
import com.openexchange.usm.api.session.SessionInitializer;
import com.openexchange.usm.util.BitSetEncoder;
import com.openexchange.usm.util.Toolkit;

/**
 * 
 * @author afe
 *
 */
public class PersistentSessionData implements Serializable {
	// Determines in ms. how often for constantly active sessions the internal cache of persistent data is updated (1/hour)
	private final static long UPDATE_INTERVAL = 60L * 60L * 1000L;

	private final static String CONFLICT_RESOLUTION = "ConflictResolution";
	private static final String CONTENT_TYPE_PREFIX = "CT_";
	public static final String FIELD_PREFIX = "F_";

	private static final int NUMBER_OF_SESSION_ID_FIELDS = 2;

	private static class SessionInformation implements Comparable<SessionInformation> {
		public final int _cid;
		public final int _id;
		public final int _usmSessionId;
		public final String _protocol;
		public final String _device;
		private List<String> _fieldData;

		public SessionInformation(int cid, int id, String protocol, String device, int usmSessionId) {
			_cid = cid;
			_id = id;
			_protocol = protocol;
			_device = device;
			_usmSessionId = usmSessionId;
		}

		@Override
		public String toString() {
			return "CID: " + _cid + ", ID: " + _id + ", Protocol: " + _protocol + ", Device: " + _device
					+ ", usmSessionId: " + _usmSessionId;
		}

		public void addFieldData(String data) {
			if (_fieldData == null)
				_fieldData = new ArrayList<String>();
			_fieldData.add(data);
		}

		public void appendFieldDataToList(List<String> list) {
			list.add(toString());
			if (_fieldData != null) {
				Collections.sort(_fieldData);
				list.addAll(_fieldData);
			}
		}

		public int compareTo(SessionInformation o) {
			int rv = _cid - o._cid;
			if (rv == 0) {
				rv = _protocol.compareTo(o._protocol);
				if (rv == 0) {
					rv = _id - o._id;
					if (rv == 0) {
						rv = _device.compareTo(o._device);
						if (rv == 0)
							rv = _usmSessionId - o._usmSessionId;
					}
				}
			}
			return rv;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + _cid;
			result = prime * result + ((_device == null) ? 0 : _device.hashCode());
			result = prime * result + _id;
			result = prime * result + ((_protocol == null) ? 0 : _protocol.hashCode());
			result = prime * result + _usmSessionId;
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (!(obj instanceof SessionInformation))
				return false;
			SessionInformation other = (SessionInformation) obj;
			if (_cid != other._cid)
				return false;
			if (_device == null) {
				if (other._device != null)
					return false;
			} else if (!_device.equals(other._device))
				return false;
			if (_id != other._id)
				return false;
			if (_protocol == null) {
				if (other._protocol != null)
					return false;
			} else if (!_protocol.equals(other._protocol))
				return false;
			if (_usmSessionId != other._usmSessionId)
				return false;
			return true;
		}
	}

	public static String[] getFieldsFromDB(SessionManagerImpl sessionManager, int cid, int id, String protocol,
			String device, String field) {
		if (cid != 0)
			return getFieldsFromContext(sessionManager, cid, id, protocol, device, field);
		List<String> result = new ArrayList<String>();
		try {
			for (int context : sessionManager.getDatabaseAccess().getAllContextIds()) {
				for (String s : getFieldsFromContext(sessionManager, context, id, protocol, device, field))
					result.add(s);
			}
			return result.toArray(new String[result.size()]);
		} catch (DatabaseAccessException e) {
			return new String[] { "An error has occurred while determining all context ids:", e.toString() };
		}
	}

	private static String[] getFieldsFromContext(SessionManagerImpl sessionManager, int cid, int id, String protocol,
			String device, String field) {
		try {
			EncapsulatedConnection con = sessionManager.getDatabaseAccess().getReadOnly(cid);
			PreparedStatement statement = null;
			ResultSet result = null;
			Map<Integer, SessionInformation> sessionIDs;
			try {
				sessionIDs = retrieveUSMSessionsFromDB(cid, id, protocol, device, con);
				if (sessionIDs.isEmpty())
					return new String[] { cid + ": >> No matching sessions <<" };
				statement = con
						.prepareStatement("SELECT usmSessionId, field, value FROM USMSessionFields WHERE cid = ? AND field like 'F%'");
				statement.setInt(1, cid);
				result = statement.executeQuery();
				field = isParameterSet(field) ? (FIELD_PREFIX + field) : null;
				while (result.next()) {
					int usmSessionId = result.getInt(1);
					SessionInformation d = sessionIDs.get(usmSessionId);
					if (d != null) {
						String f = result.getString(2);
						if (field == null || field.equals(f)) {
							d.addFieldData("    " + f.substring(FIELD_PREFIX.length()) + " = " + result.getString(3));
						}
					}
				}
			} finally {
				Toolkit.close(result);
				Toolkit.close(statement);
				Toolkit.close(con);
			}
			List<SessionInformation> sessionList = new ArrayList<SessionInformation>(sessionIDs.values());
			Collections.sort(sessionList);
			List<String> returnData = new ArrayList<String>();
			for (SessionInformation si : sessionList)
				si.appendFieldDataToList(returnData);
			return returnData.toArray(new String[returnData.size()]);
		} catch (Exception e) {
			return new String[] { cid + ": Error while retrieving session fields: " + e.getMessage(), e.toString() };
		}
	}

	public static String[] updateFieldsInDB(SessionManagerImpl sessionManager, int cid, int id, String protocol,
			String device, String field, String value) {
		if (!isParameterSet(field))
			throw new IllegalArgumentException("Parameter 'field' is required");
		if (cid != 0)
			return updateFieldsInContext(sessionManager, cid, id, protocol, device, field, value);
		try {
			List<String> result = new ArrayList<String>();
			for (int context : sessionManager.getDatabaseAccess().getAllContextIds()) {
				for (String s : updateFieldsInContext(sessionManager, context, id, protocol, device, field, value))
					result.add(s);
			}
			return result.toArray(new String[result.size()]);
		} catch (DatabaseAccessException e) {
			return new String[] { "An error has occurred while determining all context ids:", e.toString() };
		}
	}

	private static String[] updateFieldsInContext(SessionManagerImpl sessionManager, int cid, int id, String protocol,
			String device, String field, String value) {
		String dbField = FIELD_PREFIX + field;
		try {
			EncapsulatedConnection con = sessionManager.getDatabaseAccess().getWritable(cid);
			PreparedStatement statement = null;
			PreparedStatement insertStatement = null;
			ResultSet result = null;
			Map<Integer, SessionInformation> sessionIDs;
			try {
				sessionIDs = retrieveUSMSessionsFromDB(cid, id, protocol, device, con);
				if (sessionIDs.isEmpty())
					return new String[] { cid + ": >> No matching sessions <<" };
				statement = con
						.prepareStatement("DELETE FROM USMSessionFields WHERE cid = ? AND usmSessionId = ? AND field = ?");
				if (isParameterSet(value))
					insertStatement = con
							.prepareStatement("INSERT INTO USMSessionFields (cid, usmSessionId, Field, Value) VALUES(?,?,?,?)");
				for (int sessionId : sessionIDs.keySet()) {
					statement.setInt(1, cid);
					statement.setInt(2, sessionId);
					statement.setString(3, dbField);
					statement.executeUpdate();
					if (insertStatement != null) {
						insertStatement.setInt(1, cid);
						insertStatement.setInt(2, sessionId);
						insertStatement.setString(3, dbField);
						insertStatement.setString(4, value);
						insertStatement.executeUpdate();
					}
				}
			} finally {
				if (con != null) {
					Toolkit.close(result);
					Toolkit.close(statement);
					Toolkit.close(insertStatement);
					Toolkit.close(con);
				}
			}
			for (SessionImpl session : sessionManager.getSessionStorage().getSessionList()) {
				if (sessionIDs.containsKey(session.getSessionId())) {
					PersistentSessionData persistentData = session.getPersistentSessionData();
					if (insertStatement != null)
						persistentData._fields.put(field, value);
					else
						persistentData._fields.remove(field);
				}
			}
			return new String[] { cid + ": >> " + sessionIDs.size() + " sessions updated <<" };
		} catch (Exception e) {
			return new String[] { cid + ": Error while updating session fields: " + e.getMessage(), e.toString() };
		}
	}

	private static Map<Integer, SessionInformation> retrieveUSMSessionsFromDB(int cid, int id, String protocol,
			String device, EncapsulatedConnection con) throws SQLException {
		PreparedStatement statement = null;
		ResultSet result = null;
		try {
			StringBuilder selectSessionQuery = new StringBuilder(
					"SELECT cid, id, protocol, device, usmSessionId FROM USMSession WHERE cid = ?");
			if (id != 0)
				selectSessionQuery.append(" AND id = ?");
			if (isParameterSet(device))
				selectSessionQuery.append(" AND device = ?");
			if (isParameterSet(protocol))
				selectSessionQuery.append(" AND protocol = ?");
			statement = con.prepareStatement(selectSessionQuery.toString());
			int paramCount = 0;
			statement.setInt(++paramCount, cid);
			if (id != 0)
				statement.setInt(++paramCount, id);
			if (isParameterSet(device))
				statement.setString(++paramCount, device);
			if (isParameterSet(protocol))
				statement.setString(++paramCount, protocol);
			result = statement.executeQuery();
			Map<Integer, SessionInformation> sessionIDs = new HashMap<Integer, SessionInformation>();
			while (result.next()) {
				int usmSessionId = result.getInt(5);
				sessionIDs.put(
						usmSessionId,
						new SessionInformation(result.getInt(1), result.getInt(2), result.getString(3), result
								.getString(4), usmSessionId));
			}
			return sessionIDs;
		} finally {
			Toolkit.close(result);
			Toolkit.close(statement);
		}
	}

	private static boolean isParameterSet(String param) {
		return param != null && param.length() > 0 && !"String".equals(param);
	}

	private final SessionImpl _session;

	private volatile Map<ContentType, BitSet> _contentTypeFilter = new HashMap<ContentType, BitSet>();

	private volatile Map<String, String> _fields = new HashMap<String, String>();

	private volatile Map<String, String> _fieldView = Collections.unmodifiableMap(_fields);

	private ConflictResolution _conflictResolution = ConflictResolution.ERROR;

	private long _lastUpdateFromDB;

	public PersistentSessionData(SessionImpl sessionImpl) {
		_session = sessionImpl;
	}

	public void initialize(SessionInitializer initializer, JSONObject configuration) throws USMException {
		SessionManagerImpl manager = _session.getSessionManager();
		if (manager != null) {
			boolean hadFields = readPersistentData(manager, _contentTypeFilter, _fields);
			if (initializer != null) {
				if (!hadFields)
					initializer.initializeSession(_session, configuration);
				else
					initializer.reinitializeSession(_session, configuration);
			} else {
				if (!hadFields)
					// TODO Should we extract something from configuration ?
					setContentTypeFilter(manager.getContentTypeManager().getRegisteredContentTypes());
			}
		}
	}

	private void updatePersistentDataFromDB() {
		SessionManagerImpl manager = _session.getSessionManager();
		if (manager != null) {
			Map<ContentType, BitSet> contentTypeFilter = new HashMap<ContentType, BitSet>();
			Map<String, String> fields = new HashMap<String, String>();
			try {
				readPersistentData(manager, contentTypeFilter, fields);
				_contentTypeFilter = contentTypeFilter;
				_fields = fields;
				_fieldView = Collections.unmodifiableMap(_fields);
			} catch (DatabaseAccessException e) {
				throw new USMResourceAccessException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_DB_ERROR_3,
						"DB Access Exception on reading persistent fields", e);
			} catch (USMSQLException e) {
				throw new USMResourceAccessException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_DB_ERROR_4,
						"SQL Exception on reading persistent fields", e);
			}
		}
	}

	public boolean checkProperDBStorage() throws DatabaseAccessException, USMSQLException {
		EncapsulatedConnection con = _session.getReadOnlyDBConnection();
		PreparedStatement statement = null;
		boolean sessionFound = false;
		try {
			statement = con.prepareStatement("select usmSessionId from USMSession where "
					+ PersistentSessionDataSQLStatements.UNIQUE_SESSION_LONG_IDENTIFIER);

			statement.setInt(1, _session.getContextId());
			statement.setInt(2, _session.getUserIdentifier());
			statement.setString(3, _session.getDevice());
			statement.setString(4, _session.getProtocol());
			ResultSet result = statement.executeQuery();
			sessionFound = result.next();
		} catch (SQLException e) {
			String errorMessage = _session + " SQL error while retrieving session from DB";
			throw new USMSQLException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_DB_ERROR, errorMessage, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
		return sessionFound;
	}

	private boolean readPersistentData(SessionManagerImpl manager, Map<ContentType, BitSet> contentTypeFilter,
			Map<String, String> fields) throws DatabaseAccessException, USMSQLException {
		EncapsulatedConnection con = _session.getReadOnlyDBConnection();
		PreparedStatement statement = null;
		boolean hadFields = false;
		try {
			statement = con.prepareStatement("select usmSessionId from USMSession where "
					+ PersistentSessionDataSQLStatements.UNIQUE_SESSION_LONG_IDENTIFIER);

			statement.setInt(1, _session.getContextId());
			statement.setInt(2, _session.getUserIdentifier());
			statement.setString(3, _session.getDevice());
			statement.setString(4, _session.getProtocol());

			ResultSet result = statement.executeQuery();
			if (result.next()) {
				hadFields = true;
				//the session was existing
				int sessionId = result.getInt(1);
				statement.close();
				//initialize eventually existing fields
				statement = con.prepareStatement("select Field, Value from USMSessionFields where "
						+ PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER);

				_session.setUniqueSessionID(sessionId);
				addSessionIdenifierParamsToStatement(statement);
				result = statement.executeQuery();

				while (result.next()) {
					String field = result.getString(1);
					String value = result.getString(2);
					readPersistentEntry(manager, contentTypeFilter, fields, field, value);
				}

			} else {
				//the session was not existing in the db
				statement.close();
				statement = null;
				con.close();
				con = null;
				con = _session.getWritableDBConnection();
				int sessionId = _session
						.getSessionManager()
						.getDatabaseAccess()
						.getNextIdFromSequence(_session.getContextId(),
								PersistentSessionDataSQLStatements.SEQUENCE_TABLE_NAME);
				statement = con
						.prepareStatement("insert into USMSession (cid, id, device, protocol, usmSessionId) values(?,?,?,?,?)");
				statement.setInt(1, _session.getContextId());
				statement.setInt(2, _session.getUserIdentifier());
				statement.setString(3, _session.getDevice());
				statement.setString(4, _session.getProtocol());
				statement.setInt(5, sessionId);
				if (statement.executeUpdate() != 1) {
					throw new SQLException(_session + " Inserting session for user " + _session.getUserIdentifier()
							+ " in DB did not change any rows");
				}
				_session.setUniqueSessionID(sessionId);
			}
			_lastUpdateFromDB = System.currentTimeMillis();
		} catch (SQLException e) {
			String errorMessage = _session + " SQL error while retrieving persistent data from DB";
			throw new USMSQLException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER1, errorMessage, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
		return hadFields;
	}

	private void readPersistentEntry(SessionManagerImpl manager, Map<ContentType, BitSet> contentTypeFilter,
			Map<String, String> fieldMap, String field, String value) {
		if (CONFLICT_RESOLUTION.equals(field)) {
			try {
				_conflictResolution = ConflictResolution.valueOf(value);
			} catch (IllegalArgumentException iae) {
				_session.getSessionManager().getJournal()
						.error(_session + " Invalid ConflictResolution " + value + " in DB", iae);
			}
		} else if (field.startsWith(FIELD_PREFIX)) {
			fieldMap.put(field.substring(2), value);
		} else {
			if (field.startsWith(CONTENT_TYPE_PREFIX)) { // Check for ContentType
				String typeName = field.substring(CONTENT_TYPE_PREFIX.length());
				ContentType type = manager.getContentTypeManager().getContentType(typeName);
				if (type != null) {
					BitSet fields = validateFieldFilter(type, BitSetEncoder.fromBase64(value));
					contentTypeFilter.put(type, fields);
					return;
				}
			}
			manager.getJournal().warn(_session + " Unknown field " + field + " in DB");
		}
	}

	private void addSessionIdenifierParamsToStatement(PreparedStatement statement) throws SQLException {
		addSessionIdenifierParamsToStatement(statement, 1);
	}

	private void addSessionIdenifierParamsToStatement(PreparedStatement statement, int index) throws SQLException {
		statement.setInt(index++, _session.getContextId());
		statement.setInt(index, _session.getSessionId());
	}

	public ConflictResolution getConflictResolution() {
		return _conflictResolution;
	}

	public void setConflictResolution(ConflictResolution resolution) throws DatabaseAccessException, USMSQLException {
		if (resolution == null)
			throw new IllegalArgumentException("ConflictResolution not specified (null)");
		if (_conflictResolution != resolution) {
			insertOrUpdateDBField(CONFLICT_RESOLUTION, resolution.toString());
			_conflictResolution = resolution;
		}
	}

	public void setContentTypeFilter(ContentType... usedContentTypes) throws USMSQLException, DatabaseAccessException {
		for (Iterator<ContentType> i = _contentTypeFilter.keySet().iterator(); i.hasNext();) {
			removeContentTypeFilter(i, usedContentTypes);
		}
		for (ContentType newType : usedContentTypes) {
			if (!_contentTypeFilter.containsKey(newType)) {
				int fieldCount = newType.getFields().length;
				BitSet fields = new BitSet(fieldCount);
				fields.set(0, fieldCount);
				setFieldFilter(newType, fields);
			}
		}
	}

	private void removeContentTypeFilter(Iterator<ContentType> i, ContentType... usedContentTypes)
			throws USMSQLException, DatabaseAccessException {
		ContentType storedType = i.next();
		for (ContentType newType : usedContentTypes) {
			if (newType.equals(storedType))
				return;
		}
		removeContentTypeFilterFromDB(storedType);
		i.remove();
	}

	public void setFieldFilter(ContentType type, BitSet fieldSet) throws USMSQLException, DatabaseAccessException {
		BitSet fields = validateFieldFilter(type, fieldSet);
		if (fields == null || fields.isEmpty()) {
			if (_contentTypeFilter.containsKey(type)) {
				removeContentTypeFilterFromDB(type);
				_contentTypeFilter.remove(type);
			}
		} else {
			if (!fields.equals(_contentTypeFilter.get(type))) {
				insertOrUpdateDBField(CONTENT_TYPE_PREFIX + type.getID(), BitSetEncoder.toBase64(fields));
				_contentTypeFilter.put(type, fields);
			}
		}
	}

	private BitSet validateFieldFilter(ContentType type, BitSet f) {
		if (f == null || f.isEmpty())
			return f;
		ContentTypeField[] fields = type.getFields();
		BitSet result = new BitSet();
		int limit = Math.min(fields.length, f.length());
		for (int i = 0; i < limit; i++) {
			if ((areNegativeFieldIDsAllowed(type) ? true : fields[i].getFieldID() >= 0) && f.get(i))
				result.set(i);
		}
		if (!result.isEmpty()) {
			result.set(0); // Make sure that the first two fields (id and folder_id) are always retrieved
			result.set(1);
		}
		return result;
	}

	private boolean areNegativeFieldIDsAllowed(ContentType type) {
		return DefaultContentTypes.GROUPS_ID.equals(type.getID())
				|| DefaultContentTypes.RESOURCES_ID.equals(type.getID());
	}

	public void setField(String key, String value) throws USMSQLException, DatabaseAccessException {
		if (value == null || value.length() == 0) {
			if (_fields.containsKey(key)) {
				removeDBField(FIELD_PREFIX + key);
				_fields.remove(key);
			}
		} else {
			if (!value.equals(_fields.get(key))) {
				insertOrUpdateDBField(FIELD_PREFIX + key, value);
				_fields.put(key, value);
			}
		}
	}

	private void checkForPossibleDBUpdate() {
		if (System.currentTimeMillis() > _lastUpdateFromDB + UPDATE_INTERVAL)
			updatePersistentDataFromDB();
	}

	public String getField(String key) {
		checkForPossibleDBUpdate();
		String result = _fields.get(key);
		return result == null ? "" : result;
	}

	public Map<String, String> getPersistentFields() {
		checkForPossibleDBUpdate();
		return _fieldView;
	}

	private void removeContentTypeFilterFromDB(ContentType type) throws USMSQLException, DatabaseAccessException {
		removeDBField(CONTENT_TYPE_PREFIX + type.getID());
	}

	private void removeDBField(String key) throws USMSQLException, DatabaseAccessException {
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		try {
			con = _session.getWritableDBConnection();
			statement = con.prepareStatement("delete from USMSessionFields where "
					+ PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER + " and Field = ?");
			addSessionIdenifierParamsToStatement(statement);
			statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 1, key);
			if (statement.executeUpdate() != 1)
				_session.getSessionManager().getJournal()
						.warn(_session + " Deleting field " + key + " in DB did not change any rows");
		} catch (SQLException e) {
			String errorMessage = _session + " SQL error while updating field " + key;
			throw new USMSQLException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER2, errorMessage, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
	}

	public void removeAllDBFieldsForSession() throws USMSQLException, DatabaseAccessException {
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		try {
			con = _session.getWritableDBConnection();
			statement = con.prepareStatement("delete from USMSessionFields where "
					+ PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER);
			addSessionIdenifierParamsToStatement(statement);
			statement.executeUpdate();
			statement.close();
			statement = con.prepareStatement("delete from usmIdMapping where "
					+ PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER);
			addSessionIdenifierParamsToStatement(statement);
			statement.executeUpdate();
			statement.close();
			statement = con.prepareStatement("delete from USMSession where "
					+ PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER);
			addSessionIdenifierParamsToStatement(statement);
			if (statement.executeUpdate() != 1)
				_session.getSessionManager().getJournal()
						.warn(_session + " Deleting session in DB did not change any rows");
		} catch (SQLException e) {
			String errorMessage = _session + " SQL error while deleting fields ";
			throw new USMSQLException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER4, errorMessage, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
	}

	private void insertOrUpdateDBField(String key, String value) throws DatabaseAccessException, USMSQLException {
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		try {
			con = _session.getWritableDBConnection();
			con.setAutoCommit(false);
			statement = con.prepareStatement("select Field from USMSessionFields where "
					+ PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER + " and Field = ?");
			addSessionIdenifierParamsToStatement(statement);
			statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 1, key);
			ResultSet resultSet = statement.executeQuery();
			if (resultSet.next()) {
				statement.close();
				statement = con.prepareStatement("update USMSessionFields set Value = ? where "
						+ PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER + " and Field = ?");
				statement.setString(1, value);
				addSessionIdenifierParamsToStatement(statement, 2);
				statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 2, key);
			} else {
				statement.close();
				statement = con
						.prepareStatement("insert into USMSessionFields (cid, usmSessionId, Field, Value) values(?,?,?,?)");
				addSessionIdenifierParamsToStatement(statement);
				statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 1, key);
				statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 2, value);
			}
			if (statement.executeUpdate() != 1)
				_session.getSessionManager().getJournal()
						.warn(_session + " Setting field " + key + " in DB did not change any rows");
			con.commit();
		} catch (SQLException e) {
			String errorMessage = _session + " SQL error while updating field " + key;
			_session.getSessionManager().getJournal().error(errorMessage, e);
			try {
				con.rollback();
			} catch (SQLException e1) {
				_session.getSessionManager().getJournal().error(_session + " SQL error: rollback on error failed", e1);
			}
			throw new USMSQLException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER3, errorMessage, e);
		} finally {
			if (con != null) {
				Toolkit.close(statement);
				try {
					con.setAutoCommit(true);
				} catch (SQLException e) {
					_session.getSessionManager().getJournal()
							.error(_session + " SQL error: setAutocommit(false) failed", e);
				}
				Toolkit.close(con);
			}
		}
	}

	public BitSet getFieldFilter(ContentType contentType) {
		return _contentTypeFilter.get(contentType);
	}

	public ContentType[] getContentTypes() {
		Set<ContentType> keySet = _contentTypeFilter.keySet();
		return keySet.toArray(new ContentType[keySet.size()]);
	}
}
