/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.api.datatypes;

import org.json.*;

import com.openexchange.usm.api.session.Session;

/**
 * Special DataType that implements an Array of another DataType
 * 
 * @author afe
 *
 */
public class ArrayDataType implements DataType<Object[]> {
	private static final Object[] EMPTY_ARRAY = new Object[0];

	protected final DataType<?> _elementType;

	public ArrayDataType(DataType<?> elementType) {
		_elementType = elementType;
	}

	public void addToJSONArray(Session session, JSONArray parent, Object value) throws JSONException {
		parent.put(generateJSONArray(session, (Object[]) value));
	}

	public Object[] checkValue(Object o) {
		if (o == null)
			return EMPTY_ARRAY;
		if (o instanceof Object[]) {
			Object[] array = (Object[]) o;
			for (int i = 0; i < array.length; i++) {
				array[i] = _elementType.checkValue(array[i]);
			}
			return array;
		}
		try {
			return new Object[] { _elementType.checkValue(o) };
		} catch (IllegalArgumentException e) {
			throw new IllegalArgumentException(o.getClass().getName() + " is neither " + getName()
					+ " or its element type");
		}
	}

	public Object[] createCopy(Object value) {
		Object[] source = (Object[]) value;
		if (source.length == 0)
			return source;
		Object[] result = new Object[source.length];
		for (int i = 0; i < result.length; i++) {
			result[i] = _elementType.createCopy(source[i]);
		}
		return result;
	}

	public Object[] createNewObject() {
		return EMPTY_ARRAY;
	}

	public Object[] extractFromJSONArray(Session session, JSONArray array, int index) throws JSONException {
		if (array.isNull(index))
			return createNewObject();
		return extractFromJSONArray(session, array.getJSONArray(index));
	}

	public Object[] extractFromJSONObject(Session session, JSONObject jsonObject, String field) throws JSONException {
		if (jsonObject.isNull(field))
			return createNewObject();
		return extractFromJSONArray(session, jsonObject.getJSONArray(field));
	}

	public String getName() {
		return "Array(" + _elementType.getName() + ')';
	}

	public void storeInJSONObject(Session session, JSONObject parent, String field, Object value) throws JSONException {
		parent.put(field, generateJSONArray(session, (Object[]) value));
	}

	private JSONArray generateJSONArray(Session session, Object[] value) throws JSONException {
		JSONArray array = new JSONArray();
		for (Object o : value)
			_elementType.addToJSONArray(session, array, o);
		return array;
	}

	private Object[] extractFromJSONArray(Session session, JSONArray array) throws JSONException {
		Object[] result = new Object[array.length()];
		for (int i = 0; i < result.length; i++)
			result[i] = _elementType.extractFromJSONArray(session, array, i);
		return result;
	}

	public boolean isEqual(Object v1, Object v2) {
		Object[] value1 = (Object[]) v1;
		Object[] value2 = (Object[]) v2;
		if (value1.length != value2.length)
			return false;
		for (int i = 0; i < value1.length; i++) {
			if (!_elementType.isEqual(value1[i], value2[i]))
				return false;
		}
		return true;
	}

	public boolean isDefaultValue(Object object) {
		Object[] value = (Object[]) object;
		return value.length == 0;
	}

	public DataType<?> getElementType() {
		return _elementType;
	}
}
