/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import static com.google.common.collect.ImmutableList.toImmutableList;
import static com.openexchange.util.custom.base.NullUtil.build;
import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.notNull;
import static com.openexchange.util.custom.base.NullUtil.setBuilder;
import static java.util.stream.Collectors.joining;
import java.lang.annotation.Annotation;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceReference;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;


/**
 * {@link ServiceDependencyTemplate}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.3.5
 */
public abstract class ServiceDependencyTemplate<S> implements ServiceDependency<S> {
    
    private final class ServiceClassQualifier implements ServiceDependencyQualifier {

        @Override
        public @Nullable String contributeFilter() {
            for (final Class<?> indirection : RegistrationTools.KNOWN_INDIRECTIONS) {
                if (indirection.isAssignableFrom(serviceClass)) {
                    return f("(|(%s=%s)(%s=%s))",
                        Constants.OBJECTCLASS, serviceClass.getName(),
                        Constants.OBJECTCLASS, indirection.getName()
                    );
                }
            }
            return f("(%s=%s)", Constants.OBJECTCLASS, serviceClass.getName());
        }

        @Override
        public @Nullable String contributeIdentification() {
            return f("%s", serviceClass.getName());
        }
        
        @Override
        public boolean matchesService(final Object serviceInstance, final ServiceReference<?> reference) {
            return ServiceDependencyTemplate.this.serviceClass.isInstance(serviceInstance);
        }
        
        @Override
        public boolean matchesDependency(Class<?> serviceClass, @Nullable Annotation[] annotations) {
            return ServiceDependencyTemplate.this.serviceClass.isAssignableFrom(serviceClass);
        }

        @Override
        public @Nullable Set<String> contributePropertiesOfInterest() {
            return null;
        }
         
    }
    
    private static final class WrapButFirst implements Function<String, String> {
        private final String before;
        private final String after;
        private int index = 0;
        public WrapButFirst(String before, String after) {
            this.before = before;
            this.after = after;
        }
        
        @Override
        public @Nullable String apply(@Nullable String input) {
            index++;
            if (index == 1) {
                return input;
            } else if (input == null) {
                return null;
            } else {
                return new StringBuilder(input.length() + before.length() + after.length())
                    .append(before).append(input).append(after).toString();
            }
        }
        
    }
    
    private final String identification;
    private final String filter;
    private final Class<S> serviceClass;
    private final ImmutableList<ServiceDependencyQualifier> qualifiers;
    private final ImmutableSet<String> propertiesOfInterest;
    
    protected ServiceDependencyTemplate(Class<S> serviceClass, ServiceDependencyQualifier... qualifiers) {
        super();
        this.serviceClass = serviceClass;
        {
            final ImmutableList.Builder<ServiceDependencyQualifier> b = ImmutableList.builder();
            b.add(new ServiceClassQualifier());
            b.add(qualifiers);
            this.qualifiers = build(b);
        }
        
        this.identification = notNull(this.qualifiers
            .stream()
            .map(ServiceDependencyQualifier::contributeIdentification)
            .filter(Objects::nonNull)
            .map(new WrapButFirst("[", "]"))
            .collect(joining()));
        {
            final ImmutableList<String> filterParts = this.qualifiers.stream()
                .map(ServiceDependencyQualifier::contributeFilter)
                .filter(Objects::nonNull)
                .collect(toImmutableList());
            if (filterParts.isEmpty()) {
                throw new IllegalArgumentException("empty OSGi filter expression");
            }
            if (filterParts.size() == 1) {
                this.filter = notNull(filterParts.get(0));
            } else {
                this.filter = notNull(new StringBuilder()
                    .append("(&")
                    .append(filterParts.stream().collect(joining()))
                    .append(")")
                    .toString());
            }
        }
        {
            final ImmutableSet.Builder<String> b = setBuilder();
            for (final ServiceDependencyQualifier qualifier : this.qualifiers) {
                final Set<String> p = qualifier.contributePropertiesOfInterest();
                if (p != null) {
                    b.addAll(p);
                }
            }
            this.propertiesOfInterest = build(b);
        }
    }


    @Override
    public final String identify() {
        return identification;
    }
    
    
    @Override
    public final String asFilter() {
        return filter;
    }

    @Override
    public final Set<String> propertiesOfInterest() {
        return propertiesOfInterest;
    }

    @Override
    public final boolean matchesService(final Object serviceInstance, final ServiceReference<?> reference) {
        for (final ServiceDependencyQualifier qualifier : qualifiers) {
            if (! qualifier.matchesService(serviceInstance, reference)) {
                return false;
            }
        }
        return true;
    }
    
    @Override
    public final boolean matchesDependency(final Class<?> serviceClass, final @Nullable Annotation[] annotations) {
        for (final ServiceDependencyQualifier qualifier : qualifiers) {
            if (! qualifier.matchesDependency(serviceClass, annotations)) {
                return false;
            }
        }
        return true;
    }
    
    @Override
    public @Nullable String toString() {
        return identification;
    }
    
}
