/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.logger;


import javax.management.ObjectName;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import org.slf4j.Logger;
import com.openexchange.annotation.Nullable;
import com.openexchange.management.ManagementService;

/**
 * {@link MBeanRegistrar}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.2
 */
public final class MBeanRegistrar<S, T extends S> implements ServiceTrackerCustomizer<ManagementService, ManagementService> {
    
    private static final Logger LOG = logger(MBeanRegistrar.class);
    
    private final BundleContext bundleContext;
    private final ActivatorRegistry registry;
    private final ObjectName objectName;
    private final Class<S> mbeanInterface;
    private final T mbean;

    public MBeanRegistrar(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final ObjectName objectName,
        final Class<S> mbeanInterface,
        final T mbean) {
        this.bundleContext = bundleContext;
        this.registry = registry;
        this.objectName = objectName;
        this.mbeanInterface = mbeanInterface;
        this.mbean = mbean;
    }

    @Override
    public ManagementService addingService(@Nullable ServiceReference<ManagementService> reference) {
        final ManagementService management = bundleContext.getService(reference);
        if (management != null) {
            try {
                management.registerMBean(objectName, mbean);
                registry.add(new MBeanRegistration(management, objectName, mbean));
            } catch (final Exception e) {
                LOG.warn(String.format("could not register MBean %s (%s) as '%s'",
                    className(mbean), mbeanInterface.getName(), objectName), e);
            } finally {
                bundleContext.ungetService(reference);
            }
            return management;
        } else {
            throw new IllegalStateException("ManagementService reference is null");
        }
    }

    @Override
    public void modifiedService(@Nullable ServiceReference<ManagementService> reference, @Nullable ManagementService management) {
        if (reference != null) {
            bundleContext.ungetService(reference);
        }
    }

    @Override
    public void removedService(@Nullable ServiceReference<ManagementService> reference, @Nullable ManagementService management) {
        if (reference != null) {
            bundleContext.ungetService(reference);
        }
    }
}
