/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.custom.capabilities;

import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;


import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;
import com.openexchange.capabilities.Capability;
import com.openexchange.serverconfig.ClientServerConfigFilter;


/**
 * {@link DisableCapabilitiesClientServerConfigFilter}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.2.0
 */
public class DisableCapabilitiesClientServerConfigFilter implements ClientServerConfigFilter {

    private static final Function<Object, Capability> TO_CAPABILITY = new Function<Object, Capability>() {
        @Override
        public @Nullable Capability apply(final @Nullable Object input) {
            if (input == null) {
                return null;
            }
            return new Capability(input.toString());
        }
        @Override
        public boolean equals(@Nullable Object obj) {
            return super.equals(obj);
        }
        @Override
        public int hashCode() {
            return "TO_CAPABILITY".hashCode();
        }
    };

    private static final Function<Capability, String> TO_ID = new Function<Capability, String>() {
        @Override
        public @Nullable String apply(final @Nullable Capability input) {
            if (input == null) {
                return null;
            }
            return input.getId();
        }
        @Override
        public boolean equals(@Nullable Object obj) {
            return super.equals(obj);
        }
        @Override
        public int hashCode() {
            return "TO_ID".hashCode();
        }
    };
    
    @SuppressWarnings("null")
    private static final Logger LOG = LoggerFactory.getLogger(DisableCapabilitiesClientServerConfigFilter.class);
    
    @Override
    public void apply(final @Nullable Map<String, Object> config) {
        final Set<Capability> removed = new HashSet<>();
        if (config != null) {
            final ImmutableSet<Capability> capsToDisable;
            {
                final Object obj = config.remove("disableCapabilities");
                if (obj != null) {
                    if (obj instanceof String) {
                        capsToDisable = ImmutableSet.of(new Capability((String) obj));
                    } else if (obj instanceof Collection) {
                        final Collection<?> collection = (Collection<?>) obj;
                        capsToDisable = collection.stream()
                            .map(TO_CAPABILITY)
                            .collect(ImmutableSet.toImmutableSet());
                    } else if (obj instanceof Iterable) {
                        final Iterable<?> iterable = (Iterable<?>) obj;
                        // https://stackoverflow.com/a/23936723
                        capsToDisable = StreamSupport.stream(iterable.spliterator(), false)
                            .map(TO_CAPABILITY)
                            .collect(ImmutableSet.toImmutableSet());
                    } else {
                        LOG.error("unsupported format for 'disableCapabilities': {}", obj.getClass().getName());
                        capsToDisable = ImmutableSet.of();
                    }
                } else {
                    capsToDisable = ImmutableSet.of();
                }
            }
            
            if (! capsToDisable.isEmpty()) {
                @SuppressWarnings("unchecked")
                final Set<Capability> caps = (Set<Capability>) config.get("capabilities");
                if (caps != null) {
                    for (final Capability capToDisable : capsToDisable) {
                        if (caps.remove(capToDisable)) {
                            removed.add(capToDisable);
                        }
                    }
                }
            }
        }
        if (LOG.isDebugEnabled() && (! removed.isEmpty())) {
            LOG.debug("removed {} capabilities: [{}]", removed.size(), removed.stream().map(TO_ID).collect(Collectors.joining(", ")));
        }
    }

}
