/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.logger;


import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import org.slf4j.Logger;
import com.openexchange.annotation.Nullable;
import com.openexchange.osgi.SimpleRegistryListener;

/**
 * {@link SimpleRegistryListenerTrackerCustomizer}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.2
 * @param <S>
 */
public final class SimpleRegistryListenerTrackerCustomizer<S> implements ServiceTrackerCustomizer<S, S> {

    private static final Logger LOG = logger(SimpleRegistryListenerTrackerCustomizer.class);
    private final SimpleRegistryListener<S> listener;
    private final BundleContext context;

    public SimpleRegistryListenerTrackerCustomizer(final SimpleRegistryListener<S> listener, final BundleContext context) {
        this.listener = listener;
        this.context = context;
    }

    @Override
    public @Nullable S addingService(final @Nullable ServiceReference<S> serviceReference) {
        if (serviceReference == null) {
            throw new IllegalArgumentException("ServiceReference parameter is null");
        }
        final S service = context.getService(serviceReference);
        if (service == null) {
            throw new IllegalStateException("ServiceReference resolved to null");
        }
        try {
            listener.added(serviceReference, service);
            return service;
        } catch (final Exception e) {
            context.ungetService(serviceReference);
            LOG.warn("Adding service ({}) to listener failed. Service released.", className(service), e);
            return null;
        }
    }

    @Override
    public void modifiedService(final @Nullable ServiceReference<S> serviceReference, final @Nullable S service) {
        // Don't care
    }

    @Override
    public void removedService(final @Nullable ServiceReference<S> serviceReference, final @Nullable S service) {
        if (serviceReference != null) {
            try {
                if (service != null) {
                    listener.removed(serviceReference, service);
                }
            } finally {
                context.ungetService(serviceReference);
            }
        }
    }
}
