/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.emptySet;
import static com.openexchange.util.custom.base.NullUtil.immutableCopyOf;
import static com.openexchange.util.custom.base.NullUtil.notNull;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;
import com.openexchange.util.activator.ServiceDependencies;


/**
 * {@link ImmutableServiceDependencies}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.2
 */
public final class ImmutableServiceDependencies implements ServiceDependencies {
    
    private static final ImmutableServiceDependencies EMPTY = new ImmutableServiceDependencies(
        emptySet(),
        emptySet(),
        emptySet(),
        emptySet()
    );
    
    public static ServiceDependencies empty() {
        return EMPTY;
    }
    
    public static ServiceDependencies copyOf(final ServiceDependencies... serviceDependencies) {
        return copyOf(notNull(Arrays.asList(serviceDependencies)));
    }

    public static ServiceDependencies copyOf(final Iterable<ServiceDependencies> serviceDependencies) {
        final Set<MandatoryServiceDependency<?>> mandatoryServiceDependenciesBuilder = new HashSet<>();
        final Set<OptionalServiceDependency<?>> optionalServiceDependenciesBuilder = new HashSet<>();
        final Set<Class<?>> serviceSetsBuilder = new HashSet<>();
        final Set<Class<?>> serviceListingsBuilder = new HashSet<>();
        for (final ServiceDependencies d : serviceDependencies) {
            mandatoryServiceDependenciesBuilder.addAll(d.mandatoryServiceDependencies());
            optionalServiceDependenciesBuilder.addAll(d.optionalServiceDependencies());
            serviceSetsBuilder.addAll(d.serviceSetClasses());
            serviceListingsBuilder.addAll(d.serviceListingClasses());
        }
        return new ImmutableServiceDependencies(
            immutableCopyOf(mandatoryServiceDependenciesBuilder),
            immutableCopyOf(optionalServiceDependenciesBuilder),
            immutableCopyOf(serviceSetsBuilder),
            immutableCopyOf(serviceListingsBuilder)
        );
    }
    
    private final ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies; 
    private final ImmutableSet<OptionalServiceDependency<?>> optionalDependencies;
    private final ImmutableSet<Class<?>> serviceSets;
    private final ImmutableSet<Class<?>> serviceListings;

    public ImmutableServiceDependencies(
        ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies,
        ImmutableSet<OptionalServiceDependency<?>> optionalDependencies,
        ImmutableSet<Class<?>> serviceSets,
        ImmutableSet<Class<?>> serviceListings) {
        this.mandatoryDependencies = mandatoryDependencies;
        this.optionalDependencies = optionalDependencies;
        this.serviceSets = serviceSets;
        this.serviceListings = serviceListings;
    }

    @Override
    public ImmutableSet<MandatoryServiceDependency<?>> mandatoryServiceDependencies() {
        return mandatoryDependencies;
    }

    @Override
    public ImmutableSet<OptionalServiceDependency<?>> optionalServiceDependencies() {
        return optionalDependencies;
    }
    
    @Override
    public ImmutableSet<Class<?>> serviceSetClasses() {
        return serviceSets;
    }

    @Override
    public ImmutableSet<Class<?>> serviceListingClasses() {
        return serviceListings;
    }
    
    @Override
    public boolean isEmpty() {
        return mandatoryDependencies.isEmpty() && optionalDependencies.isEmpty() && serviceSets.isEmpty() && serviceListings.isEmpty();
    }


    @Override
    public int hashCode() {
        return Objects.hash(this.getClass(), mandatoryDependencies, optionalDependencies, serviceSets, serviceListings);
    }

    @Override
    public boolean equals(final @Nullable Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableServiceDependencies other = (ImmutableServiceDependencies) obj;
        return Objects.equals(mandatoryDependencies, other.mandatoryDependencies)
            && Objects.equals(optionalDependencies, other.optionalDependencies)
            && Objects.equals(serviceSets, other.serviceSets)
            && Objects.equals(serviceListings, other.serviceListings)
            ;
    }
    
    @Override
    public @Nullable String toString() {
        return MoreObjects.toStringHelper(this)
        .add("mandatoryDependencies", mandatoryDependencies)
        .add("optionalDependencies", optionalDependencies)
        .add("serviceSets", serviceSets)
        .add("serviceListings", serviceListings)
        .toString();
    }
    
}
