/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.util.activator.ServiceNotAvailableException;
import com.openexchange.util.activator.ServiceRef;

public final class AtomicServiceRef<E> implements ServiceRef<E>, UpdatableServiceRef<E> {
    
    public static <X> AtomicServiceRef<X> forClass(final Class<X> theClass) {
        return new AtomicServiceRef<X>(theClass);
    }

    private final AtomicReference<E> ref;
    private final Class<E> refClass;
    
    public AtomicServiceRef(final Class<E> refClass) {
        this.ref = new AtomicReference<>(null);
        this.refClass = refClass;
    }

    @Override
    public boolean isPresent() {
        final E e = ref.get();
        return e != null;
    }

    @Override
    public E get() throws ServiceNotAvailableException {
        final E e = ref.get();
        if (e != null) {
            return e;
        } else {
            throw new ServiceNotAvailableException(refClass);
        }
    }

    @Override
    public @Nullable E tryGet() {
        return ref.get();
    }

    @Override
    public Set<E> asSet() {
        final E e = ref.get();
        if (e != null) {
            @SuppressWarnings("null")
            final @NonNull Set<E> result = ImmutableSet.<E>of(e);
            return result;
        } else {
            @SuppressWarnings("null")
            final @NonNull Set<E> result = ImmutableSet.<E>of();
            return result;
        }
    }
    
    @Override
    public Optional<E> asOptional() {
        @SuppressWarnings("null")
        final @NonNull Optional<E> result = Optional.fromNullable(ref.get());
        return result;
    }

    @Override
    public void set(final E service) {
        ref.set(service);
    }

    @Override
    public void unset() {
        ref.set(null);
    }
    
    @Override
    public Class<E> serviceRefClass() {
        return refClass;
    }

}
