/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.custom.capabilities;

import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;


import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;
import com.openexchange.capabilities.Capability;
import com.openexchange.serverconfig.ClientServerConfigFilter;


/**
 * {@link DisableCapabilitiesClientServerConfigFilter}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.2.0
 */
public class DisableCapabilitiesClientServerConfigFilter implements ClientServerConfigFilter {

    private static final Function<Object, Capability> TO_CAPABILITY = new Function<Object, Capability>() {
        @Override
        public @Nullable Capability apply(final @Nullable Object input) {
            if (input == null) {
                return null;
            }
            return new Capability(input.toString());
        }
        @Override
        public boolean equals(@Nullable Object obj) {
            return super.equals(obj);
        }
        @Override
        public int hashCode() {
            return "TO_CAPABILITY".hashCode();
        }
    };

    private static final Function<Capability, String> TO_ID = new Function<Capability, String>() {
        @Override
        public @Nullable String apply(final @Nullable Capability input) {
            if (input == null) {
                return null;
            }
            return input.getId();
        }
        @Override
        public boolean equals(@Nullable Object obj) {
            return super.equals(obj);
        }
        @Override
        public int hashCode() {
            return "TO_ID".hashCode();
        }
    };
    
    @SuppressWarnings("null")
    private static final Logger LOG = LoggerFactory.getLogger(DisableCapabilitiesClientServerConfigFilter.class);
    
    @Override
    public void apply(final @Nullable Map<String, Object> config) {
        final Set<Capability> removed = new HashSet<>();
        if (config != null) {
            final ImmutableSet<Capability> capsToDisable;
            {
                final Object obj = config.remove("disableCapabilities");
                if (obj != null) {
                    if (obj instanceof String) {
                        capsToDisable = ImmutableSet.of(new Capability((String) obj));
                    } else if (obj instanceof Collection) {
                        final Collection<?> collection = (Collection<?>) obj;
                        capsToDisable = collection.stream()
                            .map(TO_CAPABILITY)
                            .collect(ImmutableSet.toImmutableSet());
                    } else if (obj instanceof Iterable) {
                        final Iterable<?> iterable = (Iterable<?>) obj;
                        // https://stackoverflow.com/a/23936723
                        capsToDisable = StreamSupport.stream(iterable.spliterator(), false)
                            .map(TO_CAPABILITY)
                            .collect(ImmutableSet.toImmutableSet());
                    } else {
                        LOG.error("unsupported format for 'disableCapabilities': {}", obj.getClass().getName());
                        capsToDisable = ImmutableSet.of();
                    }
                } else {
                    capsToDisable = ImmutableSet.of();
                }
            }
            
            if (! capsToDisable.isEmpty()) {
                @SuppressWarnings("unchecked")
                final Set<Capability> caps = (Set<Capability>) config.get("capabilities");
                if (caps != null) {
                    for (final Capability capToDisable : capsToDisable) {
                        if (caps.remove(capToDisable)) {
                            removed.add(capToDisable);
                        }
                    }
                }
            }
        }
        if (LOG.isDebugEnabled() && (! removed.isEmpty())) {
            LOG.debug("removed {} capabilities: [{}]", removed.size(), removed.stream().map(TO_ID).collect(Collectors.joining(", ")));
        }
    }

}
