/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.emptySet;
import static com.openexchange.util.custom.base.NullUtil.immutableCopyOf;
import static com.openexchange.util.custom.base.NullUtil.notNull;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;
import com.openexchange.util.activator.ServiceDependencies;


/**
 * {@link ImmutableServiceDependencies}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.2
 */
public final class ImmutableServiceDependencies implements ServiceDependencies {
    
    private static final ImmutableServiceDependencies EMPTY = new ImmutableServiceDependencies(
        emptySet(),
        emptySet(),
        emptySet(),
        emptySet()
    );
    
    public static ServiceDependencies empty() {
        return EMPTY;
    }
    
    public static ServiceDependencies copyOf(final ServiceDependencies... serviceDependencies) {
        return copyOf(notNull(Arrays.asList(serviceDependencies)));
    }

    public static ServiceDependencies copyOf(final Iterable<ServiceDependencies> serviceDependencies) {
        final Set<MandatoryServiceDependency<?>> mandatoryServiceDependenciesBuilder = new HashSet<>();
        final Set<OptionalServiceDependency<?>> optionalServiceDependenciesBuilder = new HashSet<>();
        final Set<Class<?>> serviceSetsBuilder = new HashSet<>();
        final Set<Class<?>> serviceListingsBuilder = new HashSet<>();
        for (final ServiceDependencies d : serviceDependencies) {
            mandatoryServiceDependenciesBuilder.addAll(d.mandatoryServiceDependencies());
            optionalServiceDependenciesBuilder.addAll(d.optionalServiceDependencies());
            serviceSetsBuilder.addAll(d.serviceSetClasses());
            serviceListingsBuilder.addAll(d.serviceListingClasses());
        }
        return new ImmutableServiceDependencies(
            immutableCopyOf(mandatoryServiceDependenciesBuilder),
            immutableCopyOf(optionalServiceDependenciesBuilder),
            immutableCopyOf(serviceSetsBuilder),
            immutableCopyOf(serviceListingsBuilder)
        );
    }
    
    private final ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies; 
    private final ImmutableSet<OptionalServiceDependency<?>> optionalDependencies;
    private final ImmutableSet<Class<?>> serviceSets;
    private final ImmutableSet<Class<?>> serviceListings;

    public ImmutableServiceDependencies(
        ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies,
        ImmutableSet<OptionalServiceDependency<?>> optionalDependencies,
        ImmutableSet<Class<?>> serviceSets,
        ImmutableSet<Class<?>> serviceListings) {
        this.mandatoryDependencies = mandatoryDependencies;
        this.optionalDependencies = optionalDependencies;
        this.serviceSets = serviceSets;
        this.serviceListings = serviceListings;
    }

    @Override
    public ImmutableSet<MandatoryServiceDependency<?>> mandatoryServiceDependencies() {
        return mandatoryDependencies;
    }

    @Override
    public ImmutableSet<OptionalServiceDependency<?>> optionalServiceDependencies() {
        return optionalDependencies;
    }
    
    @Override
    public ImmutableSet<Class<?>> serviceSetClasses() {
        return serviceSets;
    }

    @Override
    public ImmutableSet<Class<?>> serviceListingClasses() {
        return serviceListings;
    }
    
    @Override
    public boolean isEmpty() {
        return mandatoryDependencies.isEmpty() && optionalDependencies.isEmpty() && serviceSets.isEmpty() && serviceListings.isEmpty();
    }


    @Override
    public int hashCode() {
        return Objects.hash(this.getClass(), mandatoryDependencies, optionalDependencies, serviceSets, serviceListings);
    }

    @Override
    public boolean equals(final @Nullable Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableServiceDependencies other = (ImmutableServiceDependencies) obj;
        return Objects.equals(mandatoryDependencies, other.mandatoryDependencies)
            && Objects.equals(optionalDependencies, other.optionalDependencies)
            && Objects.equals(serviceSets, other.serviceSets)
            && Objects.equals(serviceListings, other.serviceListings)
            ;
    }
    
    @Override
    public @Nullable String toString() {
        return MoreObjects.toStringHelper(this)
        .add("mandatoryDependencies", mandatoryDependencies)
        .add("optionalDependencies", optionalDependencies)
        .add("serviceSets", serviceSets)
        .add("serviceListings", serviceListings)
        .toString();
    }
    
}
