/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.custom.template;

import static com.openexchange.util.custom.base.NullUtil.build;
import static com.openexchange.util.custom.base.NullUtil.emptyMap;
import static com.openexchange.util.custom.base.NullUtil.mapBuilder;
import static com.openexchange.util.custom.base.NullUtil.tableBuilder;
import java.io.StringWriter;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.concurrent.Immutable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableTable;
import com.openexchange.annotation.Nullable;
import com.openexchange.configuration.ConfigurationExceptionCodes;
import com.openexchange.exception.OXException;
import com.openexchange.templating.OXTemplate;
import com.openexchange.templating.TemplateService;


/**
 * {@link TemplateFacades}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.3
 */
public enum TemplateFacades {
    ;
    
    @SuppressWarnings("null")
    private static final Logger LOG = LoggerFactory.getLogger(TemplateFacades.class);
    
    @Immutable
    private static final class ValidTemplateFacade implements TemplateFacade {
        private final ImmutableTable<String, String, OXTemplate> byTemplateByLocale;
        private final ImmutableMap<String, OXTemplate> fallbackTemplates;
        public ValidTemplateFacade(final ImmutableTable<String, String, OXTemplate> byTemplateByLocale,
            final ImmutableMap<String, OXTemplate> fallbackTemplates) {
            this.byTemplateByLocale = byTemplateByLocale;
            this.fallbackTemplates = fallbackTemplates;
        }
        @Override
        public boolean isValid() {
            return true;
        }
        @Override
        public String render(final String template, final @Nullable String locale, final Object... keyvalues) throws OXException {
            final OXTemplate t;
            {
                if (locale == null) {
                    t = fallbackTemplates.get(template);
                } else {
                    if (byTemplateByLocale.contains(template, locale)) {
                        t = byTemplateByLocale.get(template, locale);
                    } else {
                        t = fallbackTemplates.get(template);
                    }
                }
            }
            final ImmutableMap<String, Object> vars = strMap(keyvalues);
            final StringWriter sw = new StringWriter();
            t.process(vars, sw);
            return notNull(sw.toString());            
        }
    }    

    private static final class InvalidTemplateFacade implements TemplateFacade {
        private final OXException error;
        public InvalidTemplateFacade(OXException error) {
            this.error = error;
        }
        @Override
        public boolean isValid() {
            return false;
        }
        @Override
        public String render(final String template, final @Nullable String locale, final Object... keyvalues) throws OXException {
            throw error;
        }
    }
    
    public static final TemplateFacade create(final TemplateService ts, final String... templateNames) {
        final ImmutableSet<String> templates = ImmutableSet.copyOf(templateNames);
        
        final ImmutableSet<String> all;
        try {
            all = ImmutableSet.copyOf(notNull(ts.getBasicTemplateNames()));
        } catch (final OXException e) {
            LOG.error("failed to retrieve basic template names", e);
            return new InvalidTemplateFacade(e);
        }

        final ImmutableMap.Builder<String, OXTemplate> fallbackBuilder = mapBuilder();
        final ImmutableTable.Builder<String, String, OXTemplate> templateBuilder = tableBuilder();
        for (final String template : templates) {
            if (template != null) {
                final String fallbackTemplateName = template + ".tmpl";
                
                // look for the fallback template first, that must always exist
                if (! all.contains(fallbackTemplateName)) {
                    // if even one of the fallback templates is not available, make it fail
                    LOG.error("missing fallback template '{}'", fallbackTemplateName);
                    return new InvalidTemplateFacade(notNull(ConfigurationExceptionCodes.FILE_NOT_FOUND.create(fallbackTemplateName)));
                }
                {
                    final OXTemplate t;
                    try {
                        t = ts.loadTemplate(fallbackTemplateName);
                    } catch (final Exception e) {
                        LOG.error(String.format("failed to load fallback template '%s'", fallbackTemplateName), e);
                        return new InvalidTemplateFacade(notNull(ConfigurationExceptionCodes.FILE_NOT_FOUND
                            .create(e, fallbackTemplateName)));
                    }
                    if (t != null) {
                        fallbackBuilder.put(template, t);
                    } else {
                        LOG.error(String.format("failed to load fallback template '%s'", fallbackTemplateName));
                        return new InvalidTemplateFacade(notNull(ConfigurationExceptionCodes.FILE_NOT_FOUND
                            .create(fallbackTemplateName)));
                    }
                }
                
                for (final String filename : all) {
                    if (filename != null) {
                        final Matcher m = Pattern.compile("^\\Q" + template + ".\\E([^\\.]+)\\.tmpl$").matcher(filename);
                        if (m.matches()) {
                            final String lang = m.group(1);
                            if (lang != null) {
                                final OXTemplate t;
                                try {
                                    t = ts.loadTemplate(filename);
                                } catch (final Exception e) {
                                    LOG.error(String.format("failed to load localized (%s) template file '%s'", lang, filename), e);
                                    return new InvalidTemplateFacade(notNull(ConfigurationExceptionCodes.FILE_NOT_FOUND.create(e, filename)));
                                }
                                if (t != null) {
                                    templateBuilder.put(template, lang, t);
                                } else {
                                    LOG.error(String.format("failed to load localized (%s) template file '%s'", lang, filename));
                                    return new InvalidTemplateFacade(notNull(ConfigurationExceptionCodes.FILE_NOT_FOUND.create(filename)));
                                }
                            }
                        }
                    }
                }
            }
        }
        return new ValidTemplateFacade(build(templateBuilder), build(fallbackBuilder));
    }

    private static ImmutableMap<String, Object> strMap(final Object... keyvalues) {
        if (keyvalues.length < 1) {
            return emptyMap();
        } else if (keyvalues.length % 2 == 1) {
            throw new IllegalArgumentException("the number of parameters is not even");
        } else {
            final ImmutableMap.Builder<String, Object> b = mapBuilder();
            int i = 0;
            while (i < keyvalues.length) {
                final Object keyObj = keyvalues[i];
                if (keyObj == null) {
                    throw new IllegalArgumentException("null value for a key");
                }
                final String key = notNull(keyObj.toString());
                
                i++;
                if (i < keyvalues.length) {
                    final Object value = keyvalues[i];
                    if (value == null) {
                        throw new IllegalArgumentException("null value for a value");
                    }
                    b.put(key, value);
                    i++;
                }
            }
            return build(b);
        }
    }
    
    private static <X> X notNull(final @Nullable X x) throws IllegalArgumentException {
        if (x == null) {
            throw new IllegalArgumentException("unexpected null value");
        }
        return x;
    }

    
}
