/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.activator.impl.RegistrationTools.propertyAnnotation;
import static com.openexchange.util.activator.impl.RegistrationTools.serviceAnnotation;
import static com.openexchange.util.activator.impl.RegistrationTools.serviceConstructors;
import static com.openexchange.util.activator.impl.RegistrationTools.setOfServicesAnnotation;
import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.logger;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Dictionary;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.concurrent.Immutable;
import org.osgi.framework.BundleContext;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import org.slf4j.Logger;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.util.activator.Property;
import com.openexchange.util.activator.RegistrationException;
import com.openexchange.util.activator.Service;
import com.openexchange.util.activator.ServiceClass;
import com.openexchange.util.activator.ServiceDependencyResolver;
import com.openexchange.util.activator.ServiceRef;
import com.openexchange.util.activator.ServiceRegistrationHandler;
import com.openexchange.util.activator.SetOfServices;

/**
 * {@link ClassServiceWithDependenciesTemplate}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.2
 * @param <S>
 * @param <T>
 */
@Immutable
public abstract class ClassServiceWithDependenciesTemplate<S, T extends S> extends ServiceWithDependenciesTemplate<S, T> {
    
    @SuppressWarnings("null")
    private static final @NonNull ImmutableSet<Class<?>> CLASSES_THAT_REQUIRE_SET_OF_SERVICES_ANNOTATION = ImmutableSet.<Class<?>>of(
        ServiceSet.class,
        ServiceListing.class
    );
    
    protected static final class DependencyScan<S, T> {
        public final Constructor<T> constructor;
        public final ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies;
        public final ImmutableSet<OptionalServiceDependency<?>> optionalDependencies;
        public final ImmutableSet<Class<?>> serviceSets;
        public final ImmutableSet<Class<?>> serviceListings;
        public DependencyScan(Constructor<T> constructor,
            ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies,
            ImmutableSet<OptionalServiceDependency<?>> optionalDependencies,
            ImmutableSet<Class<?>> serviceSets,
            ImmutableSet<Class<?>> serviceListings) {
            this.constructor = constructor;
            this.mandatoryDependencies = mandatoryDependencies;
            this.optionalDependencies = optionalDependencies;
            this.serviceSets = serviceSets;
            this.serviceListings = serviceListings;
        }
    }
    
    protected static final <X, Y extends X> DependencyScan<X, Y> scan(final Class<Y> serviceInstanceClass) {
        final Set<MandatoryServiceDependency<?>> mandatoryServices = new HashSet<>();
        final Set<OptionalServiceDependency<?>> optionalServices = new HashSet<>();
        final Set<Class<?>> serviceSets = new HashSet<Class<?>>();
        final Set<Class<?>> serviceListings = new HashSet<Class<?>>();
        final Constructor<Y> ctor;
        {
            @SuppressWarnings("unchecked") final Constructor<Y>[] ctors = (Constructor<Y>[]) serviceInstanceClass.getDeclaredConstructors();
            if (ctors.length < 1) {
                throw new RegistrationException(f("%s has no constructors", serviceInstanceClass.getName()));
            }
            if (ctors.length > 1) {
                final List<Constructor<Y>> serviceCtors = serviceConstructors(ctors);
                if (serviceCtors.isEmpty()) {
                    throw new RegistrationException(f("%s has more than one constructor and none has the @%s annotation",
                        serviceInstanceClass.getName(), Service.class.getSimpleName()));
                } else if (serviceCtors.size() > 1) {
                    throw new RegistrationException(f("%s has more than one constructor with the @%s annotation",
                        serviceInstanceClass.getName(), Service.class.getSimpleName()));
                }
                ctor = serviceCtors.get(0);
            } else {
                ctor = ctors[0];
            }
            
            final Class<?>[] paramTypes = ctor.getParameterTypes();
            final Annotation[][] paramAnnotations = ctor.getParameterAnnotations();
            final int numParams = paramTypes.length;
            for (int i = 0; i < numParams; i++) {
                final Class<?> paramType = paramTypes[i];
                final Annotation[] paramAnnotation = paramAnnotations[i];
                
                final @Nullable SetOfServices setOfServicesAnnotation = setOfServicesAnnotation(paramAnnotation);
                if (setOfServicesAnnotation != null) {
                    if (ServiceSet.class.isAssignableFrom(paramType)) {
                        final Class<?> dependencyClass = setOfServicesAnnotation.value();
                        serviceSets.add(dependencyClass);
                    } else if (ServiceListing.class.isAssignableFrom(paramType)) {
                        final Class<?> dependencyClass = setOfServicesAnnotation.value();
                        serviceListings.add(dependencyClass);
                    } else {
                        throw new RegistrationException(f("parameter %d in constructor '%s' of service class %s is annotated with @%s but the parameter type %s is not a %s nor a %s",
                            i + 1, ctor.toGenericString(), serviceInstanceClass.getName(),
                            SetOfServices.class.getSimpleName(),
                            paramType.getName(),
                            ServiceSet.class.getSimpleName(), ServiceListing.class.getSimpleName()));
                    }
                } else if (ServiceRef.class == paramType) {
                    final @Nullable ServiceClass serviceClassAnnotation = RegistrationTools.serviceClassAnnotation(paramAnnotation);
                    if (serviceClassAnnotation == null) {
                        throw new RegistrationException(f("parameter %d in constructor '%s' of service class %s has a parameter %s for optional service dependencies but does not have the mandatory annotation @%s",
                            i + 1, ctor.toGenericString(), serviceInstanceClass.getName(),
                            paramType.getName(),
                            ServiceClass.class.getName()
                        ));
                    }
                    final Class<?> dependencyClass = serviceClassAnnotation.value();
                    final AtomicServiceRef<?> serviceRef = AtomicServiceRef.forClass(dependencyClass);
                    final @Nullable String id;
                    {
                        final @Nullable Service serviceAnnotation = serviceAnnotation(paramAnnotation);
                        id = serviceAnnotation != null ? serviceAnnotation.id() : null;
                    }
                    if (id == null || id.isEmpty()) {
                        optionalServices.add(
                            new OptionalServiceDependency<>(serviceRef, ServiceDependencyQualifiers.NO_ID_QUALIFIER)
                        );
                    } else if ("*".equals(id)) {
                        optionalServices.add(
                            new OptionalServiceDependency<>(serviceRef, ServiceDependencyQualifiers.ANY_ID_QUALIFIER)
                        );
                    } else {
                        optionalServices.add(
                            new OptionalServiceDependency<>(serviceRef, ServiceDependencyQualifiers.mustMatchId(id))
                        );
                    }
                } else if (Dictionary.class.equals(paramType)) {
                } else {
                    for (final Class<?> c : CLASSES_THAT_REQUIRE_SET_OF_SERVICES_ANNOTATION) {
                        if (c.isAssignableFrom(paramType)) {
                            LOG.warn("service class constructor {}.{} has a parameter #{} of type {} but no @{} annotation, this is probably a mistake",
                                serviceInstanceClass.getName(),
                                ctor.toGenericString(),
                                i,
                                c.getName(),
                                SetOfServices.class.getSimpleName());
                        }
                    }
                    
                    final @Nullable Property propertyAnnotation = propertyAnnotation(paramAnnotation);
                    if (propertyAnnotation != null) {
                        /*
                        final String name = propertyAnnotation.value();
                        properties.add(name);
                        */
                    } else {
                        final @Nullable String id;
                        {
                            final @Nullable Service serviceAnnotation = serviceAnnotation(paramAnnotation);
                            id = serviceAnnotation != null ? serviceAnnotation.id() : null;
                        }
                        final Class<?> dependencyClass = paramType;
                        if (dependencyClass != null) {
                            if (id == null || id.isEmpty()) {
                                mandatoryServices.add(
                                    new MandatoryServiceDependency<>(dependencyClass, ServiceDependencyQualifiers.NO_ID_QUALIFIER)
                                );
                            } else if ("*".equals(id)) {
                                mandatoryServices.add(
                                    new MandatoryServiceDependency<>(dependencyClass, ServiceDependencyQualifiers.ANY_ID_QUALIFIER)
                                );
                            } else {
                                mandatoryServices.add(
                                    new MandatoryServiceDependency<>(dependencyClass, ServiceDependencyQualifiers.mustMatchId(id))
                                );
                            }
                        }
                    }
                }
            }
        }
        
        @SuppressWarnings("null")
        final @NonNull ImmutableSet<MandatoryServiceDependency<?>> immutableMandatoryServices = ImmutableSet.copyOf(mandatoryServices);
        @SuppressWarnings("null")
        final @NonNull ImmutableSet<OptionalServiceDependency<?>> immutableOptionalServices = ImmutableSet.copyOf(optionalServices);
        @SuppressWarnings("null")
        final @NonNull ImmutableSet<Class<?>> immutableServiceSets = ImmutableSet.copyOf(serviceSets);
        @SuppressWarnings("null")
        final @NonNull ImmutableSet<Class<?>> immutableServiceListings = ImmutableSet.copyOf(serviceListings);
        
        return new DependencyScan<>(
            ctor,
            immutableMandatoryServices,
            immutableOptionalServices,
            immutableServiceSets,
            immutableServiceListings
        );
    }
    
    private static final Logger LOG = logger(ClassServiceWithDependenciesTemplate.class);
    
    protected final Constructor<T> constructor;
    protected ClassServiceWithDependenciesTemplate(Constructor<T> constructor,
        final Class<T> serviceInstanceClass,
        @Nullable Class<S> serviceClass,
        ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies,
        ImmutableSet<OptionalServiceDependency<?>> optionalDependencies,
        ImmutableSet<Class<?>> serviceSets,
        ImmutableSet<Class<?>> serviceListings) {
        super(
            serviceInstanceClass,
            serviceClass,
            mandatoryDependencies,
            optionalDependencies,
            serviceSets,
            serviceListings
        );
        this.constructor = constructor;
    }
    
    protected ServiceTrackerCustomizer<S, T> createRegisterer(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final ImmutableSet<Class<?>> serviceClasses,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        
        @SuppressWarnings("null")
        final @NonNull String name = serviceInstanceClass.getName();
        
        return new ServiceWithDependenciesServiceRegisterer<>(bundleContext,
            registry,
            this,
            serviceClasses,
            name,
            properties,
            serviceRegistrationHandler
        );
    }
    
    @Override
    public final T createInstance(final ServiceDependencyResolver resolver) {
        final Optional<Object[]> values = resolver.resolveParameters(constructor);
        if (values.isPresent()) {
            try {
                LOG.debug("invoking {} constructor {}", serviceInstanceClass.getName(), constructor.getName());
                final T result = constructor.newInstance(values.get());
                if (result == null) {
                    throw new IllegalStateException("constructor invocation returned null for " + constructor.toGenericString());
                }
                return result;
            } catch (final Exception e) {
                @NonNull Throwable t = e;
                if (t instanceof InvocationTargetException) {
                    final Throwable c = t.getCause();
                    if (c != null) {
                        t = c;
                    }
                }
                // log as debug here, will be logged as error in the caller
                LOG.debug(f("failed to invoke service %s constructor \"%s\" with parameters [%s]: %s: %s",
                    className(serviceInstanceClass),
                    constructor.toGenericString(),
                    Stream.of(values.get()).map(RegistrationTools.TO_CLASS_NAME).collect(Collectors.joining(", ")),
                    className(t), t.getMessage()), t);
                if (t instanceof RegistrationException) {
                    throw (RegistrationException) t;
                } else if (t instanceof RuntimeException) {
                    throw (RuntimeException) t;
                } else {
                    final String msg = t.getMessage();
                    if (msg != null) {
                        throw new RegistrationException(msg, t);
                    } else {
                        throw new RegistrationException(className(t), t);
                    }
                }
            }
        } else {
            final String msg = f("%s: failed to find matching services for constructor '%s'",
                className(serviceInstanceClass), constructor.toGenericString());
            LOG.error(msg);
            throw new RegistrationException(msg);
        }              
    }

}