/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.absent;
import static com.openexchange.util.custom.base.NullUtil.build;
import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.emptyList;
import static com.openexchange.util.custom.base.NullUtil.emptySet;
import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.immutableCopyOf;
import static com.openexchange.util.custom.base.NullUtil.immutableSet;
import static com.openexchange.util.custom.base.NullUtil.listBuilder;
import static com.openexchange.util.custom.base.NullUtil.logger;
import static com.openexchange.util.custom.base.NullUtil.mapBuilder;
import static com.openexchange.util.custom.base.NullUtil.notNull;
import static com.openexchange.util.custom.base.NullUtil.optional;
import static com.openexchange.util.custom.base.NullUtil.setBuilder;
import static java.util.stream.Collectors.joining;
import java.io.Closeable;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.rmi.Remote;
import java.util.Collection;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.management.ObjectName;
import javax.servlet.http.HttpServlet;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Filter;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceRegistration;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import org.slf4j.Logger;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXActionServiceFactory;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.ForcedReloadable;
import com.openexchange.config.Reloadable;
import com.openexchange.dispatcher.DispatcherPrefixService;
import com.openexchange.groupware.update.DefaultUpdateTaskProviderService;
import com.openexchange.groupware.update.UpdateTaskV2;
import com.openexchange.management.ManagementService;
import com.openexchange.osgi.ExceptionUtils;
import com.openexchange.osgi.RankingAwareNearRegistryServiceTracker;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.osgi.SimpleRegistryListener;
import com.openexchange.util.activator.DependencyProvider;
import com.openexchange.util.activator.Property;
import com.openexchange.util.activator.Provides;
import com.openexchange.util.activator.RegistrationException;
import com.openexchange.util.activator.Service;
import com.openexchange.util.activator.ServiceClass;
import com.openexchange.util.activator.ServiceDependencies;
import com.openexchange.util.activator.ServiceDependencyResolver;
import com.openexchange.util.activator.ServiceFactory;
import com.openexchange.util.activator.ServiceRef;
import com.openexchange.util.activator.ServiceRegistrationHandler;
import com.openexchange.util.activator.ServiceWithProperties;
import com.openexchange.util.activator.SetOfServices;
import com.openexchange.util.custom.base.NullUtil;
import com.openexchange.util.custom.collect.ImmutableDictionary;

/**
 * {@link RegistrationTools}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.2
 */
public enum RegistrationTools {
    ;
    
    private static Logger LOG = logger(RegistrationTools.class);
    
    public static final <S, T extends S> T createInstance(final String serviceInstanceName,
        final Constructor<T> constructor,
        final ImmutableSet<Class<?>> serviceTypes,
        final ServiceDependencyResolver resolver,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        final Optional<Object[]> values = resolver.resolveParameters(constructor);
        if (values.isPresent()) {
            try {
                LOG.debug("invoking {} constructor {}", serviceInstanceName, constructor.getName());
                final T serviceInstance = constructor.newInstance(values.get());
                if (serviceInstance == null) {
                    throw new IllegalStateException("constructor invocation returned null for " + constructor.toGenericString());
                }
                if (serviceRegistrationHandler != null) {
                    serviceRegistrationHandler.onServiceRegistered(serviceInstance);
                }
                return serviceInstance;
            } catch (final Exception e) {
                Throwable t = e;
                if (t instanceof InvocationTargetException) {
                    final Throwable c = t.getCause();
                    if (c != null) {
                        t = c;
                    }
                }
                ExceptionUtils.handleThrowable(t);
                
                LOG.error(f("failed to invoke service %s (%s) constructor \"%s\" with parameters [%s]: %s: %s",
                    serviceInstanceName,
                    serviceTypes.stream().map(TO_SIMPLE_CLASS_NAME).collect(Collectors.joining(", ")),
                    constructor.toGenericString(),
                    Stream.of(values.get()).map(TO_CLASS_NAME).collect(Collectors.joining(", ")),
                    className(t), t.getMessage()), t);
                if (t instanceof RegistrationException) {
                    throw (RegistrationException) t;
                } else if (t instanceof RuntimeException) {
                    throw (RuntimeException) t;
                } else {
                    final String msg = t.getMessage();
                    if (msg != null) {
                        throw new RegistrationException(msg, t);
                    } else {
                        throw new RegistrationException(className(t), t);
                    }
                }
            }
        } else {
            final String msg = f("%s: failed to find matching services for constructor '%s'",
                serviceInstanceName, constructor.toGenericString());
            LOG.error(msg);
            throw new RegistrationException(msg);
        }        
    }
    
    public static final <T> void closeInstance(final T serviceInstance, final String serviceInstanceName, final Set<Class<?>> serviceTypes) {
        if (serviceInstance instanceof Closeable) {
            try {
                ((Closeable) serviceInstance).close();
            } catch (Throwable t) {
                ExceptionUtils.handleThrowable(t);
                LOG.error(String.format("failed to close %s (%s)",
                    serviceInstanceName,
                    serviceTypes.stream().map(TO_SIMPLE_CLASS_NAME).collect(Collectors.joining(", "))),
                    t);
            }
        } else {
            try {
                Method method = serviceInstance.getClass().getMethod("shutDown", new Class<?>[0]);
                if (method != null && method.isAccessible()) {
                    method.invoke(serviceInstance, new Object[0]);
                }
            } catch (SecurityException e) {
                // Service does not have a shutDown() method.
            } catch (NoSuchMethodException e) {
                // Service does not have a shutDown() method.
            } catch (Throwable t) {
                ExceptionUtils.handleThrowable(t);
                LOG.error(String.format("failed to shut down %s (%s)",
                    serviceInstanceName,
                    serviceTypes.stream().map(TO_SIMPLE_CLASS_NAME).collect(Collectors.joining(", "))),
                    t);
            }
        }        
    }
    
    public static final @Nullable Service serviceAnnotation(final @Nullable Annotation[] annotations) {
        if (annotations == null) {
            return null;
        }
        for (final Annotation a : annotations) {
            if (a instanceof Service) {
                return (Service) a;
            }
        }
        return null;
    }
    
    public static final @Nullable SetOfServices setOfServicesAnnotation(final @Nullable Annotation[] annotations) {
        if (annotations == null) {
            return null;
        }
        for (final Annotation a : annotations) {
            if (a instanceof SetOfServices) {
                return (SetOfServices) a;
            }
        }
        return null;
    }

    public static final @Nullable Property propertyAnnotation(final @Nullable Annotation[] annotations) {
        if (annotations == null) {
            return null;
        }
        for (final Annotation a : annotations) {
            if (a instanceof Property) {
                return (Property) a;
            }
        }
        return null;
    }

    public static final @Nullable ServiceClass serviceClassAnnotation(final @Nullable Annotation[] annotations) {
        if (annotations == null) {
            return null;
        }
        for (final Annotation a : annotations) {
            if (a instanceof ServiceClass) {
                return (ServiceClass) a;
            }
        }
        return null;
    }
    
    public static final <T> ImmutableList<Constructor<T>> serviceConstructors(final @Nullable Constructor<T>[] constructors) {
        if (constructors == null) {
            return emptyList();
        }
        final ImmutableList.Builder<Constructor<T>> b = listBuilder();
        for (final Constructor<T> ctor : constructors) {
            if (ctor.getAnnotation(Service.class) != null) {
                b.add(ctor);
            }
        }
        return build(b);
    }

    public static final Optional<Object[]> pickDependencies(final Method method,
        Map<MandatoryServiceDependency<?>, Object> mandatoryDependencies,
        List<OptionalServiceDependency<?>> optionalDependencies,
        Map<Class<?>, ServiceSet<?>> serviceSetMap,
        Map<Class<?>, ServiceListing<?>> serviceListingMap,
        Dictionary<String, ?> properties) {
        return pickDependencies(notNull(method.getParameterTypes()), notNull(method.getParameterAnnotations()),
            mandatoryDependencies, optionalDependencies,
            serviceSetMap, serviceListingMap,
            properties);
    }
    
    public static final Optional<Object[]> pickDependencies(final Constructor<?> ctor,
        Map<MandatoryServiceDependency<?>, Object> mandatoryDependencies,
        List<OptionalServiceDependency<?>> optionalDependencies,
        Map<Class<?>, ServiceSet<?>> serviceSetMap,
        Map<Class<?>, ServiceListing<?>> serviceListingMap,
        Dictionary<String, ?> properties) {
        return pickDependencies(notNull(ctor.getParameterTypes()), notNull(ctor.getParameterAnnotations()),
            mandatoryDependencies, optionalDependencies,
            serviceSetMap, serviceListingMap,
            properties);
    }

    private static final Optional<Object[]> pickDependencies(final Class<?>[] types,
        final Annotation[][] annotationArrays,
        Map<MandatoryServiceDependency<?>, Object> mandatoryDependencies,
        List<OptionalServiceDependency<?>> optionalDependencies,
        Map<Class<?>, ServiceSet<?>> serviceSetMap,
        Map<Class<?>, ServiceListing<?>> serviceListingMap,
        Dictionary<String, ?> properties) {
        final Object[] values = new Object[types.length];
        for (int i = 0; i < values.length; i++) {
            final Class<?> type = types[i];
            final Annotation[] annotations = annotationArrays[i];
            
            if (type.equals(ServiceSet.class)) {
                final SetOfServices a = setOfServicesAnnotation(annotations);
                if (a != null) {
                    final Class<?> serviceSetType = a.value();
                    final ServiceSet<?> serviceSet = serviceSetMap.get(serviceSetType);
                    if (serviceSet == null) {
                        return absent();
                    } else {
                        values[i] = serviceSet;
                    }
                }
            } else if (type.equals(ServiceListing.class)) {
                final SetOfServices a = setOfServicesAnnotation(annotations);
                if (a != null) {
                    final Class<?> serviceListingType = a.value();
                    final ServiceListing<?> serviceListing = serviceListingMap.get(serviceListingType);
                    if (serviceListing == null) {
                        return absent();
                    } else {
                        values[i] = serviceListing;
                    }
                }
            } else if (type.equals(ServiceRef.class)) {
                final ServiceClass a = RegistrationTools.serviceClassAnnotation(annotations);
                if (a != null) {
                    final Class<?> dependencyType = a.value();
                    boolean matched = false;
                    final Iterator<OptionalServiceDependency<?>> iter = optionalDependencies.iterator();
                    while ((! matched) && iter.hasNext()) {
                        final OptionalServiceDependency<?> e = iter.next();
                        if (e.matchesDependency(dependencyType, annotations)) {
                            values[i] = e.serviceRef();
                            matched = true;
                        }
                    }
                    if (! matched) {
                        return absent();
                    }
                }
            } else if (type.equals(Dictionary.class)) {
                values[i] = properties;
            } else {
                final Property property = propertyAnnotation(annotations);
                if (property != null) {
                    final Object value = properties.get(property.value());
                    if (value != null && type.isInstance(value)) {
                        values[i] = Objects.toString(value);
                    } else {
                        return absent();
                    }
                } else {
                    boolean matched = false;
                    final Iterator<Map.Entry<MandatoryServiceDependency<?>, Object>> iter = mandatoryDependencies.entrySet().iterator();
                    while ((! matched) && iter.hasNext()) {
                        final Map.Entry<MandatoryServiceDependency<?>, Object> e = iter.next();
                        if (e.getKey().matchesDependency(type, annotations)) {
                            values[i] = e.getValue();
                            matched = true;
                        }
                    }
                    if (! matched) {
                        return absent();
                    }
                }
            }
        }
        return optional(values);
    }
    
    public static final @Nullable Filter generateServiceFilter(final BundleContext context,
        final Collection<MandatoryServiceDependency<?>> mandatoryServiceDependencies,
        final Collection<OptionalServiceDependency<?>> optionalServiceDependencies) throws InvalidSyntaxException {
        
        @SuppressWarnings("null")
        final @NonNull List<ServiceDependency<?>> dependencies = Lists.newArrayList(Iterables.concat(mandatoryServiceDependencies, optionalServiceDependencies));
        
        if (dependencies.isEmpty()) {
            return null;
        }
        
        final String filter;
        if (dependencies.size() == 1) {
            filter = dependencies.get(0).asFilter();
        } else {
            filter = NullUtil.f("(|%s)", dependencies.stream().map(ServiceDependency::asFilter).collect(joining()));
        }
        return context.createFilter(filter);
    }
    
    public static final void close(final @Nullable Object potentiallyCloseable, final String name) {
        if (potentiallyCloseable == null) {
            return;
        }
        if (potentiallyCloseable instanceof Closeable) {
            try {
                LOG.debug("closing service {}", name);
                ((Closeable) potentiallyCloseable).close();
            } catch (final Throwable t) {
                ExceptionUtils.handleThrowable(t);
                LOG.warn(f("(suppressed) failed to close %s: %s: %s", name,
                    className(t), t.getMessage()), t);
                // exception is suppressed intentionally
            }
        }
    }    

    /**
     * Thread-safe way to close all {@link Closeable} items in a thread-safe {@link Queue}.
     * <p>
     * It logs and suppresses any {@link Exception} that might occur when calling {@link Closeable#close()}
     * on each item.
     * 
     * @param queue a thread-safe {@link Queue} with non-{@code null} {@link Closeable}s
     * @param log a {@link Logger} to use when calling {@link Closeable#close()} yields an {@link Exception}
     */
    public static final <E extends Closeable> void closeAll(final Queue<E> queue, final Logger log) {
        E item = queue.poll();
        while (item != null) {
            try {
                item.close();
            } catch (final Exception e) {
                log.error(f("error while closing %s (suppressed): %s: %s",
                    className(item), className(e), e.getMessage()), e);
            }
            item = queue.poll();
        }
    }
    
    /**
     * Thread-safe way to close all items in a thread-safe {@link Queue}.
     * <p>
     * It logs and suppresses any {@link Exception} that might occur when calling a
     * {@code close()} method if such a method exists, on each item.
     * 
     * @param queue a thread-safe {@link Queue} with non-{@code null} {@link Closeable}s
     * @param log a {@link Logger} to use when calling {@code close()} yields an {@link Exception}
     */
    public static final void closeAllReflectively(final Queue<?> queue, final Logger log) {
        Object item = queue.poll();
        while (item != null) {
            try {
                Method closeMethod = null;
                try {
                    closeMethod = item.getClass().getMethod("close", EMPTY_CLASS_ARRAY);
                } catch (NoSuchMethodException e) {
                }
                if (closeMethod != null && closeMethod.isAccessible()) {
                    closeMethod.invoke(item);
                }
            } catch (final Exception e) {
                log.error(f("error while closing %s (suppressed): %s: %s",
                    className(item), className(e), e.getMessage()), e);
            }
            item = queue.poll();
        }
    }
    private static final Class<?>[] EMPTY_CLASS_ARRAY = new Class<?>[0];
    
    @SuppressWarnings("null")
    public static final @NonNull ImmutableSet<Class<?>> WELL_KNOWN_SERVICE_INTERFACES = ImmutableSet.<Class<?>>of(
        Reloadable.class
    );
    @SuppressWarnings("null")
    public static final @NonNull ImmutableSet<Class<?>> INTERFACES_TO_IGNORE_FOR_SERVICE_DISCOVERY = ImmutableSet.of(
        Reloadable.class, ForcedReloadable.class, Closeable.class
    );
    @SuppressWarnings("null")
    public static final @NonNull ImmutableSet<Class<?>> CLASSES_TO_IGNORE = ImmutableSet.of(
        Enum.class, Object.class
    );

    public static <T> ImmutableSet<Class<?>> whatIsProvidedBy(final Class<T> serviceInstanceClass,
        final @Nullable Class<?> optServiceClass) throws RegistrationException {
        if (optServiceClass != null) {
            return NullUtil.<Class<?>>immutableSet(optServiceClass);
        }
        
        final @Nullable Provides provides = serviceInstanceClass.getAnnotation(Provides.class);
        if (provides == null) {
            final ImmutableSet<Class<?>> interfaces;
            {
                final ImmutableSet.Builder<Class<?>> b = ImmutableSet.builder();
                final Class<?>[] ifaces = serviceInstanceClass.getInterfaces();
                if (ifaces != null) {
                    for (final Class<?> c : ifaces) {
                        if (c != null) {
                            if (! CLASSES_TO_IGNORE.contains(c)) {
                                b.add(c);
                            }
                        }
                    }
                }
                final Class<?> sup = serviceInstanceClass.getSuperclass();
                if (sup != null) {
                    if (! CLASSES_TO_IGNORE.contains(sup)) {
                        b.add(sup);
                    }
                }
                interfaces = b.build();
            }
            if (interfaces.isEmpty()) {
                // if it has a @Service annotation, than it is simply its own service class, so to speak
                if (declaredAnnotation(Service.class, serviceInstanceClass) != null) {
                    return NullUtil.<Class<?>>immutableSet(serviceInstanceClass);
                } else {
                    // implements itself
                    // TODO it is not recommended to provide a class as an OSGi service facade, should we throw an exception
                    /*
                    throw new RegistrationException(f("%s cannot be registered as it has no @%s annotation and does not implement any interface",
                        serviceInstanceClass.getName(), Provides.class.getSimpleName()));
                    */
                    // ... or log a warning ?
                    LOG.warn("{} registers itself as a class without interface, consider adding a @%s annotation to confirm that that is what you want to do",
                        serviceInstanceClass.getName(), Provides.class.getSimpleName());
                    // ... and then proceed ?
                    return NullUtil.<Class<?>>immutableSet(serviceInstanceClass);
                }
            } else if (interfaces.size() > 1) {
                final Set<Class<?>> serviceInterfaces = interfaces.stream().filter(HAS_SERVICE_ANNOTATION).collect(ImmutableSet.toImmutableSet());
                if (serviceInterfaces.size() == 1) {
                    return immutableCopyOf(serviceInterfaces);
                } else {
                    // filter out known interfaces to ignore
                    final ImmutableSet<Class<?>> filteredInterfaces = interfaces.stream()
                        .filter(iface -> !INTERFACES_TO_IGNORE_FOR_SERVICE_DISCOVERY.contains(iface))
                        .collect(ImmutableSet.toImmutableSet());
                    if (filteredInterfaces.isEmpty()) {
                        throw new RegistrationException(f("%s cannot be registered as it has no @%s annotation and implements several interfaces with @%s annotation: %s",
                            serviceInstanceClass.getName(), Provides.class.getSimpleName(), Service.class.getSimpleName(), Joiner.on(", ").join(serviceInterfaces)));
                    } else if (filteredInterfaces.size() > 1) {
                        throw new RegistrationException(f("%s cannot be registered as it has no @%s annotation and implements several potential service interfaces (%s)",
                            serviceInstanceClass.getName(), Provides.class.getSimpleName(), Joiner.on(", ").join(filteredInterfaces)));
                    } else {
                        @SuppressWarnings("null")
                        final @NonNull Class<?> filteredInterface = Iterables.getOnlyElement(filteredInterfaces);
                        return NullUtil.<Class<?>>immutableSet(filteredInterface);
                    }
                }
            } else {
                @SuppressWarnings("null")
                final @NonNull Class<?> iface = Iterables.getOnlyElement(interfaces);
                return NullUtil.<Class<?>>immutableSet(iface);
            }
        }
        final @Nullable Class<?>[] serviceClasses = provides.value();
        if (serviceClasses == null || serviceClasses.length < 1) {
            throw new RegistrationException(f("%s cannot be registered because @%s is empty",
                serviceInstanceClass.getName(), Provides.class.getSimpleName()));
        }
        for (final Class<?> serviceClass : serviceClasses) {
            if (! serviceClass.isAssignableFrom(serviceInstanceClass)) {
                throw new RegistrationException(f("%s cannot be registered because it is not an instance of @%s(%s)",
                    serviceInstanceClass.getName(), Provides.class.getSimpleName(), serviceClass.getName()));
            }
        }
        
        final ImmutableSet.Builder<Class<?>> b = ImmutableSet.builder();
        b.add(serviceClasses);
        for (final Class<?> wellKnownServiceInterface : WELL_KNOWN_SERVICE_INTERFACES) {
            if (wellKnownServiceInterface.isAssignableFrom(serviceInstanceClass)) {
                b.add(wellKnownServiceInterface);
            }
        }
        
        return build(b);
    }
    
    public static final Predicate<Class<?>> HAS_SERVICE_ANNOTATION = new Predicate<Class<?>>() {
        @Override
        public boolean test(@Nullable Class<?> input) {
            return input != null && input.getAnnotation(Service.class) != null;
        }
    };
    
    public static <S, T extends S> void registerServiceFactory(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final ServiceFactory<T> serviceFactory,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        
        final ServiceWithDependencies<T> service = ServiceFactoryWithDependencies.find(serviceFactory);
        service.registerService(bundleContext, registry, properties, serviceRegistrationHandler);
    }
    
    public static <S, T extends S> void registerServiceClass(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final @Nullable Class<S> optServiceClass,
        final Class<T> serviceInstanceClass,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        final ServiceWithDependencies<T> service = ConstructorServiceWithDependencies.find(serviceInstanceClass, optServiceClass);
        service.registerService(bundleContext, registry, properties, serviceRegistrationHandler);
    }
    
    public static <S, T extends S> void registerServiceInstance(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final @Nullable Class<S> optServiceClass,
        final T serviceInstance,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {

        final ServiceWithDependencies<T> service;
        if (serviceInstance instanceof DependencyProvider) {
            service = new DependencyProviderServiceWithDependencies<S, T>(serviceInstance, optServiceClass);
        } else {
            service = new ServiceInstanceServiceWithDependencies<S, T>(
                serviceInstance, optServiceClass, null, null
            );
        }
        service.registerService(bundleContext, registry, properties, serviceRegistrationHandler);
    }
    
    @SafeVarargs
    private static Dictionary<String, ?> merge(final Dictionary<String, ?>... dictionaries) {
        final Map<String, Object> map = new HashMap<>();
        for (final Dictionary<String, ?> d : dictionaries) {
            final Enumeration<String> e = d.keys();
            while (e.hasMoreElements()) {
                final String key = e.nextElement();
                if (key != null) {
                    map.put(key, d.get(key));
                }
            }
        }
        return ImmutableDictionary.copyOf(map);
    }
    
    public static <S, T extends S> void registerServiceInstanceOrClass(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final @Nullable Class<S> serviceClass,
        final T serviceInstanceOrClass,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        if (serviceInstanceOrClass instanceof ServiceWithProperties) {
            @SuppressWarnings("unchecked") final ServiceWithProperties<T> swp = (ServiceWithProperties<T>) serviceInstanceOrClass;
            registerServiceClass(bundleContext, registry, serviceClass,
                swp.serviceInstanceClass(), merge(swp.properties(), properties),
                serviceRegistrationHandler);
        } else if (serviceInstanceOrClass instanceof ServiceFactory) {
            @SuppressWarnings("unchecked") final ServiceFactory<T> serviceFactory = (ServiceFactory<T>) serviceInstanceOrClass;
            registerServiceFactory(bundleContext, registry, serviceFactory, properties, serviceRegistrationHandler);
        } else if (Class.class.equals(serviceInstanceOrClass.getClass())) {
            // it is a service instance class
            @SuppressWarnings("unchecked") final Class<T> serviceInstanceClass = (Class<T>) serviceInstanceOrClass;
            registerServiceClass(bundleContext, registry, serviceClass, serviceInstanceClass, properties, serviceRegistrationHandler);
        } else {
            // it is a service instance
            final T serviceInstance = serviceInstanceOrClass;
            registerServiceInstance(bundleContext, registry, serviceClass, serviceInstance, properties, serviceRegistrationHandler);
        }
    }
    
    public static <T extends HttpServlet> void registerServletInstance(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final T serviceInstance,
        final Dictionary<String, ?> properties,
        final String path) {
        final ImmutableSet<Class<?>> serviceClasses = immutableSet(HttpServlet.class);

        final ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies;
        final ImmutableSet<OptionalServiceDependency<?>> optionalDependencies;
        final ImmutableSet<Class<?>> serviceSetClasses;
        final ImmutableSet<Class<?>> serviceListingClasses;
        {
            final ServiceDependencies serviceDependencies;
            if (serviceInstance instanceof DependencyProvider) {
                serviceDependencies = ((DependencyProvider) serviceInstance).dependsOn();
            } else {
                serviceDependencies = ImmutableServiceDependencies.empty();
            }
            mandatoryDependencies = ServletServiceRegistererTemplate.withMandatoryServletDependencies(serviceDependencies.mandatoryServiceDependencies(), path);
            optionalDependencies = ServletServiceRegistererTemplate.withOptionalServletDependencies(serviceDependencies.optionalServiceDependencies(), path);
            serviceSetClasses = serviceDependencies.serviceSetClasses();
            serviceListingClasses = serviceDependencies.serviceListingClasses();
        }

        final ImmutableMap<Class<?>, ServiceSet<?>> serviceSets = serviceSets(bundleContext,
            registry,
            serviceSetClasses
        );

        final ImmutableMap<Class<?>, ServiceListing<?>> serviceListings = serviceListings(bundleContext,
            registry,
            serviceListingClasses
        );
        
        final ImmutableSet<String> propertiesOfInterest = collectPropertiesOfInterest(mandatoryDependencies, optionalDependencies);
        
        final ServletServiceRegistererTemplate<T> registerer =
            new ServletServiceRegistererTemplate<T>(path,
                bundleContext,
                serviceClasses,
                className(serviceInstance),
                properties,
                mandatoryDependencies,
                optionalDependencies,
                serviceSets,
                serviceListings,
                propertiesOfInterest) {
            @Override
            protected T createInstance(ServiceDependencyResolver resolver)
                throws Exception {
                if (serviceInstance instanceof DependencyProvider) {
                    ((DependencyProvider) serviceInstance).onDependenciesAvailable(resolver);
                }
                return serviceInstance;
            }
        };
        RegistrationTools.<HttpServlet, T>trackAndOpen(
            bundleContext,
            mandatoryDependencies,
            optionalDependencies,
            registerer,
            className(serviceInstance)
        );
    }
    
    public static <T extends HttpServlet> void registerServletClass(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final Class<T> serviceInstanceClass,
        final Dictionary<String, ?> properties,
        final String path,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        final ServiceWithDependencies<T> service = ServletClassServiceWithDependencies.find(serviceInstanceClass, path);
        service.registerService(bundleContext, registry, properties, serviceRegistrationHandler);
    }
    
    public static <S, T extends S> void registerMBean(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final ObjectName objectName,
        final Class<S> mbeanInterface,
        final T mbean) {
        
        final MBeanRegistrar<S, T> r = new MBeanRegistrar<S, T>(bundleContext, registry, objectName, mbeanInterface, mbean);
        trackAndOpen(
            bundleContext,
            immutableSet(new MandatoryServiceDependency<>(ManagementService.class)),
            emptySet(),
            r,
            notNull(objectName.toString())
        );
    }

    @SafeVarargs
    public static void registerModule(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final String module,
        final Class<? extends AJAXActionService>... actions) {
        final Hashtable<String, Object> properties = new Hashtable<>(2);
        properties.put("module", module);
        properties.put("multiple", true);
        
        final ImmutableSet<Class<? extends AJAXActionService>> actionSet;
        {
            final ImmutableSet.Builder<Class<? extends AJAXActionService>> b = setBuilder();
            for (final Class<? extends AJAXActionService> action : actions) {
                if (action != null) {
                    b.add(action);
                }
            }
            actionSet = build(b);
        }
        
        registerServiceInstance(bundleContext, registry,
            AJAXActionServiceFactory.class,
            new DiscoveringActionFactory(module, actionSet),
            properties,
            null
        );
    }
    
    public static final <F extends javax.servlet.Filter> void registerLoginFilter(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final Class<F> filterClass,
        final int ranking) {
        final ServiceTrackerCustomizer<DispatcherPrefixService, DispatcherPrefixService> customizer = new DispatcherPrefixServiceTrackerCustomizer(bundleContext) {
            @Override
            protected void onDispatcherPrefixServiceAvailable(final DispatcherPrefixService dispatcherPrefixService) throws Exception {
                final Hashtable<String, Object> properties = new Hashtable<>(2);
                properties.put(org.osgi.framework.Constants.SERVICE_RANKING, Integer.valueOf(ranking));
                properties.put(com.openexchange.servlet.Constants.FILTER_PATHS, dispatcherPrefixService.getPrefix() + "login");
                registerServiceClass(bundleContext, registry, javax.servlet.Filter.class, filterClass, properties, null);
            }
        };
        final ServiceTracker<DispatcherPrefixService, DispatcherPrefixService> tracker = new ServiceTracker<DispatcherPrefixService, DispatcherPrefixService>(bundleContext,
            DispatcherPrefixService.class, customizer);
        tracker.open();
        if (customizer instanceof ExtendedServiceTrackerCustomizer) {
            ((ExtendedServiceTrackerCustomizer<DispatcherPrefixService, DispatcherPrefixService>) customizer).onOpened();
        }
        registry.add(tracker);
    }
    
    public static final <F extends javax.servlet.Filter> void registerLoginFilter(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final F filter,
        final int ranking) {
        final ServiceTrackerCustomizer<DispatcherPrefixService, DispatcherPrefixService> customizer = new DispatcherPrefixServiceTrackerCustomizer(bundleContext) {
            @Override
            protected void onDispatcherPrefixServiceAvailable(final DispatcherPrefixService dispatcherPrefixService) throws Exception {
                final Hashtable<String, Object> properties = new Hashtable<>(2);
                properties.put(org.osgi.framework.Constants.SERVICE_RANKING, Integer.valueOf(ranking));
                properties.put(com.openexchange.servlet.Constants.FILTER_PATHS, dispatcherPrefixService.getPrefix() + "login");
                RegistrationTools.registerServiceInstance(bundleContext, registry, javax.servlet.Filter.class, filter, properties, null);
            }
        };
        final ServiceTracker<DispatcherPrefixService, DispatcherPrefixService> tracker = new ServiceTracker<DispatcherPrefixService, DispatcherPrefixService>(bundleContext,
            DispatcherPrefixService.class, customizer);
        tracker.open();
        if (customizer instanceof ExtendedServiceTrackerCustomizer) {
            ((ExtendedServiceTrackerCustomizer<DispatcherPrefixService, DispatcherPrefixService>) customizer).onOpened();
        }
        registry.add(tracker);
    }
    
    public static final <U extends UpdateTaskV2> void registerUpdateTask(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final Class<U> updateTaskClass,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<DefaultUpdateTaskProviderService> serviceRegistrationHandler) {
        final ServiceWithDependencies<DefaultUpdateTaskProviderService> service = UpdateTaskServiceWithDependencies.find(updateTaskClass);
        service.registerService(bundleContext, registry, properties, serviceRegistrationHandler);
    }
    
    public static <S, T extends S> ServiceRegistration<S> registerIntoContext(final BundleContext bundleContext,
        final Collection<?> serviceClasses,
        final T serviceInstance,
        final @Nullable Dictionary<String, ?> properties) {
        final ServiceRegistration<S> serviceRegistration;
        try {
            final String[] serviceClassesAsArray = serviceClasses.stream()
                .filter(Objects::nonNull)
                .map(CLASS_AWARE_TO_STRING)
                .toArray(String[]::new);
            @SuppressWarnings("unchecked")
            final ServiceRegistration<S> r = (ServiceRegistration<S>) bundleContext.registerService(
                serviceClassesAsArray,
                serviceInstance,
                properties != null ? properties : ImmutableDictionary.of(ImmutableMap.<String, Object>of())
            );
            serviceRegistration = r;
        } catch (final Exception e) {
            final String msg = f("service registration failed for %s: %s: %s",
                className(serviceInstance),
                className(e),
                e.getMessage());
            LOG.error(msg, e);
            throw new RegistrationException(msg, e);
        }
        if (serviceRegistration == null) {
            final String msg = f("service registration returned null for %s", className(serviceInstance));
            LOG.error(msg);
            throw new RegistrationException(msg);
        }
        return serviceRegistration;
    }
    
    public static final <S> ServiceSet<S> serviceSet(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final Class<S> serviceClass) {
        final ServiceSet<S> serviceSet = new ServiceSet<S>();
        track(bundleContext, registry, serviceClass, serviceSet);
        return serviceSet;
    }
    
    public static final ImmutableMap<Class<?>, ServiceSet<?>> serviceSets(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final ImmutableSet<Class<?>> serviceSetClasses) {
        final ImmutableMap.Builder<Class<?>, ServiceSet<?>> b = mapBuilder();
        for (final Class<?> serviceSetClass : serviceSetClasses) {
            if (serviceSetClass != null) {
                final ServiceSet<?> ss = serviceSet(bundleContext, registry, serviceSetClass);
                b.put(serviceSetClass, ss);
            }
        }
        return build(b);
    }

    public static final <S> ServiceListing<S> serviceListing(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final Class<S> serviceClass) {
        final RankingAwareNearRegistryServiceTracker<S> tracker = new RankingAwareNearRegistryServiceTracker<S>(bundleContext, serviceClass);
        tracker.open();        
        registry.add(tracker);
        return tracker;
    }
    
    public static final ImmutableMap<Class<?>, ServiceListing<?>> serviceListings(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final ImmutableSet<Class<?>> serviceListingClasses) {
        final ImmutableMap.Builder<Class<?>, ServiceListing<?>> b = mapBuilder();
        for (final Class<?> serviceListingClass : serviceListingClasses) {
            if (serviceListingClass != null) {
                final ServiceListing<?> sl = serviceListing(bundleContext, registry, serviceListingClass);
                b.put(serviceListingClass, sl);
            }
        }
        return build(b);
    }
    
    public static final <S> void track(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final Class<S> serviceClass,
        final SimpleRegistryListener<S> serviceSet) {
        final ServiceTrackerCustomizer<S, S> customizer = new SimpleRegistryListenerTrackerCustomizer<S>(serviceSet, bundleContext);
        final ServiceTracker<S, S> tracker = new ServiceTracker<S, S>(bundleContext, serviceClass, customizer);
        tracker.open();
        if (customizer instanceof ExtendedServiceTrackerCustomizer) {
            ((ExtendedServiceTrackerCustomizer<S, S>) customizer).onOpened();
        }
        registry.add(tracker);
    }

    public static final <S> void track(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final Class<S> serviceClass,
        final ServiceTrackerCustomizer<S, S> customizer) {
        final ServiceTracker<S, S> tracker = new ServiceTracker<S, S>(bundleContext, serviceClass, customizer);
        tracker.open();
        if (customizer instanceof ExtendedServiceTrackerCustomizer) {
            ((ExtendedServiceTrackerCustomizer<S, S>) customizer).onOpened();
        }
        registry.add(tracker);
    }    
    
    public static <S, T extends S> ServiceTracker<S, T> trackAndOpen(final BundleContext bundleContext,
        final ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies,
        final ImmutableSet<OptionalServiceDependency<?>> optionalDependencies,
        final ServiceTrackerCustomizer<S, T> registerer,
        final String serviceInstanceName) {
        final ServiceTracker<S, T> serviceTracker;
        try {
            final Filter filter = generateServiceFilter(bundleContext, mandatoryDependencies, optionalDependencies);
            if (filter != null) {
                serviceTracker = new ServiceTracker<S, T>(bundleContext, filter, registerer);
            } else {
                throw new RegistrationException(f("%s has no dependencies", serviceInstanceName));
            }
        } catch (final RegistrationException e) {
            throw e;
        } catch (final Throwable t) {
            ExceptionUtils.handleThrowable(t);
            final String msg = f("%s instantiation registration failed for %s: %s: %s",
                ServiceTracker.class.getSimpleName(),
                serviceInstanceName,
                className(t),
                t.getMessage());
            LOG.error(msg, t);
            throw new RegistrationException(msg, t);
        }
        try {
            serviceTracker.open();
        } catch (final Throwable t) {
            ExceptionUtils.handleThrowable(t);
            final String msg = f("opening %s failed for %s: %s: %s",
                ServiceTracker.class.getSimpleName(),
                serviceInstanceName,
                className(t),
                t.getMessage());
            LOG.error(msg, t);
            throw new RegistrationException(msg, t);
        }
        
        if (registerer instanceof ExtendedServiceTrackerCustomizer) {
            ((ExtendedServiceTrackerCustomizer<S, T>) registerer).onOpened();
        }
        
        return serviceTracker;
    }
    
    @SuppressWarnings("null")
    public static final ImmutableSet<Class<?>> KNOWN_INDIRECTIONS = ImmutableSet.<Class<?>>of(
        Remote.class
    );
    
    public static final Function<Object, String> TO_CLASS_NAME = new Function<Object, String>() {
        @Override
        public @Nullable String apply(final @Nullable Object input) {
            return input != null ? className(input) : null;
        }
    };

    public static final Function<Map.Entry<Class<?>, Object>, String> ENTRY_TO_STRING = new Function<Map.Entry<Class<?>, Object>, String>() {
        @Override
        public @Nullable String apply(final @Nullable Map.Entry<Class<?>, Object> input) {
            if (input == null) {
                return null;
            }
            final Class<?> serviceClass = input.getKey();
            final Object serviceInstance = input.getValue();
            return f("%s(%s)",
                serviceInstance != null ? className(serviceInstance) : null,
                serviceClass != null ? className(serviceClass) : null
            );
        }
    };    
    public static final Function<Class<?>, String> TO_FULL_CLASS_NAME = new Function<Class<?>, String>() {
        @Override
        public @Nullable String apply(final @Nullable Class<?> input) {
            return input != null ? className(input) : null;
        }
    };
    public static final Function<Class<?>, String> TO_SIMPLE_CLASS_NAME = new Function<Class<?>, String>() {
        @Override
        public @Nullable String apply(final @Nullable Class<?> input) {
            return input != null ? input.getSimpleName() : null;
        }
    };
    
    public static final Function<Object, String> CLASS_AWARE_TO_STRING = new Function<Object, String>() {
        @Override
        public @Nullable String apply(@Nullable Object input) {
            if (input == null) {
                return null;
            }
            if (input instanceof Class) {
                return ((Class<?>) input).getName();
            }
            return input.toString();
        }
    };

    public static final String[] concatArray(final String arg0, final String... moreArgs) {
        final String[] all;
        {
            if (moreArgs.length < 1) {
                all = new String[] { arg0 };
            } else {
                all = new String[moreArgs.length + 1];
                all[0] = arg0;
                System.arraycopy(moreArgs, 0, all, 1, moreArgs.length);
            }
        }
        return all;
    }
    
    public static final @Nullable <A extends Annotation> A declaredAnnotation(final Class<A> annotationClass, final Class<?> classToExamine) {
        final Annotation[] annotations = classToExamine.getDeclaredAnnotations();
        if (annotations == null) {
            return null;
        }
        for (final Annotation a : annotations) {
            if (a.getClass().equals(annotationClass)) {
                return annotationClass.cast(a);
            }
        }
        return null;
    }

    @SafeVarargs
    public static ImmutableSet<String> collectPropertiesOfInterest(final Collection<? extends ServiceDependency<?>>... serviceDependencies) {
        final Set<String> set = new HashSet<>();
        for (final Collection<? extends ServiceDependency<?>> collection : serviceDependencies) {
            for (final ServiceDependency<?> d : collection) {
                set.addAll(d.propertiesOfInterest());
            }
        }
        return NullUtil.immutableSetCopyOf(set);
    }

}
