/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.custom.i18n;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import com.google.common.collect.ImmutableMap;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.PropertyFilter;
import com.openexchange.configuration.ConfigurationExceptionCodes;
import com.openexchange.exception.OXException;


/**
 * {@link PropertyTranslations}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.2
 */
@ParametersAreNonnullByDefault
public enum PropertyTranslations {
    ;
    public static final PropertyTranslation load(final @Nullable ConfigurationService configService, final String propertyName) {
        if (configService == null) {
            return new InvalidPropertyTranslation(propertyName,
                notNull(ConfigurationExceptionCodes.INVALID_CONFIGURATION.create("ConfigurationService is null")));
        }
        
        final Pattern pattern;
        {
            pattern = Pattern.compile("^\\Q" + propertyName + ".\\E([^_\\.]_[^_\\.])$");
        }
        
        final Map<String, String> properties;
        try {
            properties = new HashMap<String, String>(configService.getProperties(new PropertyFilter() {
                @Override
                public boolean accept(@Nullable String name, @Nullable String value) throws OXException {
                    return name != null && (name.equals(propertyName)
                        || pattern.matcher(name).matches());
                }
            }));
        } catch (final OXException e) {
            return new InvalidPropertyTranslation(propertyName, e);
        }
        
        final String fallback;
        if (properties.containsKey(propertyName)) {
            fallback = notNull(properties.remove(propertyName));
        } else {
            return new InvalidPropertyTranslation(propertyName, notNull(
                ConfigurationExceptionCodes.PROPERTY_MISSING.create(propertyName)));
        }
        
        final Map<String, String> translations = new HashMap<>(properties.size());
        for (final Map.Entry<String, String> e : properties.entrySet()) {
            final String key = e.getKey();
            if (key != null) {
                final String lang;
                {
                    final Matcher m = pattern.matcher(key);
                    if (m.matches()) {
                        lang = m.group(1);
                    } else {
                        lang = null;
                    }
                }
                
                if (lang != null) {
                    final String value = e.getValue();
                    if (value == null || value.isEmpty() || value.trim().isEmpty()) {
                        return new InvalidPropertyTranslation(propertyName, notNull(ConfigurationExceptionCodes.INVALID_CONFIGURATION
                            .create(String.format("property '%s' is empty", value))));
                    }
                    translations.put(lang, value);
                }
            }
        }
        
        return new ValidPropertyTranslation(propertyName, ImmutableMap.copyOf(translations), fallback);
    }    

    private static final <X> X notNull(final @Nullable X value) {
        if (value != null) {
            return value;
        }
        throw new IllegalArgumentException("value is null");
    }
    
}
