/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.custom.base;

import static com.openexchange.util.custom.base.NullUtil.notNull;
import java.util.Map;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import org.apache.commons.text.StrLookup;
import org.apache.commons.text.StrSubstitutor;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableSet;
import com.openexchange.config.ConfigurationService;

/**
 * {@link StringInterpolator}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.3
 */
@ParametersAreNonnullByDefault
public final class StringInterpolator implements Function<String, String> {
    
    private static final class PlaceholderResolverTrackingStrLookup<T> extends StrLookup<T> {
        
        private final PlaceholderResolver resolver;
        private final ImmutableSet.Builder<String> interpolated = ImmutableSet.builder();
        
        public PlaceholderResolverTrackingStrLookup(final PlaceholderResolver resolver) {
            super();
            this.resolver = resolver;
        }

        public ImmutableSet<String> interpolated() {
            return interpolated.build();
        }

        @Override
        public @Nullable String lookup(final @Nullable String key) {
            if (key == null) {
                return null;
            }
            
            final @Nullable String defaultValue;
            final String placeholder;
            {
                final int defaultMarkIndex = key.indexOf("|default=");
                if (defaultMarkIndex >= 0) {
                    placeholder = notNull(key.substring(0, defaultMarkIndex));
                    defaultValue = notNull(key.substring(defaultMarkIndex + "|default=".length()));
                    if (StringUtil.isEmptyOrWhitespaces(placeholder)) {
                        throw new FailedInterpolationException(key, "placeholder has a default value but the placeholder itself is empty");
                    }
                } else {
                    placeholder = key;
                    defaultValue = null;
                }
            }
            
            final String value;
            try {
                value = resolver.resolve(placeholder);
            } catch (final FailedInterpolationException e) {
                throw e;
            } catch (final Exception e) {
                throw new FailedInterpolationException(placeholder, e);
            }
            if (value == null) {
                if (defaultValue != null) {
                    interpolated.add(placeholder);                    
                    return defaultValue;
                } else {
                    throw new FailedInterpolationException(placeholder);
                }
            }
            interpolated.add(placeholder);
            return value;
        }
    }
    
    private enum NullPlaceholderResolver implements PlaceholderResolver {
        INSTANCE;
        @Override
        public @Nullable String resolve(String placeholder) throws Exception {
            return "";
        }
    }
    
    public static ImmutableSet<String> placeholders(final String template) {
        return new StringInterpolator(NullPlaceholderResolver.INSTANCE).interpolate(template).interpolated();
    }
    
    public static final StringInterpolator forConfigurationService(final ConfigurationService configurationService) {
        return new StringInterpolator(new ConfigurationServicePlaceholderResolver(configurationService));           
    }
    public static final StringInterpolator forMap(final Map<String, ?> map) {
        return new StringInterpolator(new MapPlaceholderResolver(map));
    }
    
    private final PlaceholderResolver resolver;
    
    public StringInterpolator(final PlaceholderResolver resolver) {
        this.resolver = resolver;
    }

    @Override
    @Nullable
    public String apply(@Nullable String input) throws FailedInterpolationException {
        if (input != null) {
            return interpolate(input).result();
        } else {
            return null;
        }
    }
    
    public StringInterpolation interpolate(final CharSequence input) throws FailedInterpolationException {
        return interpolate(input, "${", "}");
    }
    
    public StringInterpolation interpolate(final CharSequence input, final String prefix, final String suffix) throws FailedInterpolationException {
        final PlaceholderResolverTrackingStrLookup<String> lookup = new PlaceholderResolverTrackingStrLookup<String>(resolver);
        final String result = new StrSubstitutor(lookup, prefix, suffix, '\\').replace(input);
        if (result == null) {
            throw new IllegalStateException("interpolation result is null");
        }
        return new StringInterpolation(result, lookup.interpolated());
    }    
    
}
