/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.activator.impl.RegistrationTools.TO_SIMPLE_CLASS_NAME;
import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.logger;
import java.util.Dictionary;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;
import javax.servlet.http.HttpServlet;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.http.HttpService;
import org.slf4j.Logger;
import com.google.common.base.Joiner;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.Interests;
import com.openexchange.config.Reloadable;
import com.openexchange.config.Reloadables;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.util.activator.RegistrationException;
import com.openexchange.util.activator.ServiceDependencyResolver;
import com.openexchange.util.custom.base.FailedInterpolationException;
import com.openexchange.util.custom.base.StringInterpolation;
import com.openexchange.util.custom.base.StringInterpolator;


/**
 * {@link ServletServiceRegistererTemplate}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.2
 */
@ParametersAreNonnullByDefault
public abstract class ServletServiceRegistererTemplate<T extends HttpServlet> extends ServiceAwareDependentServiceRegistererTemplate<HttpServlet, T> {

    private static final Logger LOG = logger(ServletServiceRegistererTemplate.class);
    
    public static ImmutableSet<MandatoryServiceDependency<?>> withMandatoryServletDependencies(
        final Iterable<MandatoryServiceDependency<?>> neededServices,
        final String path) {
        final ImmutableSet.Builder<MandatoryServiceDependency<?>> b = ImmutableSet.builder();
        b.addAll(neededServices);
        b.add(new MandatoryServiceDependency<>(HttpService.class));
        if (path.contains("${")) {
            b.add(new MandatoryServiceDependency<>(ConfigurationService.class));
        }
        return b.build();
    }
    
    public static ImmutableSet<OptionalServiceDependency<?>> withOptionalServletDependencies(
        final Iterable<OptionalServiceDependency<?>> neededServices,
        final String path) {
        return ImmutableSet.copyOf(neededServices);
    }

    private final class ServletPathReloader implements Reloadable {

        private final HttpService httpService;
        private final ImmutableSet<String> properties;
        private final String pathTemplate;
        private final HttpServlet serviceInstance;
        private final Dictionary<String, ?> initParams;
        
        public ServletPathReloader(final HttpService httpService,
            final ImmutableSet<String> properties,
            final String pathTemplate,
            final String path,
            final HttpServlet serviceInstance,
            final Dictionary<String, ?> initParams) {
            this.httpService = httpService;
            this.properties = properties;
            this.pathTemplate = pathTemplate;
            this.serviceInstance = serviceInstance;
            this.initParams = initParams;
        }

        @Override
        public void reloadConfiguration(final @Nullable ConfigurationService configService) {
            if (configService != null) {
                final String newAlias;
                try {
                    newAlias = StringInterpolator.forConfigurationService(configService)
                        .interpolate(pathTemplate)
                        .result();
                } catch (final Exception e) {
                    final String oldAlias = aliasRef.getAndSet(null);
                    if (oldAlias != null) {
                        try {
                            httpService.unregister(oldAlias);
                            LOG.error(f("failed to interpolate the new path for %s from the path template '%s', servlet has been unregistered: %s: %s",
                                serviceInstanceName, pathTemplate, e.getClass().getName(), e.getMessage()));
                        } catch (final IllegalArgumentException iae) {
                            // nothing was registered under oldAlias
                            LOG.error(f("failed to interpolate the new path for %s from the path template '%s' (was not registered): %s: %s",
                                serviceInstanceName, pathTemplate, e.getClass().getName(), e.getMessage()));
                        }
                        
                    } else {
                        LOG.error(f("failed to interpolate the new path for %s from the path template '%s' (was not registered): %s: %s",
                            serviceInstanceName, pathTemplate, e.getClass().getName(), e.getMessage()));
                    }
                    return;
                }
                
                final String oldAlias = aliasRef.getAndSet(newAlias);
                if (! newAlias.equals(oldAlias)) {
                    if (oldAlias != null) {
                        try {
                            httpService.unregister(oldAlias);
                            LOG.debug("un-registered {} due to path replacement (changing from '{}' to '{}')",
                                serviceInstanceName, oldAlias, newAlias);
                        } catch (final IllegalArgumentException iae) {
                            // nothing was registered under oldAlias
                        }
                    }
                    try {
                        httpService.registerServlet(newAlias, serviceInstance, initParams, null);
                        if (LOG.isInfoEnabled()) {
                            if (oldAlias != null) {
                                LOG.info("re-registered {} under new path '{}' (was under '{}')", serviceInstanceName, newAlias, oldAlias);
                            } else {
                                LOG.info("registered {} under path '{}' (was not registered)", serviceInstanceName, newAlias, oldAlias);
                            }
                        }
                    } catch (final Exception e) {
                        aliasRef.compareAndSet(newAlias, null);
                        final String msg = f("failed to re-register %s (%s) with %s under '%s'",
                            serviceInstanceName,
                            FluentIterable.from(serviceTypes).transform(TO_SIMPLE_CLASS_NAME).join(Joiner.on(", ")),
                            HttpService.class.getName(),
                            newAlias);
                        LOG.error(msg, e);
                    }
                }
            }
        }

        @Override
        public @Nullable Interests getInterests() {
            return Reloadables.interestsForProperties(properties.toArray(new String[0]));
        }
        
    }
    
    @Immutable
    private static final class ServletRegistrationSet<T> {
        public final T servletInstance;
        public final HttpService httpService;
        public ServletRegistrationSet(final T servletInstance, final HttpService httpService) {
            this.servletInstance = servletInstance;
            this.httpService = httpService;
        }
    }
    
    private final String pathTemplate;
    private final AtomicReference<ServiceRegistration<ServletPathReloader>> reloaderRegistration;
    private final AtomicReference<ServletRegistrationSet<T>> registration;
    private final AtomicReference<String> aliasRef;
    
    protected ServletServiceRegistererTemplate(final String path,
        final BundleContext context,
        final ImmutableSet<Class<?>> serviceTypes,
        final String serviceInstanceName,
        final Dictionary<String, ?> properties,
        final Iterable<MandatoryServiceDependency<?>> mandatoryServices,
        final Iterable<OptionalServiceDependency<?>> optionalServices,
        final ImmutableMap<Class<?>, ServiceSet<?>> serviceSets,
        final ImmutableMap<Class<?>, ServiceListing<?>> serviceListings,
        final Set<String> propertiesOfInterest) {
        super(context, serviceTypes, serviceInstanceName, properties,
            withMandatoryServletDependencies(mandatoryServices, path),
            withOptionalServletDependencies(optionalServices, path),
            serviceSets, serviceListings, propertiesOfInterest);
        this.aliasRef = new AtomicReference<>(null);
        this.registration = new AtomicReference<>(null);
        this.reloaderRegistration = new AtomicReference<>(null);
        this.pathTemplate = path;
    }
    
    protected abstract T createInstance(ServiceDependencyResolver resolver) throws Exception;

    @Override
    protected @Nullable Object register(final @Nullable ServiceDependencyResolver resolver) {
        if (resolver == null) {
            throw new IllegalArgumentException(f("%s parameter is null", ServiceDependencyResolver.class.getName()));
        }
        final HttpService httpService = resolver.resolveMandatoryService(HttpService.class, null);
        if (httpService == null) {
            throw new RegistrationException(f("%s not available even though register() is invoked", HttpService.class.getName()));
        }
        
        final T serviceInstance;
        try {
            serviceInstance = createInstance(resolver);
        } catch (final Exception e) {
            final String msg = f("failed to create instance of %s (%s)",
                serviceInstanceName,
                FluentIterable.from(serviceTypes).transform(TO_SIMPLE_CLASS_NAME).join(Joiner.on(", ")));
            LOG.error(msg, e);
            throw new RegistrationException(msg, e);
        }
        
        final String resolvedPath;
        final ImmutableSet<String> reloadableProps;
        if (pathTemplate.contains("${")) {
            final ConfigurationService configurationService = resolver.resolveMandatoryService(ConfigurationService.class, null);
            if (configurationService == null) {
                final String msg = f("%s not available even though register() is invoked (needed because the pathTemplate '%s' is a template",
                    ConfigurationService.class.getName(), pathTemplate);
                LOG.error(msg);
                throw new RegistrationException(msg);
            }
            final StringInterpolation interpolation;
            try {
                interpolation = StringInterpolator.forConfigurationService(configurationService).interpolate(pathTemplate);
            } catch (final FailedInterpolationException e) {
                final String msg = f("failed to resolve path template '%s' for %s using %s: %s",
                    pathTemplate,  serviceInstanceName, ConfigurationService.class.getSimpleName(), e.getMessage());
                LOG.error(msg, e);
                throw new RegistrationException(msg, e);
            }
            resolvedPath = interpolation.result();
            if (LOG.isDebugEnabled()) {
                LOG.debug("{}: pathTemplate template '{}' resolved as '{}'", serviceInstanceName, pathTemplate, resolvedPath);
            }
            reloadableProps = interpolation.interpolated();
        } else {
            resolvedPath = pathTemplate;
            reloadableProps = ImmutableSet.of();
        }
        
        try {
            httpService.registerServlet(resolvedPath, serviceInstance, properties, null);
        } catch (final Exception e) {
            final String msg = f("failed to register %s (%s) with %s under '%s'",
                serviceInstanceName,
                FluentIterable.from(serviceTypes).transform(TO_SIMPLE_CLASS_NAME).join(Joiner.on(", ")),
                HttpService.class.getName(),
                resolvedPath);
            LOG.error(msg, e);
            throw new RegistrationException(msg, e);
        }
        
        if (! this.aliasRef.compareAndSet(null, resolvedPath)) {
            final String msg = f("servlet %s was already registered under '%s'", serviceInstanceName, resolvedPath);
            LOG.error(msg);
            throw new RegistrationException(msg);
        }

        if (! this.registration.compareAndSet(null, new ServletRegistrationSet<T>(serviceInstance, httpService))) {
            final String msg = f("servlet %s was already registered under '%s'", serviceInstanceName, resolvedPath);
            LOG.error(msg);
            throw new RegistrationException(msg);
        }
        
        if (! reloadableProps.isEmpty()) {
            final ServletPathReloader reloader = new ServletPathReloader(httpService, reloadableProps, pathTemplate, resolvedPath, serviceInstance, properties);
            final ServiceRegistration<ServletPathReloader> r = RegistrationTools.registerIntoContext(
                context,
                ImmutableSet.of(Reloadable.class),
                reloader,
                properties
            );
            this.reloaderRegistration.set(r);
        }
        
        return serviceInstance;
    }

    @Override
    protected void unregister(final @Nullable Object service) {
        try {
            final ServiceRegistration<?> reloader = this.reloaderRegistration.getAndSet(null);
            if (reloader != null) {
                context.ungetService(reloader.getReference());
            }
        } finally {
            final ServletRegistrationSet<T> unregister = this.registration.getAndSet(null);
            if (unregister != null) {
                try {
                    final String alias = aliasRef.getAndSet(null);
                    if (alias != null) {
                        unregister.httpService.unregister(alias);
                    }
                } finally {
                    RegistrationTools.closeInstance(unregister.servletInstance, serviceInstanceName, serviceTypes);
                }
            }
        }
    }

}
