/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.activator.impl.RegistrationTools.propertyAnnotation;
import static com.openexchange.util.activator.impl.RegistrationTools.serviceAnnotation;
import static com.openexchange.util.activator.impl.RegistrationTools.serviceClassAnnotation;
import static com.openexchange.util.activator.impl.RegistrationTools.setOfServicesAnnotation;
import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.logger;
import static com.openexchange.util.custom.base.NullUtil.notNull;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import org.slf4j.Logger;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.util.activator.Property;
import com.openexchange.util.activator.RegistrationException;
import com.openexchange.util.activator.Service;
import com.openexchange.util.activator.ServiceClass;
import com.openexchange.util.activator.ServiceDependencyResolver;
import com.openexchange.util.activator.ServiceFactory;
import com.openexchange.util.activator.ServiceRef;
import com.openexchange.util.activator.SetOfServices;


/**
 * {@link ServiceFactoryWithDependencies}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.2
 */
@ParametersAreNonnullByDefault
public final class ServiceFactoryWithDependencies<S, T extends S> extends ServiceWithDependenciesTemplate<S, T> {
    
    private static final Logger LOG = logger(ServiceFactoryWithDependencies.class);

    private static final <T> ImmutableList<Method> serviceFactoryMethods(final @Nullable Method[] methods) {
        if (methods == null) {
            return ImmutableList.of();
        }
        final List<Method> list = new ArrayList<>();
        for (final Method m : methods) {
            if (m.getAnnotation(Service.class) != null) {
                if (Void.class.equals(m.getReturnType())) {
                    throw new RegistrationException(f("@%s factory method returns void: %s",
                        Service.class.getSimpleName(), m.toGenericString()));
                }
                if (! m.isAccessible()) {
                    throw new RegistrationException(f("@%s factory method is not accessible: %s",
                        Service.class.getSimpleName(), m.toGenericString()));
                }
                list.add(m);
            }
        }
        if (list.isEmpty()) {
            for (final Method m : methods) {
                if (m.isAccessible() && m.getName().startsWith("create") && ! Void.class.equals(m.getReturnType())) {
                    list.add(m);
                }
            }
        }
        return ImmutableList.copyOf(list);
    }
    
    public static final <X> ServiceWithDependencies<X> find(final ServiceFactory<X> serviceFactory) {
        final Set<MandatoryServiceDependency<?>> mandatoryServices = new HashSet<>();
        final Set<OptionalServiceDependency<?>> optionalServices = new HashSet<>();
        final Set<Class<?>> serviceSets = new HashSet<Class<?>>();
        final Set<Class<?>> serviceListings = new HashSet<Class<?>>();
        final Set<String> properties = new HashSet<String>();
        final Method ctor;
        {
            final Method[] methods = serviceFactory.getClass().getDeclaredMethods();
            if (methods.length < 1) {
                throw new RegistrationException(f("%s has no methods", className(serviceFactory)));
            }
            if (methods.length > 1) {
                final List<Method> serviceCtors = serviceFactoryMethods(methods);
                if (serviceCtors.isEmpty()) {
                    throw new RegistrationException(f("%s has no factory methods",
                        className(serviceFactory), Service.class.getSimpleName()));
                } else if (serviceCtors.size() > 1) {
                    throw new RegistrationException(f("%s has more than one factory method with the @%s annotation",
                        className(serviceFactory), Service.class.getSimpleName()));
                }
                ctor = serviceCtors.get(0);
            } else {
                ctor = methods[0];
            }
            
            final Class<?>[] paramTypes = ctor.getParameterTypes();
            final Annotation[][] paramAnnotations = ctor.getParameterAnnotations();
            final int numParams = paramTypes.length;
            for (int i = 0; i < numParams; i++) {
                final @Nullable SetOfServices setOfServicesAnnotation = setOfServicesAnnotation(paramAnnotations[i]);
                if (setOfServicesAnnotation != null) {
                    if (ServiceSet.class.isAssignableFrom(paramTypes[i])) {
                        final Class<?> dependencyClass = setOfServicesAnnotation.value();
                        serviceSets.add(dependencyClass);
                    } else if (ServiceListing.class.isAssignableFrom(paramTypes[i])) {
                        final Class<?> dependencyClass = setOfServicesAnnotation.value();
                        serviceListings.add(dependencyClass);
                    } else {
                        throw new RegistrationException(f("parameter %d in constructor '%s' of service class %s is annotated with @%s but the parameter type %s is not assignable a %s or a %s",
                            i + 1, ctor.toGenericString(), className(serviceFactory),
                            SetOfServices.class.getSimpleName(),
                            paramTypes[i].getName(),
                            ServiceSet.class.getSimpleName(),
                            ServiceListing.class.getSimpleName()));
                    }
                } else if (ServiceRef.class == paramTypes[i]) {
                    final @Nullable ServiceClass serviceClass = serviceClassAnnotation(paramAnnotations[i]);
                    if (serviceClass == null) {
                        throw new RegistrationException(f("parameter %d in constructor '%s' of service class %s has an optional service holder parameter of type %s but not the mandatory annotation @%s on that parameter",
                            i + 1, ctor.toGenericString(), paramTypes[i].getName(), ServiceClass.class.getName())); 
                    } else {
                        final Class<?> dependencyClass = serviceClass.value();
                        final AtomicServiceRef<?> sr = AtomicServiceRef.forClass(dependencyClass);
                        final @Nullable String id;
                        {
                            final @Nullable Service serviceAnnotation = serviceAnnotation(paramAnnotations[i]);
                            id = serviceAnnotation != null ? serviceAnnotation.id() : null;
                        }
                        if (id == null || id.isEmpty()) {
                            optionalServices.add(
                                new OptionalServiceDependency<>(sr, ServiceDependencyQualifiers.NO_ID_QUALIFIER)
                            );
                        } else if ("*".equals(id)) {
                            optionalServices.add(
                                new OptionalServiceDependency<>(sr, ServiceDependencyQualifiers.ANY_ID_QUALIFIER)
                            );
                        } else {
                            optionalServices.add(
                                new OptionalServiceDependency<>(sr, ServiceDependencyQualifiers.mustMatchId(id))
                            );
                        }
                    }
                } else {
                    final @Nullable Property propertyAnnotation = propertyAnnotation(paramAnnotations[i]);
                    if (propertyAnnotation != null) {
                        final String name = propertyAnnotation.value();
                        properties.add(name);
                    } else {
                        final Class<?> dependencyClass = paramTypes[i];
                        if (dependencyClass != null) {
                            final @Nullable String id;
                            {
                                final @Nullable Service serviceAnnotation = serviceAnnotation(paramAnnotations[i]);
                                id = serviceAnnotation != null ? serviceAnnotation.id() : null;
                            }
                            if (id == null || id.isEmpty()) {
                                mandatoryServices.add(
                                    new MandatoryServiceDependency<>(dependencyClass, ServiceDependencyQualifiers.NO_ID_QUALIFIER)
                                );
                            } else if ("*".equals(id)) {
                                mandatoryServices.add(
                                    new MandatoryServiceDependency<>(dependencyClass, ServiceDependencyQualifiers.ANY_ID_QUALIFIER)
                                );
                            } else {
                                mandatoryServices.add(
                                    new MandatoryServiceDependency<>(dependencyClass, ServiceDependencyQualifiers.mustMatchId(id))
                                );
                            }                            
                        }
                    }
                }
            }
        }
        return new ServiceFactoryWithDependencies<X, X>(serviceFactory, ctor,
            ImmutableSet.copyOf(mandatoryServices),
            ImmutableSet.copyOf(optionalServices),
            ImmutableSet.copyOf(serviceSets),
            ImmutableSet.copyOf(serviceListings)
        );
    }
    
    private final ServiceFactory<T> serviceFactory;
    private final Method factoryMethod;
    
    @SuppressWarnings("unchecked")
    public ServiceFactoryWithDependencies(ServiceFactory<T> serviceFactory,
        Method factoryMethod,
        ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies,
        ImmutableSet<OptionalServiceDependency<?>> optionalDependencies,
        ImmutableSet<Class<?>> serviceSets,
        ImmutableSet<Class<?>> serviceListings) {
        super(
            (Class<T>) notNull(factoryMethod.getReturnType()),
            (Class<S>) null,
            mandatoryDependencies,
            optionalDependencies,
            serviceSets,
            serviceListings
        );
            
        this.serviceFactory = serviceFactory;
        this.factoryMethod = factoryMethod;
    }
    
    @Override
    public T createInstance(ServiceDependencyResolver resolver) {
        final Optional<Object[]> values = resolver.resolveParameters(factoryMethod);
        if (values.isPresent()) {
            try {
                LOG.debug("invoking factory method {}:{}", className(serviceFactory), factoryMethod.getName());
                @SuppressWarnings("unchecked")
                final T serviceInstance = (T) factoryMethod.invoke(serviceFactory, values.get());
                if (serviceInstance == null) {
                    final String msg = f("null result when invoking factory method %s:%s with parameters [%s]",
                        className(serviceFactory),
                        factoryMethod.toGenericString(),
                        FluentIterable.of(values.get()).transform(RegistrationTools.TO_CLASS_NAME).join(Joiner.on(", ")));
                    LOG.error(msg);
                    throw new RegistrationException(msg);
                }
                return serviceInstance;
            } catch (final Exception e) {
                Throwable t = e;
                if (t instanceof InvocationTargetException) {
                    final Throwable c = t.getCause();
                    if (c != null) {
                        t = c;
                    }
                }
                LOG.error(f("failed to invoke factory method %s:%s with parameters [%s]: %s: %s",
                    className(serviceFactory),
                    factoryMethod.toGenericString(),
                    FluentIterable.of(values.get()).transform(RegistrationTools.TO_CLASS_NAME).join(Joiner.on(", ")),
                    className(t), t.getMessage()), t);
                if (t instanceof RegistrationException) {
                    throw (RegistrationException) t;
                } else if (t instanceof RuntimeException) {
                    throw (RuntimeException) t;
                } else {
                    final String msg = t.getMessage();
                    if (msg != null) {
                        throw new RegistrationException(msg, t);
                    } else {
                        throw new RegistrationException(className(t), t);
                    }
                }
            }
        } else {
            final String msg = f("%s: failed to find matching services for factory method %s:%s",
                className(serviceFactory), factoryMethod.toGenericString());
            LOG.error(msg);
            throw new RegistrationException(msg);
        }          
    }

}
