/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.notNull;
import java.lang.annotation.Annotation;
import java.util.Set;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceReference;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Predicates;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;


/**
 * {@link ServiceDependencyTemplate}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.3.5
 */
@ParametersAreNonnullByDefault
public abstract class ServiceDependencyTemplate<S> implements ServiceDependency<S> {
    
    private final class ServiceClassQualifier implements ServiceDependencyQualifier {

        @Override
        public @Nullable String contributeFilter() {
            for (final Class<?> indirection : RegistrationTools.KNOWN_INDIRECTIONS) {
                if (indirection.isAssignableFrom(serviceClass)) {
                    return f("(|(%s=%s)(%s=%s))",
                        Constants.OBJECTCLASS, serviceClass.getName(),
                        Constants.OBJECTCLASS, indirection.getName()
                    );
                }
            }
            return f("(%s=%s)", Constants.OBJECTCLASS, serviceClass.getName());
        }

        @Override
        public @Nullable String contributeIdentification() {
            return f("%s", serviceClass.getName());
        }
        
        @Override
        public boolean matchesService(final Object serviceInstance, final ServiceReference<?> reference) {
            return ServiceDependencyTemplate.this.serviceClass.isInstance(serviceInstance) && reference.getProperty("id") == null;
        }
        
        @Override
        public boolean matchesDependency(Class<?> serviceClass, @Nullable Annotation[] annotations) {
            return ServiceDependencyTemplate.this.serviceClass.isAssignableFrom(serviceClass);
        }

        @Override
        public @Nullable Set<String> contributePropertiesOfInterest() {
            return null;
        }
         
    }
    
    private enum IdentificationPartFunction implements Function<ServiceDependencyQualifier, String> {
        INSTANCE;
        @Override
        public @Nullable String apply(final @Nullable ServiceDependencyQualifier input) {
            return input != null ? input.contributeIdentification() : null;
        }
    }

    private enum OsgiFilterFunction implements Function<ServiceDependencyQualifier, String> {
        INSTANCE;
        @Override
        public @Nullable String apply(final @Nullable ServiceDependencyQualifier input) {
            return input != null ? input.contributeFilter() : null;
        }
    }

    private enum WrapInBracesFunction implements Function<String, String> {
        INSTANCE;
        @Override
        public @Nullable String apply(final @Nullable String input) {
            return input != null ? new StringBuilder(input.length() + 2).append("(").append(input).append(")").toString() : null;
        }
    }
    
    private static final class WrapButFirst implements Function<String, String> {
        private final String before;
        private final String after;
        private int index = 0;
        public WrapButFirst(String before, String after) {
            this.before = before;
            this.after = after;
        }
        
        @Override
        public @Nullable String apply(@Nullable String input) {
            index++;
            if (index == 1) {
                return input;
            } else if (input == null) {
                return null;
            } else {
                return new StringBuilder(input.length() + before.length() + after.length())
                    .append(before).append(input).append(after).toString();
            }
        }
        
    }
    
    private final String identification;
    private final String filter;
    private final Class<S> serviceClass;
    private final ImmutableList<ServiceDependencyQualifier> qualifiers;
    private final ImmutableSet<String> propertiesOfInterest;
    
    protected ServiceDependencyTemplate(Class<S> serviceClass, ServiceDependencyQualifier... qualifiers) {
        super();
        this.serviceClass = serviceClass;
        {
            final ImmutableList.Builder<ServiceDependencyQualifier> b = ImmutableList.builder();
            b.add(new ServiceClassQualifier());
            b.add(qualifiers);
            this.qualifiers = b.build();
        }
        
        this.identification = FluentIterable.from(this.qualifiers)
            .transform(IdentificationPartFunction.INSTANCE)
            .filter(Predicates.notNull())
            .transform(new WrapButFirst("[", "]"))
            .join(Joiner.on(""));
        {
            final ImmutableList<String> filterParts = FluentIterable.from(this.qualifiers)
                .transform(OsgiFilterFunction.INSTANCE)
                .filter(Predicates.notNull())
                .toList();
            if (filterParts.isEmpty()) {
                throw new IllegalArgumentException("empty OSGi filter expression");
            }
            if (filterParts.size() == 1) {
                this.filter = notNull(filterParts.get(0));
            } else {
                this.filter = notNull(new StringBuilder()
                    .append("(&(")
                    .append(FluentIterable.from(filterParts).transform(WrapInBracesFunction.INSTANCE).join(Joiner.on("")))
                    .append("))")
                    .toString());
            }
        }
        {
            final ImmutableSet.Builder<String> b = ImmutableSet.builder();
            for (final ServiceDependencyQualifier qualifier : this.qualifiers) {
                final Set<String> p = qualifier.contributePropertiesOfInterest();
                if (p != null) {
                    b.addAll(p);
                }
            }
            this.propertiesOfInterest = b.build();
        }
    }


    @Override
    public final String identify() {
        return identification;
    }
    
    
    @Override
    public final String asFilter() {
        return filter;
    }

    @Override
    public final Set<String> propertiesOfInterest() {
        return propertiesOfInterest;
    }

    @Override
    public final boolean matchesService(final Object serviceInstance, final ServiceReference<?> reference) {
        for (final ServiceDependencyQualifier qualifier : qualifiers) {
            if (! qualifier.matchesService(serviceInstance, reference)) {
                return false;
            }
        }
        return true;
    }
    
    @Override
    public final boolean matchesDependency(final Class<?> serviceClass, final @Nullable Annotation[] annotations) {
        for (final ServiceDependencyQualifier qualifier : qualifiers) {
            if (! qualifier.matchesDependency(serviceClass, annotations)) {
                return false;
            }
        }
        return true;
    }
    
}
