/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.custom.collect;

import java.util.Dictionary;
import java.util.Enumeration;
import java.util.Map;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterators;


/**
 * {@link ImmutableDictionary}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.1
 */
@ParametersAreNonnullByDefault
public final class ImmutableDictionary<K,V> extends Dictionary<K,V> {
    
    public static final <X,Y> ImmutableDictionary<X,Y> of(final @Nullable ImmutableMap<X,Y> map) {
        if (map != null) {
            return new ImmutableDictionary<>(map);
        } else {
            return new ImmutableDictionary<>(ImmutableMap.<X,Y>of());
        }
    }

    public static final <X,Y> ImmutableDictionary<X,Y> copyOf(final @Nullable Map<X,Y> map) {
        if (map != null) {
            return new ImmutableDictionary<>(ImmutableMap.copyOf(map));
        } else {
            return new ImmutableDictionary<>(ImmutableMap.<X,Y>of());
        }
    }
    
    public static final <X,Y> ImmutableDictionary<X,Y> copyOf(final @Nullable Dictionary<X,Y> dictionary) {
        if (dictionary != null) {
            final ImmutableMap.Builder<X, Y> builder = ImmutableMap.builder();
            final Enumeration<X> e = dictionary.keys();
            while (e.hasMoreElements()) {
                final X key = e.nextElement();
                if (key != null) {
                    final Y value = dictionary.get(key);
                    if (value != null) {
                        builder.put(key, value);
                    }
                }
            }
            return new ImmutableDictionary<>(builder.build());
        } else {
            return new ImmutableDictionary<>(ImmutableMap.<X,Y>of());
        }
    }
    
    private final ImmutableMap<K, V> map;
    
    private ImmutableDictionary(final ImmutableMap<K, V> map) {
        this.map = map;
    }

    @Override
    public int size() {
        return map.size();
    }

    @Override
    public boolean isEmpty() {
        return map.isEmpty();
    }

    @Override
    public Enumeration<K> keys() {
        return Iterators.asEnumeration(map.keySet().iterator());
    }

    @Override
    public Enumeration<V> elements() {
        return Iterators.asEnumeration(map.values().iterator());
    }

    @Override
    public V get(@Nullable Object key) {
        return map.get(key);
    }

    @Override
    public V put(@Nullable K key, @Nullable V value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public V remove(@Nullable Object key) {
        throw new UnsupportedOperationException();
    }

}
