/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.custom.base;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Utility functions for dealing with @Null and @NonNull values.
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.2.0
 */
@ParametersAreNonnullByDefault
public enum NullUtil {
    ;

    /**
     * Get an SLF4J {@link Logger} without having to ignore the null warning.
     * <p>
     * From its implementation, it is known that {@link LoggerFactory} does not
     * return null values.
     * 
     * @param c the {@link Class} for which to get a {@link Logger}
     * @return a {@link Logger} for the specified {@link Class}
     */
    @SuppressWarnings("null")
    public static Logger logger(final Class<?> c) {
        return LoggerFactory.getLogger(c);
    }
    
    /**
     * Check a value as being non-null.
     * <p>
     * Note that it does perform a null check and that it throws an {@link IllegalArgumentException}
     * whenever the value actually is {@code null}, using the specified {@code name} parameter
     * in the exception message.
     * 
     * @param value the value
     * @param name the name of the parameter, or value
     * @return the non-{@code null} value
     * @throws IllegalArgumentException thrown if the value is null
     */
    public static <X> X notNull(final @Nullable X value, final String name) throws IllegalArgumentException {
        if (value == null) {
            throw new IllegalArgumentException(name + " is null");
        }
        return value;
    }

    /**
     * "Cast" a non-null value as such, when you know for sure it isn't {@code null}.
     * <p>
     * Note that it does perform a null check and that it throws an {@link IllegalArgumentException}
     * whenever the value actually is {@code null}.
     * 
     * @param value the value
     * @return the non-{@code null} value
     * @throws IllegalArgumentException thrown if the value is null
     */    
    public static <X> X notNull(final @Nullable X value) throws IllegalArgumentException {
        if (value == null) {
            throw new IllegalArgumentException("unexpected null value");
        }
        return value;
    }
    
    /**
     * Perform a {@code String.format()} without having to null-check the result.
     * <p>
     * {@link String#format} is known for never returning {@code null} yet Eclipse doesn't know
     * about that and in order to avoid having to null-check the result of that frequently used
     * method, use this one instead.
     * 
     * @param format
     * @param args
     * @return
     */
    public static String f(final String format, final @Nullable Object... args) {
        final String result = String.format(format, args);
        if (result == null) {
            throw new IllegalStateException("result of String.format is null");
        }
        return result;
    }

    /**
     * Return the class name of a {@link Class} or an {@link Object}.
     * @param classOrObject can be a {@link Class} or an {@link Object}
     * @return the class name of the {@link Class} or {@link Object}
     */
    @SuppressWarnings("null")
    public static String className(final Object classOrObject) {
        if (classOrObject.getClass().equals(Class.class)) {
            return ((Class<?>) classOrObject).getName();
        } else {
            return classOrObject.getClass().getName();
        }
    }
    
}
