/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Dictionary;
import java.util.Map;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableMap;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.util.activator.Service;
import com.openexchange.util.activator.ServiceDependencyResolver;
import com.openexchange.util.custom.collect.ImmutableDictionary;


/**
 * {@link ServiceDependencyResolverImpl}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.2
 */
@ParametersAreNonnullByDefault
@Immutable
public final class ServiceDependencyResolverImpl implements ServiceDependencyResolver {

    private final ImmutableMap<ServiceDependency<?>, Object> dependencies;
    private final ImmutableMap<Class<?>, ServiceSet<?>> serviceSets;
    private final ImmutableMap<Class<?>, ServiceListing<?>> serviceListings;
    private final ImmutableDictionary<String, ?> properties;
    
    public ServiceDependencyResolverImpl(Map<ServiceDependency<?>, Object> dependencies,
        Map<Class<?>, ServiceSet<?>> serviceSets,
        Map<Class<?>, ServiceListing<?>> serviceListings,
        Map<String, ?> properties) {
        this.dependencies = ImmutableMap.copyOf(dependencies);
        this.serviceSets = ImmutableMap.copyOf(serviceSets);
        this.serviceListings = ImmutableMap.copyOf(serviceListings);
        this.properties = ImmutableDictionary.copyOf(properties);
    }

    public ServiceDependencyResolverImpl(Map<ServiceDependency<?>, Object> dependencies,
        Map<Class<?>, ServiceSet<?>> serviceSets,
        Map<Class<?>, ServiceListing<?>> serviceListings,
        Dictionary<String, ?> properties) {
        this.dependencies = ImmutableMap.copyOf(dependencies);
        this.serviceSets = ImmutableMap.copyOf(serviceSets);
        this.serviceListings = ImmutableMap.copyOf(serviceListings);
        this.properties = ImmutableDictionary.copyOf(properties);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    @Nullable
    public <X> X resolveService(final Class<X> serviceClass, final @Nullable Service serviceAnnotation) {
        for (final Map.Entry<ServiceDependency<?>, Object> e : dependencies.entrySet()) {
            if (e.getKey().matchesDependency(serviceClass, serviceAnnotation)) {
                return (X) e.getValue();
            }
        }
        return null;
    }

    @Override
    public Optional<Object[]> resolveParameters(Constructor<?> constructor) {
        return RegistrationTools.pickDependencies(
            constructor,
            this.dependencies,
            this.serviceSets,
            this.serviceListings,
            this.properties
        );
    }

    @Override
    public Optional<Object[]> resolveParameters(Method method) {
        return RegistrationTools.pickDependencies(
            method,
            this.dependencies,
            this.serviceSets,
            this.serviceListings,
            this.properties
        );
    }
}
