/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.f;
import java.util.Objects;
import java.util.Set;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceReference;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableSet;
import com.openexchange.util.activator.Service;


/**
 * {@link ServiceClassWithIdDependency}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.1
 */
@ParametersAreNonnullByDefault
public final class ServiceClassWithIdDependency<S> implements ServiceDependency<S> {

    public static final <X> ServiceClassWithIdDependency<X> ofClassAndId(final Class<X> serviceClass,
        final String id) {
        return new ServiceClassWithIdDependency<>(serviceClass, id);
    }
    
    private static final ImmutableSet<String> INTERESTS = ImmutableSet.of("id");
    
    private final Class<S> serviceClass;
    private final String id;
    
    public ServiceClassWithIdDependency(final Class<S> serviceClass, final String id) {
        this.serviceClass = serviceClass;
        this.id = id;
    }

    @Override
    public String asFilter() {
        for (final Class<?> indirection : RegistrationTools.KNOWN_INDIRECTIONS) {
            if (indirection.isAssignableFrom(serviceClass)) {
                return f("(&(|(%s=%s)(%s=%s))(%s=%s))",
                    Constants.OBJECTCLASS, serviceClass.getName(),
                    Constants.OBJECTCLASS, indirection.getName(),
                    "id", id
                );
            }
        }
        return f("(&(%s=%s)(%s=%s))",
            Constants.OBJECTCLASS, serviceClass.getName(),
            "id", id);
    }

    @Override
    public boolean matchesService(Object serviceInstance, ServiceReference<?> reference) {
        return this.serviceClass.isInstance(serviceInstance) && this.id.equals(reference.getProperty("id"));
    }

    @Override
    public boolean matchesDependency(Class<?> serviceClass, @Nullable Service serviceAnnotation) {
        return this.serviceClass.isAssignableFrom(serviceClass) &&
            serviceAnnotation != null && serviceAnnotation.id() != null
            && serviceAnnotation.id().equals(this.id);
    }
    
    @Override
    public String identify() {
        return f("%s[%s=%s]", serviceClass.getName(), "id", id);
    }

    @Override
    public Set<String> propertiesOfInterest() {
        return INTERESTS;
    }
    
    @Override
    public int hashCode() {
        return Objects.hash(this.getClass(), serviceClass, id);
    }
    
    @Override
    public boolean equals(@Nullable Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || ! obj.getClass().equals(this.getClass())) {
            return false;
        }
        final ServiceClassWithIdDependency<?> other = (ServiceClassWithIdDependency<?>) obj;
        return Objects.equals(serviceClass, other.serviceClass)
            && Objects.equals(id, other.id);
    }
    
    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
        .add("serviceClass", serviceClass)
        .add("id", id)
        .toString();
    }

}
