/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import java.util.ArrayList;
import java.util.Dictionary;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.openexchange.config.Reloadable;
import com.openexchange.osgi.DeferredActivator;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.osgi.console.ServiceStateLookup;
import com.openexchange.util.activator.RegistrationException;
import com.openexchange.util.activator.ServiceDependencyResolver;
import com.openexchange.util.custom.collect.ImmutableDictionary;

/**
 * This abstract service tracker can wait for some service and invoke {@link #register}
 * when all dependencies are available.
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
@ThreadSafe
public abstract class ServiceAwareDependentServiceRegistererTemplate<S, T extends S> implements ServiceTrackerCustomizer<S, T> {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(ServiceAwareDependentServiceRegistererTemplate.class);
    
    private static final class ResolvedService {
        public @Nonnull final Object serviceObject; 
        public @Nonnull final Dictionary<String, ?> properties;
        public ResolvedService(@Nonnull Object serviceObject, @Nonnull Dictionary<String, ?> properties) {
            this.serviceObject = serviceObject;
            this.properties = properties;
        }
        @Override
        public int hashCode() {
            return Objects.hash(serviceObject, properties);
        }
        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (! obj.getClass().equals(this.getClass())) {
                return false;
            }
            final ResolvedService other = (ResolvedService) obj;
            return Objects.equals(serviceObject, other.serviceObject)
                && Objects.equals(properties, other.properties);
        }
    }
    
    private final Lock lock = new ReentrantLock();
    protected @Nonnull final BundleContext context;
    protected @Nonnull final ImmutableSet<Class<?>> serviceTypes;
    protected @Nonnull final String serviceInstanceName;
    protected @Nonnull final Dictionary<String, ?> properties;
    protected @Nonnull final ImmutableMap<Class<?>, ServiceSet<?>> serviceSets;
    protected @Nonnull final ImmutableMap<Class<?>, ServiceListing<?>> serviceListings;
    protected @Nonnull final ImmutableSet<String> propertiesOfInterest;
    private final ImmutableMap<ServiceDependency<?>, AtomicReference<ResolvedService>> dependencyMap;
    private final AtomicBoolean registered = new AtomicBoolean(false);

    public ServiceAwareDependentServiceRegistererTemplate(@Nonnull BundleContext context,
        @Nonnull ImmutableSet<Class<?>> serviceTypes,
        @Nonnull String serviceInstanceName,
        @Nonnull Dictionary<String, ?> properties,
        @Nonnull Iterable<ServiceDependency<?>> neededServices,
        @Nonnull ImmutableMap<Class<?>, ServiceSet<?>> serviceSets,
        @Nonnull ImmutableMap<Class<?>, ServiceListing<?>> serviceListings,
        @Nonnull Set<String> propertiesOfInterest) {
        super();
        this.context = context;
        this.serviceTypes = serviceTypes;
        this.serviceInstanceName = serviceInstanceName;
        this.properties = properties;
        {
            final ImmutableMap.Builder<ServiceDependency<?>, AtomicReference<ResolvedService>> b = ImmutableMap.builder();
            for (final ServiceDependency<?> neededService : neededServices) {
                if (neededService != null) {
                    b.put(neededService, new AtomicReference<ResolvedService>(null));
                }
            }
            this.dependencyMap = b.build();
        }
        this.serviceSets = serviceSets;
        this.serviceListings = serviceListings;
        this.propertiesOfInterest = ImmutableSet.copyOf(propertiesOfInterest);
        setState();
    }
    
    protected final void setRegistered() {
        if (! this.registered.compareAndSet(false, true)) {
            throw new RegistrationException("service was already registered");
        }
    }

    private static @Nonnull ImmutableDictionary<String, ?> snapshotProperties(final ServiceReference<?> reference, final Set<String> keys) {
        final ImmutableMap.Builder<String, Object> b = ImmutableMap.builder();
        for (final String key : keys) {
            final Object value = reference.getProperty(key);
            if (key != null && value != null) {
                b.put(key, value);
            }
        }
        return ImmutableDictionary.copyOf(b.build());
    }
    
    @Override
    public final T addingService(final ServiceReference<S> reference) {
        if (reference == null) {
            throw new IllegalArgumentException("ServiceReference parameter is null");
        }
        final @Nonnull T obj;
        {
            final S s = context.getService(reference);
            if (s == null) {
                throw new IllegalStateException("reference resolution returned null");
            }
            @SuppressWarnings("unchecked") final T t = (T) s;
            obj = t;
        }
        final ResolvedService r = new ResolvedService(obj, snapshotProperties(reference, propertiesOfInterest));
        boolean needsRegistration = true;
        final List<ServiceDependency<?>> unresolvedServiceDependencies = new LinkedList<>(); 
        lock.lock();
        try {
            for (final Map.Entry<ServiceDependency<?>, AtomicReference<ResolvedService>> e : dependencyMap.entrySet()) {
                final ServiceDependency<?> dep = e.getKey();
                if (dep != null && dep.matchesService(obj, reference)) {
                    final ResolvedService oldService = e.getValue().getAndSet(r);
                    needsRegistration &= (oldService == null);
                } else {
                    if (e.getValue().get() == null) {
                        unresolvedServiceDependencies.add(e.getKey());
                    }
                }
            }
            needsRegistration &= (! registered.get());
        } finally {
            lock.unlock();
        }
        if (! unresolvedServiceDependencies.isEmpty()) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} has {} unresolved service dependencies: {}", serviceInstanceName,
                    unresolvedServiceDependencies.size(), Joiner.on(", ").join(unresolvedServiceDependencies));
            }
            needsRegistration = false;
        }
        
        if (needsRegistration) {
            LOG.trace("{} has its {} service(s) resolved, proceeding to registration",
                serviceInstanceName, dependencyMap.size());
            registerService(snapshotDependencies());
        }
        setState();
        return obj;
    }
    
    private final ServiceDependencyResolver snapshotDependencies() {
        final ImmutableMap.Builder<ServiceDependency<?>, Object> b = ImmutableMap.builder();
        lock.lock();
        try {
            for (final Map.Entry<ServiceDependency<?>, AtomicReference<ResolvedService>> e : dependencyMap.entrySet()) {
                final ServiceDependency<?> dep = e.getKey();
                if (dep != null) {
                    final ResolvedService value = e.getValue().get();
                    if (value != null) {
                        b.put(dep, value.serviceObject);
                    }
                }
            }
        } finally {
            lock.unlock();
        }
        return new ServiceDependencyResolverImpl(b.build(), serviceSets, serviceListings, properties);
    }
    
    protected abstract @Nullable Object register(final ServiceDependencyResolver resolver);
    protected final void registerService(final ServiceDependencyResolver resolver) {
        try {
            final Object serviceInstance = register(resolver);
            setRegistered();
            if (serviceInstance != null) {
                if (serviceInstance instanceof Reloadable) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("'{}' is {}", serviceInstanceName, Reloadable.class.getSimpleName());
                    }
                    RegistrationTools.registerIntoContext(context, ImmutableSet.of(Reloadable.class), serviceInstance, null);
                }
            }
        } finally {
        }
    }
    

    @Override
    public final void modifiedService(ServiceReference<S> reference, T service) {
        // Nothing to do.
    }

    @Override
    public final void removedService(ServiceReference<S> reference, T service) {
        if (reference == null || service == null) {
            return;
        }
        
        boolean someServiceMissing = false;
        lock.lock();
        try {
            for (final Map.Entry<ServiceDependency<?>, AtomicReference<ResolvedService>> e : dependencyMap.entrySet()) {
                if (e.getKey().matchesService(service, reference)) {
                    final ResolvedService oldService = e.getValue().getAndSet(null);
                    if (oldService != null) {
                        someServiceMissing = true;
                    }
                }
            }
        } finally {
            lock.unlock();
        }
        if (someServiceMissing) {
            unregister(service);
        }
        setState();
        context.ungetService(reference);
    }

    /**
     * You can overwrite this method to do something else to shut down your code if one of the needed services is gone. If you overwrote
     * {@link #register()} and you did not register an OSGi service, just ignore the unregister parameter. If you registered a servlet and
     * the HttpService is gone, it is not available anymore in neededServices but from the given service parameter.
     * @param unregister OSGi service registration that needs to be unregistered.
     * @param service OSGi service that is taken down.
     */
    protected abstract void unregister(Object service);

    private void setState() {
        final ServiceStateLookup lookup = DeferredActivator.getLookup();
        final List<String> missing = new ArrayList<>();
        final List<String> present = new ArrayList<>();
        for (final Map.Entry<ServiceDependency<?>, AtomicReference<ResolvedService>> e : dependencyMap.entrySet()) {
            final String id = e.getKey().identify();
            if (e.getValue().get() == null) {
                missing.add(id);
            } else {
                present.add(id);
            }
        }
        lookup.setState(serviceInstanceName, missing, present);
    }
}
