/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.activator.impl.RegistrationTools.TO_SIMPLE_CLASS_NAME;
import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.f;
import java.util.Dictionary;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import com.google.common.base.Joiner;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.openexchange.osgi.ExceptionUtils;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.util.activator.ServiceDependencyResolver;

/**
 * Template for registering service instances.
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
@ThreadSafe
public abstract class InstanceFactoryServiceAwareDependentServiceRegistererTemplate<S, T extends S> extends ServiceAwareDependentServiceRegistererTemplate<S, T> {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(InstanceFactoryServiceAwareDependentServiceRegistererTemplate.class);
    
    private final AtomicReference<ServiceRegistrationSet<T>> registrationRef;
    
    public InstanceFactoryServiceAwareDependentServiceRegistererTemplate(@Nonnull BundleContext context,
        @Nonnull ImmutableSet<Class<?>> serviceTypes,
        @Nonnull String serviceInstanceName,
        @Nonnull Dictionary<String, ?> properties,
        @Nonnull Iterable<ServiceDependency<?>> neededServices,
        @Nonnull ImmutableMap<Class<?>, ServiceSet<?>> serviceSets,
        @Nonnull ImmutableMap<Class<?>, ServiceListing<?>> serviceListings,
        @Nonnull Set<String> propertiesOfInterest) {
        super(context, serviceTypes, serviceInstanceName, properties, neededServices, serviceSets, serviceListings, propertiesOfInterest);
        this.registrationRef = new AtomicReference<ServiceRegistrationSet<T>>(null);
    }

    protected abstract @Nonnull T createInstance(final @Nonnull ServiceDependencyResolver resolver) throws Exception;

    @Override
    protected Object register(final ServiceDependencyResolver resolver) {
        if (resolver == null) {
            throw new IllegalArgumentException(f("%s parameter is null", ServiceDependencyResolver.class.getName()));
        }
        try {
            final T serviceInstance = createInstance(resolver);
            if (LOG.isTraceEnabled()) {
                LOG.trace("registering service {} ({})",
                    className(serviceInstance),
                    FluentIterable.from(serviceTypes).transform(TO_SIMPLE_CLASS_NAME).join(Joiner.on(", ")));
            }
            final ServiceRegistration<?> r = RegistrationTools
                .registerIntoContext(context, serviceTypes, serviceInstance, properties);
            registrationRef.set(new ServiceRegistrationSet<T>(serviceInstance, r));
            return serviceInstance;
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            LOG.error(f("failed to register %s (%s)",
                serviceInstanceName,
                FluentIterable.from(serviceTypes).transform(TO_SIMPLE_CLASS_NAME).join(Joiner.on(", "))),
                t);
            return null;
        }
    }
    
    @Override
    protected void unregister(final Object service) {
        final ServiceRegistrationSet<T> unregister = registrationRef.getAndSet(null);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Unregistering service {} ({})",
                serviceInstanceName,
                FluentIterable.from(serviceTypes).transform(TO_SIMPLE_CLASS_NAME).join(Joiner.on(", "))
            );
        }
        
        if (unregister != null) {
            unregister.registration.unregister();
            RegistrationTools.closeInstance(unregister.registeredService, serviceInstanceName, serviceTypes);
        }
    }

}
