/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.logger;
import static com.openexchange.util.custom.base.NullUtil.notNull;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import org.slf4j.Logger;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXActionServiceFactory;
import com.openexchange.documentation.annotations.Action;
import com.openexchange.exception.OXException;
import com.openexchange.tools.servlet.AjaxExceptionCodes;
import com.openexchange.util.activator.AjaxAction;
import com.openexchange.util.activator.DependencyProvider;
import com.openexchange.util.activator.RegistrationException;
import com.openexchange.util.activator.ServiceDependencies;
import com.openexchange.util.activator.ServiceDependencyResolver;

/**
 * {@link AJAXActionServiceFactory} implementation that uses the {@code @}{@link Action}
 * (deprecated) or {@code @}{@link AjaxAction} annotation to discover {@link AJAXActionService}
 * implementations.
 * <p>
 * Note that if a class is annotated with neither {@code @}{@link Action} nor
 * {@code @}{@link AJAXActionService}, it will attempt to infer its action from the beginning
 * of its class name (e.g. {@code GetFooAction} yields {@code "get"}), but that approach
 * is less deterministic and the preferred method is to use the {@code @}{@link AjaxAction}
 * annotation.
 * <p>
 * See {@link #CLASS_NAME_PREFIX_ACTION} for a list of supported class name prefixes.
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.2
 */
@SuppressWarnings("deprecation")
@ParametersAreNonnullByDefault
public final class DiscoveringActionFactory implements AJAXActionServiceFactory, DependencyProvider, Closeable {
    
    private static final Logger LOG = logger(DiscoveringActionFactory.class);
    
    public static final ImmutableSet<String> CLASS_NAME_PREFIX_ACTION = ImmutableSet.of(
        "Get", "Set", "List", "Delete", "All", "Copy", "New", "Move", "Edit", "Update", "Search"
    );
    
    private final String module;
    private final ImmutableMap<String, Class<? extends AJAXActionService>> actions;
    private final ServiceDependencies needs;
    private final AtomicReference<ImmutableMap<String, AJAXActionService>> instancesRef; 
    
    public DiscoveringActionFactory(final String module, ImmutableSet<Class<? extends AJAXActionService>> ajaxActionServiceClasses) {
        this.module = module;

        {
            final ImmutableMap.Builder<String, Class<? extends AJAXActionService>> actionsBuilder = ImmutableMap.builder();
            final List<ServiceDependencies> deps = new ArrayList<>(ajaxActionServiceClasses.size());
            for (final Class<? extends AJAXActionService> c : ajaxActionServiceClasses) {
                final ImmutableSet<String> actions;
                {
                    final Action action = c.getAnnotation(Action.class);
                    final AjaxAction ajaxAction = c.getAnnotation(AjaxAction.class);
                    if (ajaxAction != null) {
                        // AjaxAction wins, we just ignore the @Action annotation
                        actions = ImmutableSet.copyOf(notNull(ajaxAction.value()));
                    } else if (action != null) {
                        actions = ImmutableSet.of(notNull(action.name()));
                    } else {
                        String a = null;
                        final Iterator<String> iter = CLASS_NAME_PREFIX_ACTION.iterator();
                        while (iter.hasNext() && a == null) {
                            final String prefix = iter.next();
                            if (c.getSimpleName().startsWith(prefix)) {
                                a = prefix.toLowerCase(Locale.US);
                            }
                        }
                        if (a != null) {
                            actions = ImmutableSet.of(a);                            
                        } else {
                            final String msg = f(
                                "%s implementation class '%s' has no @%s annotation and its class name does not start with a well-known action prefix in [%s]",
                                AJAXActionService.class.getSimpleName(),
                                c.getName(),
                                AjaxAction.class.getSimpleName(),
                                Joiner.on(", ").join(CLASS_NAME_PREFIX_ACTION));
                            LOG.error(msg);
                            throw new RegistrationException(msg);
                        }
                    }
                }
                for (final String a : actions) {
                    if (a != null) {
                        actionsBuilder.put(a, c);
                    }
                }
                
                final ServiceWithDependencies<?> d = ConstructorServiceWithDependencies.find(c, AJAXActionService.class);
                deps.add(d);
            }
            
            {
                final Set<ServiceDependency<?>> serviceDependenciesBuilder = new HashSet<>();
                final Set<Class<?>> serviceSetsBuilder = new HashSet<>();
                final Set<Class<?>> serviceListingsBuilder = new HashSet<>();
                for (final ServiceDependencies d : deps) {
                    serviceDependenciesBuilder.addAll(d.serviceDependencies());
                    serviceSetsBuilder.addAll(d.serviceSetClasses());
                    serviceListingsBuilder.addAll(d.serviceListingClasses());
                }
                this.needs = new ImmutableServiceDependencies(
                    ImmutableSet.copyOf(serviceDependenciesBuilder),
                    ImmutableSet.copyOf(serviceSetsBuilder),
                    ImmutableSet.copyOf(serviceListingsBuilder)
                );                
            }
            this.actions = actionsBuilder.build();
        }
        this.instancesRef = new AtomicReference<>(ImmutableMap.<String,AJAXActionService>of());
    }

    @SuppressWarnings({ "null", "unused" })
    @Override
    public AJAXActionService createActionService(final @Nullable String action) throws OXException {
        final AJAXActionService service = instancesRef.get().get(action);
        if (service != null) {
            return service;
        } else {
            throw AjaxExceptionCodes.UNKNOWN_ACTION_IN_MODULE.create(action, module);
        }
    }

    @Override
    public Collection<?> getSupportedServices() {
        return actions.values();
    }

    @Override
    public ServiceDependencies dependsOn() {
        return needs;
    }
    
    @Override
    public void onDependenciesAvailable(ServiceDependencyResolver resolver) {
        final ImmutableMap.Builder<String, AJAXActionService> b = ImmutableMap.builder();
        for (final Map.Entry<String, Class<? extends AJAXActionService>> e : actions.entrySet()) {
            final String name = e.getKey();
            if (name != null) {
                final Class<? extends AJAXActionService> serviceClass = e.getValue();
                if (serviceClass != null) {
                    final ServiceWithDependencies<?> s = ConstructorServiceWithDependencies.find(serviceClass, AJAXActionService.class);
                    b.put(name, (AJAXActionService) s.createInstance(resolver));
                }
            }
        }
        close(this.instancesRef.getAndSet(b.build()));
    }

    @Override
    public void close() throws IOException {
        final ImmutableMap<String, AJAXActionService> oldValue = instancesRef.getAndSet(ImmutableMap.<String, AJAXActionService>of());
        close(oldValue);
    }
    
    private static void close(final @Nullable ImmutableMap<String, AJAXActionService> map) {
        if (map != null) {
            for (final AJAXActionService s : map.values()) {
                if (s != null) {
                    if (s instanceof Closeable) {
                        try {
                            ((Closeable) s).close();
                        } catch (final Exception e) {
                            LOG.error(f("failed to close %s (suppressed)", className(s), e));
                        }
                    }
                }
            }            
        }
    }

}
