package com.openexchange.util.activator;

import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.logger;
import static com.openexchange.util.custom.base.NullUtil.notNull;
import java.util.Dictionary;
import java.util.Hashtable;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.servlet.Filter;
import javax.servlet.http.HttpServlet;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import org.slf4j.Logger;
import com.openexchange.ajax.login.LoginRequestHandler;
import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.groupware.update.DefaultUpdateTaskProviderService;
import com.openexchange.groupware.update.UpdateTaskV2;
import com.openexchange.management.Managements;
import com.openexchange.osgi.ExceptionUtils;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.osgi.SimpleRegistryListener;
import com.openexchange.util.activator.impl.ActivatorRegistry;
import com.openexchange.util.activator.impl.ActivatorRegistryImpl;
import com.openexchange.util.activator.impl.RegistrationTools;

/**
 * OSGi activator template.
 * <p>
 * Use this class for your own {@link BundleActivator} implementations, as it provides
 * a lot of commodities to minimize the amount of code, as well as a lot of sugar for
 * common service registration patterns.
 * <p>
 * <h3>Example:</h3>
 * <pre>
public final class ProximusActivator extends ActivatorTemplate {
    {@code @}Override
    protected void registerServices(final Registrar registrar) {
        registrar
        
        // Register a service class by its class name.
        // There is no need to specify the service interface that is
        // implemented by the class as long as the class itself is
        // annotated with {@code @}Provides(SomeServiceClass).
        // The library will look up its dependencies using reflection
        // by looking at the method signature of the constructor and
        // collecting the ones that are annotated with {@code @}Service.
        // Once all those dependencies are available, it will invoke
        // the constructor, passing the dependencies as parameters
        // (see below for SomeServiceClassImpl):
        .service(SomeServiceClassImpl.class)
        
        // Register a service instance:
        // There is no need to specify the service interface that
        // is implemented by the object as long as its class itself
        // is annotated with {@code @}Provides.
        // Obviously, it cannot have dependencies as the object
        // already exists, unless on implements the
        // DependencyProvider interface: 
        .service(SomeJslobService.INSTANCE)
        
        // One may still explicitly specify which service interface
        // is implemented and provided by a class:
        .service(AmazingService.class, AmazingServiceImpl.class)
        
        // .. or an instance:
        .service(WonderfulService.class, new WonderfulServiceImpl())
        
        // Register an AJAX module under an API URI part:
        // The first parameter defines the API URI under which all the
        // AJAXActionService classes will be available under.
        // The {@code ?action=...} and is lookup using, in that order:
        // * the {@code @}{@link AjaxAction} annotation
        // * the (deprecated) {@code @}{@link com.openexchange.documentation.annotation.Action} annotation's {@code name} parameter
        // * guessing by the beginning of the class name ({@code GetFooAction} yields {@code "get"}
        .module("custom/amazing", GetAmazingAction.class, SetAmazingAction.class)
        
        // Register a HttpServlet:
        // It will be registered with the OSGi HttpService under the specified path:
        .servlet("/servlet/wonderful", WonderfulServlet.class)
        // Register another HttpServlet with a configuration property in the path:
        // It will be registered with the OSGi HttpService under the specified path,
        // after the path has been resolved using properties in ConfigurationService.
        // Note that it automatically registers a Reloadable that is interested in the
        // placeholders that are used and when those change, the servlet is automatically
        // un-registered and re-registered under the new path.
        .servlet("/servlet/api/${com.acme.api.path}", AcmeApiServlet.class)
        ;
    }
}

// The {@code @}Provides annotation tells the library under which
// service class (interface) it should be registered in OSGi:
{@code @}Provides(SomeServiceClass.class)
public final SomeServiceClassImpl implements SomeServiceClass {
    private final NeededService1 need1;
    private final NeededService2 need2;
    // The signature of the constructor will be used to determine
    // the dependencies:
    public SomeServiceClassImpl(final {@code @}Service NeededService1 need1,
        final {@code @}Service NeededService2 need2) {
        this.need1 = need1;
        this.need2 = need2;
    }
    // ...
}
</pre>
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.0
 */
@ParametersAreNonnullByDefault
@ThreadSafe
public abstract class ActivatorTemplate implements BundleActivator {

    protected final class Registrar {

        private final BundleContext bundleContext;
        
        private Registrar(final BundleContext bundleContext) {
            this.bundleContext = bundleContext;
        }
        
        // --- Basic service registration
        
        /**
         * Register a service instance or a service instance class.
         * <p>
         * If it is a {@link Class}, dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * If it is an object instance, no dependencies can be injected after the fact as the object
         * already exists, but one may implement the {@link DependencyProvider} interface to
         * {@linkplain DependencyProvider#dependsOn provide a list of dependencies} from an object,
         * and be {@linkplain DependencyProvider#onDependenciesAvailable notified when they are available}.
         * <p>
         * In both cases, the @{@link Provides} annotation will be used to determine which service class
         * it provides and under which it shall be registered.
         * 
         * @param serviceInstanceOrClass the service instance or service instance class, or an instance of
         *  {@link ServiceWithProperties}
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public <S, T extends S> Registrar service(final T serviceInstanceOrClass) {
            RegistrationTools.registerServiceInstanceOrClass(bundleContext, registry, null, serviceInstanceOrClass, emptyProperties(), null);
            return this;
        }
        
        /**
         * Register a service instance or a service instance class.
         * <p>
         * If it is a {@link Class}, dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * If it is an object instance, no dependencies can be injected after the fact as the object
         * already exists, but one may implement the {@link DependencyProvider} interface to
         * {@linkplain DependencyProvider#dependsOn provide a list of dependencies} from an object,
         * and be {@linkplain DependencyProvider#onDependenciesAvailable notified when they are available}.
         * <p>
         * In both cases, the @{@link Provides} annotation will be used to determine which service class
         * it provides and under which it shall be registered.
         * 
         * @param serviceInstanceOrClass the service instance or service instance class, or an instance of
         *  {@link ServiceWithProperties}
         * @param serviceRegistrationHandler will be invoked once all the dependencies have been resolved,
         *  the constructor has been called (if it is a service class and not a service instance) and the
         *  service has been registered
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public <S, T extends S> Registrar serviceWithHandler(final T serviceInstanceOrClass,
            final ServiceRegistrationHandler<T> serviceRegistrationHandler) {
            RegistrationTools.registerServiceInstanceOrClass(bundleContext, registry, null, serviceInstanceOrClass, emptyProperties(), serviceRegistrationHandler);
            return this;
        }
        
        /**
         * Register a service instance or a service instance class.
         * <p>
         * If it is a {@link Class}, dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * If it is an object instance, no dependencies can be injected after the fact as the object
         * already exists, but one may implement the {@link DependencyProvider} interface to
         * {@linkplain DependencyProvider#dependsOn provide a list of dependencies} from an object,
         * and be {@linkplain DependencyProvider#onDependenciesAvailable notified when they are available}.
         * <p>
         * In both cases, the @{@link Provides} annotation will be used to determine which service class
         * it provides and under which it shall be registered.
         * 
         * @param serviceInstanceOrClass the service instance or service instance class, or an instance of
         *  {@link ServiceWithProperties}
         * @param serviceRegistrationHandler will be invoked once all the dependencies have been resolved,
         *  the constructor has been called (if it is a service class and not a service instance) and the
         *  service has been registered
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public <S, T extends S> Registrar serviceWithHandler(final Class<T> serviceInstanceOrClass,
            final ServiceRegistrationHandler<T> serviceRegistrationHandler) {
            RegistrationTools.registerServiceClass(bundleContext, registry, null, serviceInstanceOrClass, emptyProperties(), serviceRegistrationHandler);
            return this;
        }
        
        /**
         * Register a service instance or a service instance class with properties.
         * <p>
         * If it is a {@link Class}, dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * If it is an object instance, no dependencies can be injected after the fact as the object
         * already exists, but one may implement the {@link DependencyProvider} interface to
         * {@linkplain DependencyProvider#dependsOn provide a list of dependencies} from an object,
         * and be {@linkplain DependencyProvider#onDependenciesAvailable notified when they are available}.
         * <p>
         * In both cases, the @{@link Provides} annotation will be used to determine which service class
         * it provides and under which it shall be registered.
         * 
         * @param serviceInstanceOrClass the service instance or service instance class, or an instance of
         *  {@link ServiceWithProperties}
         * @param properties properties to pass to the registration
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public <S, T extends S> Registrar service(final T serviceInstanceOrClass, final Dictionary<String, ?> properties) {
            RegistrationTools.registerServiceInstanceOrClass(bundleContext, registry, null, serviceInstanceOrClass, properties, null);
            return this;
        }
        
        /**
         * Register a service instance or a service instance class with properties.
         * <p>
         * If it is a {@link Class}, dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * If it is an object instance, no dependencies can be injected after the fact as the object
         * already exists, but one may implement the {@link DependencyProvider} interface to
         * {@linkplain DependencyProvider#dependsOn provide a list of dependencies} from an object,
         * and be {@linkplain DependencyProvider#onDependenciesAvailable notified when they are available}.
         * <p>
         * In both cases, the @{@link Provides} annotation will be used to determine which service class
         * it provides and under which it shall be registered.
         * 
         * @param serviceInstanceOrClass the service instance or service instance class, or an instance of
         *  {@link ServiceWithProperties}
         * @param properties properties to pass to the registration
         * @param serviceRegistrationHandler will be invoked once all the dependencies have been resolved,
         *  the constructor has been called (if it is a service class and not a service instance) and the
         *  service has been registered
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public <S, T extends S> Registrar serviceWithHandler(final T serviceInstanceOrClass, final Dictionary<String, ?> properties,
            final ServiceRegistrationHandler<T> serviceRegistrationHandler) {
            RegistrationTools.registerServiceInstanceOrClass(bundleContext, registry, null, serviceInstanceOrClass, properties, serviceRegistrationHandler);
            return this;
        }
        
        /**
         * Register a service instance or a service instance class with an explicit service class.
         * <p>
         * If it is a {@link Class}, dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * If it is an object instance, no dependencies can be injected after the fact as the object
         * already exists, but one may implement the {@link DependencyProvider} interface to
         * {@linkplain DependencyProvider#dependsOn provide a list of dependencies} from an object,
         * and be {@linkplain DependencyProvider#onDependenciesAvailable notified when they are available}.
         * <p>
         * Note that the {@link Provides} annotation will <em>NOT</em> be used to determine which service class
         * it provides and under which it shall be registered but, instead, the specified {@code serviceClass}
         * will be used.
         * 
         * @param serviceClass the service class
         * @param serviceInstanceOrClass the service instance or service instance class
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public <S, T extends S> Registrar service(final Class<S> serviceClass, final T serviceInstanceOrClass) {
            RegistrationTools.registerServiceInstanceOrClass(bundleContext, registry, serviceClass, serviceInstanceOrClass, emptyProperties(), null);
            return this;
        }

        /**
         * Register a service instance or a service instance class with an explicit service class.
         * <p>
         * If it is a {@link Class}, dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * If it is an object instance, no dependencies can be injected after the fact as the object
         * already exists, but one may implement the {@link DependencyProvider} interface to
         * {@linkplain DependencyProvider#dependsOn provide a list of dependencies} from an object,
         * and be {@linkplain DependencyProvider#onDependenciesAvailable notified when they are available}.
         * <p>
         * Note that the {@link Provides} annotation will <em>NOT</em> be used to determine which service class
         * it provides and under which it shall be registered but, instead, the specified {@code serviceClass}
         * will be used.
         * 
         * @param serviceClass the service class
         * @param serviceInstanceOrClass the service instance or service instance class
         * @param serviceRegistrationHandler will be invoked once all the dependencies have been resolved,
         *  the constructor has been called (if it is a service class and not a service instance) and the
         *  service has been registered
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public <S, T extends S> Registrar serviceWithHandler(final Class<S> serviceClass, final T serviceInstanceOrClass,
            final ServiceRegistrationHandler<T> serviceRegistrationHandler) {
            RegistrationTools.registerServiceInstanceOrClass(bundleContext, registry, serviceClass, serviceInstanceOrClass, emptyProperties(), serviceRegistrationHandler);
            return this;
        }
        
        /**
         * Register a service instance or a service instance class with an explicit service class
         * and properties.
         * <p>
         * If it is a {@link Class}, dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * If it is an object instance, no dependencies can be injected after the fact as the object
         * already exists, but one may implement the {@link DependencyProvider} interface to
         * {@linkplain DependencyProvider#dependsOn provide a list of dependencies} from an object,
         * and be {@linkplain DependencyProvider#onDependenciesAvailable notified when they are available}.
         * <p>
         * Note that the {@link Provides} annotation will <em>NOT</em> be used to determine which service class
         * it provides and under which it shall be registered but, instead, the specified {@code serviceClass}
         * will be used.
         * 
         * @param serviceClass the service class
         * @param serviceInstanceOrClass the service instance or service instance class
         * @param properties properties to pass to the registration
         * @return the {@link Registrar}
         * @throws RegistrationException
         */        
        public <S, T extends S> Registrar service(final Class<S> serviceClass, final T serviceInstanceOrClass, final Dictionary<String, ?> properties) {
            RegistrationTools.registerServiceInstanceOrClass(bundleContext, registry, serviceClass, serviceInstanceOrClass, properties, null);
            return this;
        }

        /**
         * Register a service instance or a service instance class with an explicit service class
         * and properties.
         * <p>
         * If it is a {@link Class}, dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * If it is an object instance, no dependencies can be injected after the fact as the object
         * already exists, but one may implement the {@link DependencyProvider} interface to
         * {@linkplain DependencyProvider#dependsOn provide a list of dependencies} from an object,
         * and be {@linkplain DependencyProvider#onDependenciesAvailable notified when they are available}.
         * <p>
         * Note that the {@link Provides} annotation will <em>NOT</em> be used to determine which service class
         * it provides and under which it shall be registered but, instead, the specified {@code serviceClass}
         * will be used.
         * 
         * @param serviceClass the service class
         * @param serviceInstanceOrClass the service instance or service instance class
         * @param properties properties to pass to the registration
         * @param serviceRegistrationHandler will be invoked once all the dependencies have been resolved,
         *  the constructor has been called (if it is a service class and not a service instance) and the
         *  service has been registered
         * @return the {@link Registrar}
         * @throws RegistrationException
         */        
        public <S, T extends S> Registrar serviceWithHandler(final Class<S> serviceClass, final T serviceInstanceOrClass,
            final Dictionary<String, ?> properties, final ServiceRegistrationHandler<T> serviceRegistrationHandler) {
            RegistrationTools.registerServiceInstanceOrClass(bundleContext, registry, serviceClass, serviceInstanceOrClass, properties, serviceRegistrationHandler);
            return this;
        }
        
        // --- Luxury

        /**
         * Register a {@link LoginRequestHandler}.
         * @param serviceInstanceOrClass the {@link LoginRequestHandler} instance or a class that extends {@link LoginRequestHandler}
         * @param action the login action for which this {@link LoginRequestHandler} should be registered or, if {@code null}, for all 
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public <T extends LoginRequestHandler> Registrar loginHandler(final T serviceInstanceOrClass, final String action) {
            RegistrationTools.registerServiceInstanceOrClass(bundleContext, registry, LoginRequestHandler.class, serviceInstanceOrClass, properties("action", action), null);
            return this;
        }
        
        /**
         * Register a servlet {@link Filter} for one or more paths.
         * @param filter the {@link Filter} instance
         * @param ranking the ranking order for this {@link Filter}, the higher the value, the higher the priority
         * @param path the path expression the {@link Filter} will apply to, see the description of {@link com.openexchange.servlet.Constants.FILTER_PATHS}
         * @param morePaths optionally, additional path expressions
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <F extends Filter> Registrar filter(final F filter, final int ranking, final String path, final String... morePaths) {
            RegistrationTools.registerServiceInstance(bundleContext, registry, Filter.class, filter, properties(
                notNull(org.osgi.framework.Constants.SERVICE_RANKING), ranking,
                notNull(com.openexchange.servlet.Constants.FILTER_PATHS), RegistrationTools.concatArray(path, morePaths)
            ), null);
            return this;
        }
        
        /**
         * Register a servlet {@link Filter} class for one or more paths.
         * @param filterClass the {@link Filter} instance class, optionally with dependencies
         * @param ranking the ranking order for this {@link Filter}, the higher the value, the higher the priority
         * @param path the path expression the {@link Filter} will apply to, see the description of {@link com.openexchange.servlet.Constants.FILTER_PATHS}
         * @param morePaths optionally, additional path expressions
         * @return the {@link Registrar}
         * @throws RegistrationException
         */        
        public final <F extends Filter> Registrar filter(final Class<F> filterClass, final int ranking, final String path, final String... morePaths) {
            RegistrationTools.registerServiceClass(bundleContext, registry, Filter.class, filterClass, properties(
                notNull(org.osgi.framework.Constants.SERVICE_RANKING), ranking,
                notNull(com.openexchange.servlet.Constants.FILTER_PATHS), RegistrationTools.concatArray(path, morePaths)
            ), null);
            return this;
        }
        
        /**
         * Register a servlet {@link Filter} around login API calls.
         * @param filter the {@link Filter} instance
         * @param ranking the ranking order for this {@link Filter}, the higher the value, the higher the priority
         * @return the {@link Registrar}
         * @throws RegistrationException
         */        
        public final <F extends Filter> Registrar loginFilter(final F filter, final int ranking) {
            RegistrationTools.registerLoginFilter(bundleContext, registry, filter, ranking);
            return this;
        }
        
        /**
         * Register a servlet {@link Filter} around login API calls.
         * @param filterClass the {@link Filter} instance class, optionally with dependencies
         * @param ranking the ranking order for this {@link Filter}, the higher the value, the higher the priority
         * @return the {@link Registrar}
         * @throws RegistrationException
         */        
        public final <F extends Filter> Registrar loginFilter(final Class<F> filterClass, final int ranking) {
            RegistrationTools.registerLoginFilter(bundleContext, registry, filterClass, ranking);
            return this;
        }
        
        /**
         * Register {@link AJAXActionService}s under a module URL.
         * <p>
         * Note that the {@code module} parameter denotes an URL part which
         * will be prefixed by the AppSuite AJAX API base URL.
         * <p>
         * Note that each {@link AJAXActionService} specified here must be annotated
         * with @Action, where the HTTP verb will be looked up in the {@code method}
         * parameter, and the {@code ?action=...} value in the {@code name} parameter.
         * @param module the URI path part under which the actions will be exposed
         * @param actions the {@link AJAXActionService} classes to expose
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        @SafeVarargs
        public final Registrar module(final String module,
            final Class<? extends AJAXActionService>... actions) {
            RegistrationTools.registerModule(bundleContext, registry, module, actions);
            return this;
        }
                
        /**
         * Track a set of services that implement the specified {@code serviceClass}.
         * <p>
         * Whenever an implementation of the {@code serviceClass} is registered into
         * the OSGi bundle context, it will be added and made available in the
         * {@link ServiceSet} that results from this method.
         * 
         * @param serviceClass the service class of which implementations shall
         *      be tracked
         * @return the {@link ServiceSet} that contains the dynamically registered
         *      implementations of the specified {@code serviceClass}
         */
        public final <S> ServiceSet<S> serviceSet(final Class<S> serviceClass) {
            return RegistrationTools.serviceSet(bundleContext, registry, serviceClass);
        }

        /**
         * Track a ranked set of services that implement the specified {@code serviceClass}.
         * <p>
         * Whenever an implementation of the {@code serviceClass} is registered into
         * the OSGi bundle context, it will be added and made available in the
         * {@link ServiceListing} that results from this method.
         * 
         * @param serviceClass the service class of which implementations shall
         *      be tracked
         * @return the {@link ServiceListing} that contains the dynamically registered
         *      implementations of the specified {@code serviceClass}
         */
        public final <S> ServiceListing<S> serviceListing(final Class<S> serviceClass) {
            return RegistrationTools.serviceListing(bundleContext, registry, serviceClass);
        }
        
        /**
         * Low-level tracking method.
         * <p>
         * Consider using a more high-level and convenient method instead.
         * @param serviceClass
         * @param serviceSet
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <S> Registrar track(final Class<S> serviceClass, final SimpleRegistryListener<S> serviceSet) {
            RegistrationTools.track(bundleContext, registry, serviceClass, serviceSet);
            return this;
        }

        /**
         * Low-level tracking method.
         * <p>
         * Consider using a more high-level and convenient method instead.
         * @param serviceClass
         * @param customizer
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <S> Registrar track(final Class<S> serviceClass, final ServiceTrackerCustomizer<S, S> customizer) {
            RegistrationTools.track(bundleContext, registry, serviceClass, customizer);
            return this;
        }

        /**
         * Register a JMX MBean.
         * @param domain
         * @param mbeanInterface
         * @param mbean
         * @return the {@link Registrar}
         * @throws RegistrationException
         * @throws MalformedObjectNameException
         */
        public final <S, T extends S> Registrar mbean(final String domain, final Class<S> mbeanInterface, final T mbean) throws MalformedObjectNameException {
            return mbean(notNull(Managements.getObjectName(mbeanInterface.getName(), domain)), mbeanInterface, mbean);
        }
        
        /**
         * Register a JMX MBean.
         * @param objectName
         * @param mbeanInterface
         * @param mbean
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <S, T extends S> Registrar mbean(final ObjectName objectName, final Class<S> mbeanInterface, final T mbean) {
            RegistrationTools.registerMBean(bundleContext, registry, objectName, mbeanInterface, mbean);
            return this;
        }

        /**
         * Register a {@link HttpServlet} class.
         * <p>
         * Dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * Note that the {@link Provides} annotation will <em>NOT</em> be used to determine which service class
         * it provides and under which it shall be registered as it the service interface is well known
         * to be {@link HttpServlet}.
         * <p>
         * The {@code path} may contain placeholders using the format <code>${placeholder}</code>.
         * When so, then the {@code ConfigurationService} will be used to replace the placeholders
         * with values defined in configuration properties.
         * When a placeholder cannot be resolved as a {@code ConfigurationService} property, then the
         * servlet will not be registered. If the missing placeholder is added to a configuration properties
         * file and a {@code reloadconfiguration} is triggered, then the servlet will be registered.
         * <p>
         * When there are placeholders in the {@code path}, it automatically registers a {@code Reloadable}
         * that is interested in those placeholder values and, if their value changes, the servlet will
         * automatically be un-registered and re-registered under the new path value.
         * 
         * @param path the path under which the {@link HttpServlet} will be made available, should
         *      normally start with {@code /servlet/} and may contain <code>${...}</code> placeholders
         * @param servletClass the {@link HttpServlet} class
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <T extends HttpServlet> Registrar servlet(final String path, final Class<T> servletClass) {
            RegistrationTools.registerServletClass(bundleContext, registry, servletClass, emptyProperties(), path, null);
            return this;
        }
        
        /**
         * Register a {@link HttpServlet} class.
         * <p>
         * Dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * Note that the {@link Provides} annotation will <em>NOT</em> be used to determine which service class
         * it provides and under which it shall be registered as it the service interface is well known
         * to be {@link HttpServlet}.
         * <p>
         * The {@code path} may contain placeholders using the format <code>${placeholder}</code>.
         * When so, then the {@code ConfigurationService} will be used to replace the placeholders
         * with values defined in configuration properties.
         * When a placeholder cannot be resolved as a {@code ConfigurationService} property, then the
         * servlet will not be registered. If the missing placeholder is added to a configuration properties
         * file and a {@code reloadconfiguration} is triggered, then the servlet will be registered.
         * <p>
         * When there are placeholders in the {@code path}, it automatically registers a {@code Reloadable}
         * that is interested in those placeholder values and, if their value changes, the servlet will
         * automatically be un-registered and re-registered under the new path value.
         * 
         * @param path the path under which the {@link HttpServlet} will be made available, should
         *      normally start with {@code /servlet/} and may contain <code>${...}</code> placeholders
         * @param servletClass the {@link HttpServlet} class
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <T extends HttpServlet> Registrar servlet(final String path, final Class<T> servletClass,
            final ServiceRegistrationHandler<T> serviceRegistrationHandler) {
            RegistrationTools.registerServletClass(bundleContext, registry, servletClass, emptyProperties(), path, serviceRegistrationHandler);
            return this;
        }        

        /**
         * Register a {@link HttpServlet} class with properties
         * <p>
         * Dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * Note that the {@link Provides} annotation will <em>NOT</em> be used to determine which service class
         * it provides and under which it shall be registered as it the service interface is well known
         * to be {@link HttpServlet}.
         * <p>
         * The {@code path} may contain placeholders using the format <code>${placeholder}</code>.
         * When so, then the {@code ConfigurationService} will be used to replace the placeholders
         * with values defined in configuration properties.
         * When a placeholder cannot be resolved as a {@code ConfigurationService} property, then the
         * servlet will not be registered. If the missing placeholder is added to a configuration properties
         * file and a {@code reloadconfiguration} is triggered, then the servlet will be registered.
         * <p>
         * When there are placeholders in the {@code path}, it automatically registers a {@code Reloadable}
         * that is interested in those placeholder values and, if their value changes, the servlet will
         * automatically be un-registered and re-registered under the new path value.
         * 
         * @param path the path under which the {@link HttpServlet} will be made available, should
         *      normally start with {@code /servlet/} and may contain <code>${...}</code> placeholders
         * @param servletClass the {@link HttpServlet} class
         * @param properties will be used for OSGi registration and also passed to {@link HttpServlet#init}
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <T extends HttpServlet> Registrar servlet(final String path, final Class<T> servletClass, final Dictionary<String, ?> properties) {
            RegistrationTools.registerServletClass(bundleContext, registry, servletClass, properties, path, null);
            return this;
        }
        
        /**
         * Register a {@link HttpServlet} class with properties
         * <p>
         * Dependencies will be analyzed using the @{@link Service} annotations
         * on constructor parameters, as well as to determine the constructor if ambiguous (if the class
         * has more than once constructor, mark one and only one of them with the @{@link Service}
         * annotation).
         * <p>
         * An instance will be constructed with all dependencies injected into the constructor of the class,
         * as specified using the @{@link Service} annotations on the constructor parameters.
         * <p>
         * Note that the {@link Provides} annotation will <em>NOT</em> be used to determine which service class
         * it provides and under which it shall be registered as it the service interface is well known
         * to be {@link HttpServlet}.
         * <p>
         * The {@code path} may contain placeholders using the format <code>${placeholder}</code>.
         * When so, then the {@code ConfigurationService} will be used to replace the placeholders
         * with values defined in configuration properties.
         * When a placeholder cannot be resolved as a {@code ConfigurationService} property, then the
         * servlet will not be registered. If the missing placeholder is added to a configuration properties
         * file and a {@code reloadconfiguration} is triggered, then the servlet will be registered.
         * <p>
         * When there are placeholders in the {@code path}, it automatically registers a {@code Reloadable}
         * that is interested in those placeholder values and, if their value changes, the servlet will
         * automatically be un-registered and re-registered under the new path value.
         * 
         * @param path the path under which the {@link HttpServlet} will be made available, should
         *      normally start with {@code /servlet/} and may contain <code>${...}</code> placeholders
         * @param servletClass the {@link HttpServlet} class
         * @param properties will be used for OSGi registration and also passed to {@link HttpServlet#init}
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <T extends HttpServlet> Registrar servlet(final String path, final Class<T> servletClass,
            final Dictionary<String, ?> properties, final ServiceRegistrationHandler<T> serviceRegistrationHandler) {
            RegistrationTools.registerServletClass(bundleContext, registry, servletClass, properties, path, serviceRegistrationHandler);
            return this;
        }        
        
        /**
         * Register a {@link HttpServlet} instance.
         * <p>
         * No dependencies can be injected after the fact as the object already exists.
         * <p>
         * Note that the {@link Provides} annotation will <em>NOT</em> be used to determine which service class
         * it provides and under which it shall be registered as it the service interface is well known
         * to be {@link HttpServlet}.
         * <p>
         * The {@code path} may contain placeholders using the format <code>${placeholder}</code>.
         * When so, then the {@code ConfigurationService} will be used to replace the placeholders
         * with values defined in configuration properties.
         * When a placeholder cannot be resolved as a {@code ConfigurationService} property, then the
         * servlet will not be registered. If the missing placeholder is added to a configuration properties
         * file and a {@code reloadconfiguration} is triggered, then the servlet will be registered.
         * <p>
         * When there are placeholders in the {@code path}, it automatically registers a {@code Reloadable}
         * that is interested in those placeholder values and, if their value changes, the servlet will
         * automatically be un-registered and re-registered under the new path value.
         * 
         * @param path the path under which the {@link HttpServlet} will be made available, should
         *      normally start with {@code /servlet/} and may contain <code>${...}</code> placeholders
         * @param serviceInstanceOrClass the {@link HttpServlet} instance
         * @return the {@link Registrar}
         * @throws RegistrationException
         */        
        public final <T extends HttpServlet> Registrar servlet(final String path, final T servletInstance) {
            RegistrationTools.registerServletInstance(bundleContext, registry, servletInstance, emptyProperties(), path);
            return this;
        }

        /**
         * Register a {@link HttpServlet} instance with properties.
         * <p>
         * No dependencies can be injected after the fact as the object already exists.
         * <p>
         * Note that the {@link Provides} annotation will <em>NOT</em> be used to determine which service class
         * it provides and under which it shall be registered as it the service interface is well known
         * to be {@link HttpServlet}.
         * <p>
         * The {@code path} may contain placeholders using the format <code>${placeholder}</code>.
         * When so, then the {@code ConfigurationService} will be used to replace the placeholders
         * with values defined in configuration properties.
         * When a placeholder cannot be resolved as a {@code ConfigurationService} property, then the
         * servlet will not be registered. If the missing placeholder is added to a configuration properties
         * file and a {@code reloadconfiguration} is triggered, then the servlet will be registered.
         * <p>
         * When there are placeholders in the {@code path}, it automatically registers a {@code Reloadable}
         * that is interested in those placeholder values and, if their value changes, the servlet will
         * automatically be un-registered and re-registered under the new path value.
         * 
         * @param path the path under which the {@link HttpServlet} will be made available, should
         *      normally start with {@code /servlet/} and may contain <code>${...}</code> placeholders
         * @param serviceInstanceOrClass the {@link HttpServlet} instance
         * @param properties will be used for OSGi registration and also passed to {@link HttpServlet#init}
         * @return the {@link Registrar}
         * @throws RegistrationException
         */        
        public final <T extends HttpServlet> Registrar servlet(final String path, final T servletInstance, final Dictionary<String, ?> properties) {
            RegistrationTools.registerServletInstance(bundleContext, registry, servletInstance, properties, path);
            return this;
        }

        /**
         * Register an {@link UpdateTaskV2}.
         * @param updateTaskClass the update task class with dependencies in its constructor
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <T extends UpdateTaskV2> Registrar updateTask(Class<T> updateTaskClass) {
            RegistrationTools.registerUpdateTask(bundleContext, registry, updateTaskClass, emptyProperties(), null);
            return this;
        }
        
        /**
         * Register an {@link UpdateTaskV2}.
         * @param updateTaskClass the update task class with dependencies in its constructor
         * @param properties properties to pass to the registration
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <T extends UpdateTaskV2> Registrar updateTask(Class<T> updateTaskClass, final Dictionary<String, ?> properties) {
            RegistrationTools.registerUpdateTask(bundleContext, registry, updateTaskClass, properties, null);
            return this;
        }
        
        /**
         * Register an {@link UpdateTaskV2}.
         * @param updateTaskClass the update task class with dependencies in its constructor
         * @param serviceRegistrationHandler will be invoked once all the dependencies have been resolved,
         *  the constructor has been called (if it is a service class and not a service instance) and the
         *  service has been registered
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <T extends UpdateTaskV2> Registrar updateTask(Class<T> updateTaskClass, final ServiceRegistrationHandler<DefaultUpdateTaskProviderService> serviceRegistrationHandler) {
            RegistrationTools.registerUpdateTask(bundleContext, registry, updateTaskClass, emptyProperties(), serviceRegistrationHandler);
            return this;
        }
        
        /**
         * Register an {@link UpdateTaskV2}.
         * @param updateTaskClass the update task class with dependencies in its constructor
         * @param properties properties to pass to the registration
         * @param serviceRegistrationHandler will be invoked once all the dependencies have been resolved,
         *  the constructor has been called (if it is a service class and not a service instance) and the
         *  service has been registered
         * @return the {@link Registrar}
         * @throws RegistrationException
         */
        public final <T extends UpdateTaskV2> Registrar updateTask(Class<T> updateTaskClass, final Dictionary<String, ?> properties, final ServiceRegistrationHandler<DefaultUpdateTaskProviderService> serviceRegistrationHandler) {
            RegistrationTools.registerUpdateTask(bundleContext, registry, updateTaskClass, properties, serviceRegistrationHandler);
            return this;
        }
        
    }
    
    protected abstract void registerServices(final Registrar registrar);
    protected void onBeforeRegistration(final BundleContext context) {
    }
    protected void onAfterRegistration(final BundleContext context) {
    }
    protected void onBeforeStop(final BundleContext context) {
    }
    protected void onAfterStop(final BundleContext context) {
    }
    
    private final Logger LOG;
    private final ActivatorRegistry registry;
    
    protected ActivatorTemplate() {
        this.LOG = logger(this.getClass());
        this.registry = new ActivatorRegistryImpl();
    }
    
    @Override
    public final void start(final @Nullable BundleContext context) throws Exception {
        if (context == null) {
            return;
        }
        LOG.trace("starting registration for {}", context.getBundle().getSymbolicName());
        onBeforeRegistration(context);
        try {
            final Registrar registrar = new Registrar(context);
            registerServices(registrar);
            postProcess(registrar);
        } catch (final Throwable e) {
            ExceptionUtils.handleThrowable(e);
            LOG.error(f("%s: failed to register services: %s: %s", context.getBundle().getSymbolicName(), className(e), e.getMessage()), e);
            throw e;
        } finally {
            try {
                onAfterRegistration(context);
            } finally {
                LOG.trace("ended registration for {}", context.getBundle().getSymbolicName());
            }
        }
    }
    
    @Override
    public final void stop(final @Nullable BundleContext context) {
        if (context != null) {
            onBeforeStop(context);
        }
        try {
            ((ActivatorRegistryImpl) registry).close();
        } finally {
            if (context != null) {
                onAfterStop(context);
            }
        }
    }
    
    protected void postProcess(final Registrar registrar) {
    }
    
    protected static final Dictionary<String, ?> emptyProperties() {
        return new Hashtable<>(0);
    }
    
    protected static final Dictionary<String, ?> id(final String id) {
        return properties("id", id);
    }
    
    protected static final Dictionary<String, ?> properties(final String key, final Object value) {
        final Hashtable<String, Object> properties = new Hashtable<>(1);
        properties.put(key, value);
        return properties;
    }

    protected static final Dictionary<String, ?> properties(
        final String key1, final Object value1,
        final String key2, final Object value2
    ) {
        final Hashtable<String, Object> properties = new Hashtable<>(2);
        properties.put(key1, value1);
        properties.put(key2, value2);
        return properties;
    }

    protected static final Dictionary<String, ?> properties(
        final String key1, final Object value1,
        final String key2, final Object value2,
        final String key3, final Object value3
    ) {
        final Hashtable<String, Object> properties = new Hashtable<>(3);
        properties.put(key1, value1);
        properties.put(key2, value2);
        properties.put(key3, value3);
        return properties;
    }

    protected static final Dictionary<String, ?> properties(
        final String key1, final Object value1,
        final String key2, final Object value2,
        final String key3, final Object value3,
        final String key4, final Object value4
    ) {
        final Hashtable<String, Object> properties = new Hashtable<>(4);
        properties.put(key1, value1);
        properties.put(key2, value2);
        properties.put(key3, value3);
        properties.put(key4, value4);
        return properties;
    }

}
