/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.impl;

/**
 * {@link CustomPropertyLimit}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class CustomPropertyLimit implements Comparable<CustomPropertyLimit> {

    private final long _storageLimit;

    private final String _key;

    /**
     * Initializes a new {@link CustomPropertyLimit} that will store the timeout for a given key
     * 
     * @param key CustomProperty key
     */
    public CustomPropertyLimit(String key, long timeout) {
        _key = key;
        if (timeout <= 0L)
            throw new IllegalArgumentException("Timeout must be > 0");
        long now = System.currentTimeMillis();
        _storageLimit = now + timeout;
        if (_storageLimit < now || _storageLimit < timeout)
            throw new IllegalArgumentException("Timeout too high");
    }

    /**
     * Initializes a new {@link CustomPropertyLimit} that will match any CustomPropertyLimit with the same key (useful for searching without
     * knowing the timeout)
     * 
     * @param key CustomProperty key
     */
    public CustomPropertyLimit(String key) {
        _key = key;
        _storageLimit = 0L;
    }

    /**
     * @return CustomProperty key referenced by this {@link CustomPropertyLimit}
     */
    public String getKey() {
        return _key;
    }

    /**
     * @return timestamp after which this {@link CustomPropertyLimit} should be removed from the USM-Session.
     */
    public long getStorageLimit() {
        return _storageLimit;
    }

    @Override
    public int compareTo(CustomPropertyLimit o) {
        if (_storageLimit < o._storageLimit)
            return -1;
        if (_storageLimit > o._storageLimit)
            return 1;
        return 0;
    }

    @Override
    public int hashCode() {
        // Only key is used for hashCode() so that object is stored/retrieved correctly with specific / unspecified timeout
        return _key.hashCode();
    }

    /**
     * Returns true if obj is a CustomPropertyLimit the keys match, and the storageLimit either matches or one of the two storageLimits is 0
     * (which matches any limit).
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        CustomPropertyLimit other = (CustomPropertyLimit) obj;
        if (_key == null) {
            if (other._key != null)
                return false;
        } else if (!_key.equals(other._key))
            return false;
        // Special handling of storageLimit: Allow 0 to be a wild card that matches all limits
        if (_storageLimit == 0L || other._storageLimit == 0L)
            return true;
        return _storageLimit == other._storageLimit;
    }

    @Override
    public String toString() {
        if (_storageLimit == 0L)
            return "Limit(" + _key + ')';
        return "Limit(" + _key + ':' + _storageLimit + ')';
    }
}
