/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.api.cache;

import java.io.Serializable;

/**
 * The Cache class extends the ICacheAccess interface, adding a
 * methods to store in and read  data objects from cache.
 *
 * @author afe
 *
 * @param <K> key type
 * @param <V> value type
 */
public interface Cache<K extends Serializable, V extends Serializable> {
	/**
	 * Basic get method.
	 * <p>
	 * @param name
	 * @return Object or null if not found.
	 */
	V get(K name);

	/**
	 * Puts in cache if an item does not exist with the name in that region.
	 * <p>
	 * @param name - any serializable object.
	 *               If removing key/value pairs in a group is needed than the name should be String in a hierarchical form X:Y:Z
	 * @param obj  - can be any serializable object
	 */
	void putSafe(K name, V obj);

	/**
	 * Puts and/or overrides an element with the name in that region.
	 * <p>
	 * @param name - any serializable object.
	 *               If removing key/value pairs in a group is needed than the name should be String in a hierarchical form X:Y:Z
	 * @param obj - can be any serializable object
	 * @param invalidate - <code>true</code> to trigger remote invalidation processing for the cache entry, <code>false</code>, otherwise.
	 */
	void put(K name, V obj, boolean invalidate);

	/**
	 * Remove an object for this key if one exists, else do nothing.
	 * To remove more key/value pairs at once use hierarchical String keys in a form X:Y:Z
	 * ex. remove ("X:Y:") will remove all keys which start with "X:Y:"
	 * <p>
	 * @param name
	 */
	void remove(K name);

	/**
	 * Can be called to free resources if the cache will not be used any more.
	 */
	void dispose();
}
