/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.datatypes.contacts;

import java.io.Serializable;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.datatypes.AbstractDataType;
import com.openexchange.usm.api.session.Session;

public class ImageDataType extends AbstractDataType<Image> {

    public static final String IMAGE1 = "image1";
    
    private static final String IMAGE_LAST_MODIFIED = "image_last_modified";

    private static final String IMAGE1_URL = "image1_url";

    private static final String IMAGE1_CONTENT_TYPE = "image1_content_type";
    
    public ImageDataType() {
        super(IMAGE1);
    }

    @Override
    public Image checkValue(Object o) {
        if (o != null && !(o instanceof Image))
            generateTypeError(o);
        return (Image) o;
    }

    @Override
    public Image extractFromJSONArray(Session session, JSONArray array, int index) throws JSONException {
        // Special case, image consists of multiple columns: image_last_modified, image1_url and image1_content_type
        // String imageData = array.getString(index); // "image1"
        long timestamp = array.isNull(index) ? 0L : array.getLong(index); // "image_last_modified"
        if (timestamp == 0L)
            return null;
        String url = array.isNull(index + 1) ? null : array.getString(index + 1); // "image1_url"
        String contentType = array.isNull(index + 2) ? null : array.getString(index + 2); // "image1_content_type"
        return new Image(timestamp, url, contentType);
    }

    @Override
    public Image extractFromJSONObject(Session session, JSONObject jsonObject, String field) throws JSONException {
        // String imageData = jsonObject.getString(IMAGE1); // "image1"
        long timestamp = jsonObject.isNull(IMAGE_LAST_MODIFIED) ? 0L : jsonObject.getLong(IMAGE_LAST_MODIFIED);
        if (timestamp == 0L)
            return null;
        String url = jsonObject.isNull(IMAGE1_URL) ? null : jsonObject.getString(IMAGE1_URL);
        String contentType = jsonObject.isNull(IMAGE1_CONTENT_TYPE) ? null : jsonObject.getString(IMAGE1_CONTENT_TYPE);
        return new Image(timestamp, url, contentType);
    }

    @Override
    public void storeInJSONObject(Session session, JSONObject parent, String field, Object value) throws JSONException {
        Image image = (Image) value;
        if (image == null || image.getContentType() == null || image.getData() == null)
            parent.put(IMAGE1, JSONObject.NULL);
        // All other cases will be handled in special case in ContactContentTypeTransferHandler
    }

    @Override
    public void addToJSONArray(Session session, JSONArray parent, Object value) {
        // We store the lastModified timestamp in the JSONArray. This is mainly for toString() of SimpleDataObject, since the Contact Image
        // is not stored in an array.
        Image image = (Image) value;
        parent.put((image == null) ? 0L : image.getLastModified());
    }

    @Override
    public Serializable serialize(Object value) {
        return (value instanceof Image) ? ((Image) value).getLastModified() : null;
    }

    @Override
    public Image deserialize(Serializable value) {
        if (value instanceof Number)
            return new Image(((Number) value).longValue(), (String) null, (String) null);
        return (value instanceof Image) ? ((Image) value) : null;
    }
}
