/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.impl;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.lang.ref.WeakReference;

/**
 * Stores a SoftReference to a SessionImpl object, and also tracks the last access and allows access to the time of the last access to the
 * SessionImpl. {@link SessionReference}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class SessionReference {

    /**
     * SoftReference to keep SessionImpl in memory until the memory is needed for other objects
     */
    private final SoftReference<SessionImpl> _softReference;

    /**
     * WeakReference to allow access to SessionImpl without updating the last access timestamp in the SoftReference
     */
    private final WeakReference<SessionImpl> _weakReference;

    /**
     * Own timestamp of last access since the timestamp in SoftReference is not accessible
     */
    private long _lastAccess;

    public SessionReference(SessionImpl session, ReferenceQueue<? super SessionImpl> queue) {
        _softReference = new SoftReference<SessionImpl>(session, queue);
        _weakReference = new WeakReference<SessionImpl>(session);
        _lastAccess = System.currentTimeMillis();
    }

    /**
     * @return SoftReference that is referencing the SessionImpl
     */
    public SoftReference<SessionImpl> getReference() {
        return _softReference;
    }

    /**
     * @return Cached SessionImpl or null if it has been removed by JVM, updates the last access timestamp
     */
    public SessionImpl get() {
        _lastAccess = System.currentTimeMillis();
        return _softReference.get();
    }

    /**
     * @return Cached SessionImpl or null if it has been removed by JVM, does not update the last access timestamp
     */
    public SessionImpl poll() {
        return _weakReference.get();
    }

    /**
     * @return Timestamp of last access via get() (or since creation if get() has not been called)
     */
    public long getLastAccess() {
        return _lastAccess;
    }
}
