
package com.openexchange.usm.datatypes.mail;

import java.util.ArrayList;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;


/**
 * POJO for defining vacation objects. Vacation objects are not synchronized!
 * 
 * @author <a href="mailto:ljupka.grabsh@microdoc.com">Ljupka Grabsch</a>
 */
public class Vacation {

	private static final String SUBJECT = "subject";
	public static final String ID = "id";
	private static final String ADDRESSES = "addresses";
	private static final String TEXT = "text";
    private static final String DAYS = "days";
    private static final int DAYS_DEFAULT = 7;
    private static final String ACTIVE = "active";
    public static final String RULE_ID = "vacation";
    private static final String RULENAME = "rulename";
    private static final String RULENAME_DEFAULT = "Abwesenheit";
    private static final String ACTIONCMDS = "actioncmds";
    private static final String TEST = "test";
    private static final String TESTS = "tests";
    private static final String FLAGS = "flags";
    private static final String COMPARISON = "comparison";
    private static final String COMPARISON_GE = "ge";
    private static final String COMPARISON_LE = "le";
    private static final String DATEVALUE = "datevalue";

    private boolean active;
    private int days;
    private String text;
    private String subject;
    private int id;
    private String rulename;

    private List<String> addresses;

    private long startTime = 0;
    private long endTime = 0;

    public Vacation() {
        this.active = false;
        this.days = DAYS_DEFAULT;;
        this.text = "";
        this.addresses = new ArrayList<String>();
        this.id = 0;
        this.subject = "";
    }

	public Vacation(JSONObject rule) throws JSONException {
		// Example:
		// {"id":0,"position":0,"rulename":"vacation notice","active":true,"flags":["vacation"],
		// Cyrus: "test":{"id":"true"},
		// Dovecot:
		// "test":{"id":"allof","tests":[{"id":"currentdate","comparison":"ge","datepart":"date","datevalue":[1435968000000]},{"id":"currentdate","comparison":"le","datepart":"date","datevalue":[1437609600000]}]},
		// ))
		// "actioncmds":[{"id":"vacation","days":"1","addresses":["user4@ox.microdoc.de"],"text":"This is the OOF message"}]}]}
		// {"active":true,"id":0,"test":{"id":"true"},"rulename":"Abwesenheit","position":0,"actioncmds":[{"id":"vacation","days":"7","text":"This is a test Out Of Office email.","addresses":["user5@ox-p5.microdoc.com","user5@ox.
		active = rule.optBoolean(ACTIVE);
		id = rule.optInt(ID);
		rulename = rule.optString(RULENAME, RULENAME_DEFAULT);

		JSONObject test = rule.getJSONObject(TEST);
		JSONArray tests = test.optJSONArray(TESTS);
		for (int i = 0; tests != null && i < tests.length(); i++) {
			JSONObject dateTest = tests.getJSONObject(i);
			String comparison = dateTest.getString(COMPARISON);
			long datevalue = dateTest.getJSONArray(DATEVALUE).getLong(0);
			if (comparison.equals(COMPARISON_GE)) {
				startTime = datevalue;
			} else if (comparison.equals(COMPARISON_LE)) {
				endTime = datevalue;
			}
		}

		JSONArray actionCmds = rule.optJSONArray(ACTIONCMDS);
		if (actionCmds != null && actionCmds.length() > 0) {
			JSONObject actionCmd = actionCmds.optJSONObject(0);
			days = actionCmd.optInt(DAYS, DAYS_DEFAULT);
			text = actionCmd.optString(TEXT);
			subject = actionCmd.optString(SUBJECT);
			JSONArray addressesJSON = actionCmd.has(ADDRESSES) ? actionCmd
					.optJSONArray(ADDRESSES) : new JSONArray();
			addresses = new ArrayList<String>();
			for (int i = 0; i < addressesJSON.length(); i++) {
				addresses.add(addressesJSON.optString(i));
			}
		}

	}
	
	//{"id":"currentdate","comparison":"ge","datepart":"date","datevalue":[1435968000000]}
	private JSONObject createDateTest(String comparison, long time) throws JSONException{
		 JSONObject dateTest = new JSONObject();
		 dateTest.put(ID, "currentdate");
		 dateTest.put(COMPARISON, comparison);
		 dateTest.put("datepart", "date");
		 JSONArray datevalue = (new JSONArray()).put(time);
		 dateTest.put("datevalue", datevalue);
		 return dateTest;
	}

	public boolean hasTimes() {
	    return getStartTime() != 0 && getEndTime() != 0;
	}
	
    public void removeTimes() {
        setStartTime(0);
        setEndTime(0);
    }
	
    public JSONObject toJSONObject() throws JSONException {
        JSONObject actionCmd = new JSONObject();
        actionCmd.put(ID, RULE_ID);
        actionCmd.put(DAYS, days);
        if (!addresses.isEmpty())
            actionCmd.put(ADDRESSES, new JSONArray(addresses));
        actionCmd.put(TEXT, text);
        actionCmd.put(SUBJECT, subject);
        
        JSONObject test = new JSONObject();
        if (hasTimes()){
            test.put(ID, "allof");
            JSONArray tests = new JSONArray();
            tests.put(createDateTest(COMPARISON_GE, startTime));
            tests.put(createDateTest(COMPARISON_LE, endTime));
            test.put(TESTS, tests); 
        } else {
            test.put(ID, "true");
        }

        JSONObject rule = new JSONObject();
        rule.put(RULENAME, rulename);
        rule.put(ACTIVE, isActive());
        rule.put(FLAGS, (new JSONArray()).put(RULE_ID));
        rule.put(ID, getId());
        rule.put(TEST, test);
        rule.put(ACTIONCMDS,(new JSONArray()).put(actionCmd));
        
        return rule;
    }

    public boolean isActive() {
        return active;
    }

    public int getDays() {
        return days;
    }

    public String getText() {
        return text;
    }

    public List<String> getAddresses() {
        return addresses;
    }

    public String getSubject() {
        return subject;
    }

    public int getId() {
        return id;
    }

    public void setActive(boolean active) {
        this.active = active;
    }

    public void setDays(int days) {
        this.days = days;
    }

    public void setText(String text) {
        this.text = text;
    }

    public void setSubject(String subject) {
        this.subject = subject;
    }

    public void setId(int id) {
        this.id = id;
    }

    public void setAddresses(List<String> addresses) {
        this.addresses = addresses;
    }

    public long getStartTime() {
        return startTime;
    }

    public void setStartTime(long startTime) {
        this.startTime = startTime;
    }

    public long getEndTime() {
        return endTime;
    }

    public void setEndTime(long endTime) {
        this.endTime = endTime;
    }
    
    public String getRulename() {
        return rulename;
    }

    public void setRulename(String rulename) {
        this.rulename = rulename;
    }
    

    @Override
    public String toString() {
        return "Vacation [active=" + active + ", text=" + text + ", subject=" + subject + ", id=" + id + ", addresses=" + addresses + "]";
    }

}
