/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.CONFLICT_RESOLUTION;
import static com.openexchange.usm.connector.commands.CommandConstants.CREATED;
import static com.openexchange.usm.connector.commands.CommandConstants.DELETED;
import static com.openexchange.usm.connector.commands.CommandConstants.DELETED_IF_EXIST;
import static com.openexchange.usm.connector.commands.CommandConstants.FOLDERID;
import static com.openexchange.usm.connector.commands.CommandConstants.LIMIT;
import static com.openexchange.usm.connector.commands.CommandConstants.MODIFIED;
import static com.openexchange.usm.connector.commands.CommandConstants.REFRESH;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.SYNCID;
import static com.openexchange.usm.json.response.ResponseStatusCode.*;
import static com.openexchange.usm.json.ConnectorBundleErrorCodes.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import javax.servlet.http.HttpServletRequest;

import org.json.JSONArray;
import org.json.JSONObject;

import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.FolderConstants;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.exceptions.InvalidUUIDException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.api.session.assets.ConflictResolution;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.api.session.exceptions.SlowSyncRequiredException;
import com.openexchange.usm.api.session.exceptions.SynchronizationConflictException;
import com.openexchange.usm.connector.exceptions.DataObjectNotFoundException;
import com.openexchange.usm.connector.exceptions.MultipleOperationsOnDataObjectException;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.session.dataobject.FolderHierarchyComparator;
import com.openexchange.usm.util.UUIDToolkit;

/**
 * Handler for the SyncUpdate USM-JSON-Command.
 * 
 * @author ldo
 */
public class SyncUpdateHandler extends SyncCommandHandler {

    private static final String[] REQUIRED_PARAMETERS = { SESSIONID, SYNCID };

    private static final String[] OPTIONAL_PARAMETERS = {
        FOLDERID, CREATED, MODIFIED, DELETED, DELETED_IF_EXIST, LIMIT, CONFLICT_RESOLUTION, REFRESH };

    private static class MergeData {

        public final DataObject _deletion;

        public final DataObject _creation;

        public final DataObject _change;

        public boolean _errorOccurred;

        private MergeData(DataObject deletion, DataObject creation, DataObject change) {
            _deletion = deletion;
            _creation = creation;
            _change = change; // TODO Should we add the code to create the change object in this constructor ?
        }
    }

    private long _syncID;

    private boolean _serverChangesMayBeAvailable;

    private final List<MergeData> _mergedCreationsAndDeletions = new ArrayList<MergeData>();

    private final Set<UUID> _refreshUUIDs = new HashSet<UUID>();

    public SyncUpdateHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        String folderUUID = getStringParameter(FOLDERID, null);
        String lockKey = (folderUUID == null) ? "<FolderHierarchy>" : folderUUID;
        String acquirer = "syncUpdate:" + lockKey;
        String previousAcquirer = _session.tryLock(lockKey, acquirer);
        if (previousAcquirer != null)
            return createCannotLockFolderResponse("syncUpdate", "", lockKey, previousAcquirer);
        try {
            readOptionalLimit();
            ConflictResolution conflictResolution = getOptionalConflictResolution();
            _syncID = getSyncID();
            if(folderUUID != null)
                setFolderToSync(folderUUID);
            // Check if the client used an old SyncKey (repeated usage of an old syncKey indicates a problem)
            boolean usedOldSyncKey = _session.getNewestTimestamp(_folderToSync == null ? "" : _folderToSync.getID()) > _syncID;
            if (usedOldSyncKey) {
                _servlet.getJournal().debug(_session + " Synchronization with old sync id " + _syncID + " for folder " + folderUUID);
            }
            readOptionalRefreshUUIDs();
            _syncResult = computeSyncUpdate(folderUUID, _syncID, conflictResolution);
            if (folderUUID == null) {
                _allFolders = null; // make sure current folder hierarchy is read again if required for generating response
                BitSet deletedStandardFolders = getStandardFolderDeletionsFromSyncResult();
                if(!deletedStandardFolders.isEmpty()) {
                    String errorMessage = "The following required standard folder types are currently not available: " + deletedStandardFolders;
                    _servlet.getJournal().warn(
                        _session + " Aborting sync of folder hierarchy: " + errorMessage);
                    return new ResponseObject(TEMPORARY_NOT_AVAILABLE, SYNC_UPDATE_STANDARD_FOLDERS_REMOVED_ON_SERVER, errorMessage, null, null);
                }
            }
            initializeErrorMap(_syncResult);
            long syncId = _syncResult.getTimestamp();
            if (_recurrenceUUIDMap.size() > 0 && _folderToSync != null) {
                // make additional update to retrieve all created exception objects on the server,
                // and replace the uuids created on the server with the original uuids sent from the client
                retrieveExceptionsFromServerAndSetUUIDs(syncId, _folderToSync);
            }
            syncId = finishProcessingAndSaveNewState(_syncResult, _folderToSync, syncId);
            JSONObject response = createResponseFromSyncResults(_syncResult, syncId);
            logSyncData(" syncUpdate", folderUUID, _syncID, syncId, _syncResult.getTimestamp());
            if (usedOldSyncKey || isUnnecessaryAccess(folderUUID, response))
                _servlet.getAccessLimiter().badAccessEncountered(_session);
            return new ResponseObject(getSuccessResponseCode(), response);
        } finally {
            _session.unlock(lockKey);
        }
    }

	private boolean isUnnecessaryAccess(String folderUUID, JSONObject response) {
		// Check if the sync was not requested by the server, if the client sent no data and the server sent no data -> unnecessary
		if (_serverChangesMayBeAvailable || response.has("errors")
				|| countObjects(_parameters, "created", "modified", "deleted", "refresh") > 0
				|| countObjects(response, "created", "modified", "deleted") > 0)
			return false;
		_servlet.getJournal().debug(_session + " Unnecessary synchronization for folder " + folderUUID);
		return true;
	}

	private int countObjects(JSONObject o, String... keys) {
		int count = 0;
		for (String key : keys) {
			JSONArray array = o.optJSONArray(key);
			if (array != null)
				count += array.length();
		}
		return count;
	}

    private void readOptionalRefreshUUIDs() throws USMJSONAPIException {
        if (_parameters.has(REFRESH)) {
            JSONArray list = getJSONArray(_parameters, REFRESH);
            int size = list.length();
            for (int i = 0; i < size; i++) {
                String uuid = getString(list, i);
                try {
                    _refreshUUIDs.add(UUIDToolkit.extractUUIDFromString(uuid));
                } catch (InvalidUUIDException e) {
                    addError(uuid, null, e, ErrorStatusCode.UNKNOWN_UUID);
                }
            }
        }
    }

	/**
	 * Modified behavior from default implementation if client sent a non-empty "refresh" array:
	 * - Check for all UUIDs in the refresh array if an object is in the client result list.
	 * - If it is not, but it is present in the new sync state, add the object to the changes array
	 * - If it is not, and it isn't present in the new sync state, add the UUID to the "deleted" result list
	 */
	@Override
	protected DataObject[] buildClientChangesArray(DataObject[] changes, JSONArray deleted) {
		if (!_refreshUUIDs.isEmpty()) {
			DataObjectSet allObjects = (_newStateToSave == null) ? new DataObjectSet(_syncResult.getNewState())
					: _newStateToSave;
			DataObjectSet extendedChanges = new DataObjectSet(changes);
			for (UUID uuid : _refreshUUIDs) {
				if (!extendedChanges.contains(uuid)) {
					DataObject o = allObjects.get(uuid);
					if (o == null)
						deleted.put(uuid);
					else
						extendedChanges.add(o);
				}
			}
			changes = extendedChanges.toArray();
		}
		return super.buildClientChangesArray(changes, deleted);
	}

	/**
	 * Determine ChangeState to report to the client:
	 * - If UUID is present in refresh parameter, and object wasn't deleted, report it as created
	 * - If object was deleted or its UUID is not present in refresh parameter, report it as normal (created, modified, deleted)
	 */
	@Override
	protected ChangeState determineClientResultChangeState(DataObject o) {
		return (_refreshUUIDs.contains(o.getUUID()) && o.getChangeState() != ChangeState.DELETED) ? ChangeState.CREATED
				: super.determineClientResultChangeState(o);
	}

    /**
     * If the UUID is present in the refresh parameter, always report all fields If not, use normal behavior (i.e. non-default on creation,
     * only modified fields for modifications)
     */
    @Override
    protected boolean shouldIncludeFieldInClientResult(DataObject o, int i, DataType<?> fieldType, Object v, String fieldName) {
        return _refreshUUIDs.contains(o.getUUID()) || isInClientChangesArray(o) ? true : super.shouldIncludeFieldInClientResult(
            o,
            i,
            fieldType,
            v,
            fieldName);
    }

	@Override
	protected boolean shouldIncludeContactPicture(DataObject o) {
		return _refreshUUIDs.contains(o.getUUID()) ? true : super.shouldIncludeContactPicture(o);
	}

	private boolean isInClientChangesArray(DataObject change) {
		if (_clientChangesArray != null)
			for (DataObject object : _clientChangesArray) {
				if (change.getUUID().equals(object.getUUID()))
					return true;
			}
		if (_folderClientChangesArray != null)
			for (DataObject object : _folderClientChangesArray) {
				if (change.getUUID().equals(object.getUUID()))
					return true;
			}
		return false;
	}

	@Override
	protected void initializeCachedElements() throws StorageAccessException, USMStorageException {
		_cachedElements.addAll(_session.getCachedFolderElements(_folderToSync.getID(), _folderToSync
				.getElementsContentType(), _syncID));
	}

	private void retrieveExceptionsFromServerAndSetUUIDs(long syncId, Folder folder) throws USMJSONAPIException {
		for (Map.Entry<DataObject, UUID> entry : _recurrenceUUIDMap.entrySet()) {
			DataObject object = entry.getKey();
			if (object.getChangeState() != ChangeState.MODIFIED && hasNoError(object)) {
				try {
					_newStateToSave = getNewStateToSave(folder, syncId, object, entry.getValue());
				} catch (USMException e) {
					addError(object, USMJSONAPIException.createInternalError(
							ConnectorBundleErrorCodes.SYNC_UPDATE_ERROR_APP_EXCEPTION, e), ErrorStatusCode.OTHER);
				}
			}
		}
	}

    protected Folder[] _folderClientChangesArray;

   
	private SyncResult computeSyncUpdate(String folderID, long syncId, ConflictResolution conflictResolution)
			throws USMJSONAPIException {
		try {
			if (folderID == null) {
				//update folder hierarchy
				_serverChangesMayBeAvailable = _session.needsSynchronization(null);
				_folderClientChangesArray = filterFoldersWithErrors(createFolderChangesArray(syncId));
				Arrays.sort(_folderClientChangesArray, new FolderHierarchyComparator(_folderClientChangesArray,
						_servlet.getJournal()));
				return _session.syncFolderChangesWithServer(syncId, _limit, null, true, conflictResolution,
                    _folderClientChangesArray);
			}
			_serverChangesMayBeAvailable = _session.needsSynchronization(_folderToSync.getID());
			_clientChangesArray = filterDataObjectsWithErrors(buildClientRequestArray(_folderToSync, syncId, conflictResolution));
			if (_syncIdForUpdate > 0) {
				syncId = _syncIdForUpdate;
				_syncIdForUpdate = 0L;
			}
			return _session.syncChangesWithServer(_folderToSync.getID(), syncId, _limit, getFilter(_folderToSync),
					true, conflictResolution, _clientChangesArray);
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (SlowSyncRequiredException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SYNC_UPDATE_INVALID_SYNCID,
					ResponseStatusCode.UNKNOWN_SYNCID, "Unknown SyncID "
							+ syncId
							+ " for "
							+ ((folderID == null) ? "folder hierarchy" : ((_folderToSync != null) ? (_folderToSync
									.getID()
									+ "/" + folderID) : folderID)));
		} catch (SynchronizationConflictException e) {
			throw generateConflictException(e);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_UPDATE_INTERNAL_ERROR, e);
		}
	}

    private BitSet getStandardFolderDeletionsFromSyncResult() {
        return getStandardFolderDeletionsFromSyncResultCore(_syncResult.getNewState());
    }

    // core function for testing
    public static BitSet getStandardFolderDeletionsFromSyncResultCore(DataObject[] newSyncState) {
        
        BitSet notExistingStandardFolders = new BitSet(); // all standard folder types that do not exist
        // initialize the bit set
        notExistingStandardFolders.set(1,4);
        notExistingStandardFolders.set(7);
        notExistingStandardFolders.set(9,13);
        
        for (DataObject o : newSyncState) {
            int standardFolderType = getRequiredStandardFolderType(o, false);
            if (standardFolderType > 0 && o.getChangeState() != ChangeState.DELETED)
                notExistingStandardFolders.clear(standardFolderType);
        }
        return notExistingStandardFolders;
    }

    /**
     * @param folder
     * @param useOriginalContent for modifications, determine if the current or original content of the object should be used
     * @return standard_folder_type if required standard folder, otherwise 0
     */
    private static int getRequiredStandardFolderType(DataObject folder, boolean useOriginalContent) {
        Object fieldValue = useOriginalContent ? folder.getOriginalFieldContent(FolderConstants.STANDARD_FOLDER) : folder.getFieldContent(FolderConstants.STANDARD_FOLDER);
        if (!(fieldValue instanceof Boolean) || !((Boolean) fieldValue).booleanValue())
            return 0; // No standard folder -> not required
        fieldValue = useOriginalContent ? folder.getOriginalFieldContent(FolderConstants.STANDARD_FOLDER_TYPE) : folder.getFieldContent(FolderConstants.STANDARD_FOLDER_TYPE);
        if (fieldValue instanceof Number) {
            int standardFolderType = ((Number) fieldValue).intValue();
            switch (standardFolderType) {
            case 1: // Task
            case 2: // Calendar
            case 3: // Contact
            case 7: // Inbox
            case 9: // Drafts
            case 10: // Sent
            case 11: // Spam
            case 12: // Trash
                return standardFolderType;
            default:
                break;
            }
        }
        return 0;
    }

    private Folder[] createFolderChangesArray(long syncId) throws StorageAccessException, USMStorageException,
			USMJSONAPIException, SlowSyncRequiredException {
		Folder[] originalFolders = _session.getCachedFolders(syncId);
		if (originalFolders == null)
			throw new SlowSyncRequiredException(-1, null);
		_allFolders = originalFolders;
		Folder[] createdFolders = readCreatedFolders(originalFolders);
		Folder[] modifiedFolders = readModifiedFolders(originalFolders);
		Folder[] deletedFolders = readDeletedFolders(originalFolders);
		Folder[] result = new Folder[createdFolders.length + modifiedFolders.length + deletedFolders.length];
		int index = 0;
		for (Folder folder : createdFolders) {
			result[index++] = folder;
		}
		for (Folder folder : modifiedFolders) {
			result[index++] = folder;
		}
		for (Folder folder : deletedFolders) {
			result[index++] = folder;
		}
		return result;
	}

    private DataObject[] buildClientRequestArray(Folder folder, long syncId, ConflictResolution conflictResolution) throws USMException {
        DataObject[] originalElements = _session.getCachedFolderElements(folder.getID(), folder.getElementsContentType(), syncId);
        if (originalElements == null)
            throw new SlowSyncRequiredException(-1, null);
        ContentType elementsContentType = folder.getElementsContentType();
        if (elementsContentType == null)
            return EMPTY_DATAOBJECT_ARRAY;
        DataObject[] createdElements = readCreatedFolderElements(folder, syncId, conflictResolution, originalElements);
        DataObject[] modifiedElements;
        if (elementsContentType.getCode() == DefaultContentTypes.MAIL_CODE) {
            modifiedElements = readModifiedMailDataObjects(folder, elementsContentType, originalElements, syncId);
        } else {
            modifiedElements = readModifiedFolderElements(folder, originalElements, createdElements);
        }
        DataObject[] deletedElements = readDeletedFolderElements(originalElements);
        if (createdElements.length == 0 || deletedElements.length == 0 || !isTaskOrAppointmentType(elementsContentType))
            return buildSimpleClientRequestArray(createdElements, modifiedElements, deletedElements);
        return buildComplexClientRequestArray(createdElements, modifiedElements, deletedElements);
    }

    private DataObject[] buildComplexClientRequestArray(DataObject[] createdElements, DataObject[] modifiedElements, DataObject[] deletedElements) {
        List<DataObject> result = new ArrayList<DataObject>();
        Map<String, DataObject> deletions = new HashMap<String, DataObject>();
        for (DataObject deletion : deletedElements) {
            String uid = getUID(deletion);
            if (uid != null)
                deletions.put(uid, deletion);
            else
                result.add(deletion);
        }
        for (DataObject object : modifiedElements) {
            result.add(object);
        }
        for (DataObject creation : createdElements) {
            String uid = getUID(creation);
            if (uid != null) {
                DataObject deletion = deletions.remove(uid);
                if (deletion == null) {
                    result.add(creation);
                } else {
                    // Create copy of original object in SyncState
                    DataObject change = deletion.createCopy(false);
                    change.rollbackChanges();
                    // Apply all fields from the creation so that they will become changes in the new change object
                    DataObjectUtil.modify(change, creation);
                    // Make sure the OX ID is switched back to the original ID so that the sync system can find the object on the server
                    change.setID(deletion.getID());
                    _mergedCreationsAndDeletions.add(new MergeData(deletion, creation, change));
                    result.add(change);
                }
            } else {
                result.add(creation);
            }
        }
        for (DataObject deletion : deletions.values()) {
            result.add(deletion);
        }
        return result.toArray(new DataObject[result.size()]);
    }

    private DataObject[] buildSimpleClientRequestArray(DataObject[] createdElements, DataObject[] modifiedElements, DataObject[] deletedElements) {
        DataObject[] result = new DataObject[createdElements.length + modifiedElements.length + deletedElements.length];
        int index = 0;
        for (DataObject object : createdElements) {
            result[index++] = object;
        }
        for (DataObject object : modifiedElements) {
            result[index++] = object;
        }
        for (DataObject object : deletedElements) {
            result[index++] = object;
        }
        return result;
    }

    private boolean isTaskOrAppointmentType(ContentType type) {
        int code = type.getCode();
        return code == DefaultContentTypes.CALENDAR_CODE || code == DefaultContentTypes.TASK_CODE;
    }

    private String getUID(DataObject o) {
        try {
            Object v = o.getFieldContent("uid");
            return (v instanceof String) ? ((String) v) : null;
        } catch (IllegalArgumentException iae) {
            return null;
        }
    }

    private DataObject[] readModifiedMailDataObjects(Folder folder, ContentType contentType, DataObject[] originalObjects, long syncId) throws USMException {
        if (!_parameters.has(MODIFIED))
            return EMPTY_DATAOBJECT_ARRAY;
        JSONArray list = getJSONArray(_parameters, MODIFIED);
        int size = list.length();
        List<DataObject> result = new ArrayList<DataObject>(size);
        DataObject[] syncStateToStore = new DataObject[originalObjects.length];
        syncStateToStore = originalObjects;
        for (int i = 0; i < size; i++) {
            JSONObject data = getJSONObject(list, i);
            String uuidString = getUUIDString(data);
            DataObject object;
            try {
                object = getDataObjectFromJSONObjectAndOriginalObjects(contentType, data, originalObjects);
            } catch (DataObjectNotFoundException e) {
                addError(uuidString, null, e, ErrorStatusCode.UNKNOWN_UUID);
                continue;
            } catch (MultipleOperationsOnDataObjectException e) {
                addError(uuidString, e.getDataObject(), e, ErrorStatusCode.MULTIPLE_OPERATIONS_ON_SAME_UUID);
                continue;
            } catch (InvalidUUIDException e) {
                addError(uuidString, null, e, ErrorStatusCode.UNKNOWN_UUID);
                continue;
            }
            if (hasNoError(object)) {
                try {
                    _syncIdForUpdate = updateMailObject(
                        folder,
                        originalObjects,
                        syncId,
                        result,
                        data,
                        object,
                        _syncIdForUpdate,
                        syncStateToStore);
                } catch (USMJSONAPIException e) {
                    addError(object, e);
                }
            }
        }
        // if syncStateToStore contains objects from other folder, reset syncStateToStore to originalObjects
        if (containsForeignObjects(syncStateToStore, folder.getID()))
            syncStateToStore = originalObjects;

        _syncIdForUpdate = _session.storeSyncState(getOriginalSyncID(), syncId, folder.getID(), syncStateToStore);
        DataObject[] savedElements = _session.getCachedFolderElements(folder.getID(), contentType, _syncIdForUpdate);
        for (int i = 0; i < size; i++) {
            JSONObject data = getJSONObject(list, i);
            String uuidString = getUUIDString(data);
            try {
                DataObject object = getDataObjectFromJSONObjectAndOriginalObjects(contentType, data, savedElements);
                if (hasNoError(object))
                    result.add(object);
            } catch (DataObjectNotFoundException e) {
                addError(uuidString, null, e, ErrorStatusCode.UNKNOWN_UUID);
            } catch (MultipleOperationsOnDataObjectException e) {
                addError(uuidString, e.getDataObject(), e, ErrorStatusCode.MULTIPLE_OPERATIONS_ON_SAME_UUID);
            } catch (InvalidUUIDException e) {
                addError(uuidString, null, e, ErrorStatusCode.UNKNOWN_UUID);
            }
        }
        return result.toArray(new DataObject[result.size()]);
    }

    private boolean containsForeignObjects(DataObject[] syncStateToStore, String folderId) {
        for (DataObject dataObject : syncStateToStore) {
            if (!folderId.equals(dataObject.getParentFolderID()))
                return true;
        }
        return false;
    }

    @Override
    protected String[] getOptionalParameters() {
        return OPTIONAL_PARAMETERS;
    }

    @Override
    protected String[] getRequiredParameters() {
        return REQUIRED_PARAMETERS;
    }

    @Override
    protected long getOriginalSyncID() {
        return _syncID;
    }

    @Override
    protected long saveNewSyncState(SyncResult result, Folder folder, long syncId) throws USMJSONAPIException {
        if (!_mergedCreationsAndDeletions.isEmpty()) {
            for (MergeData md : _mergedCreationsAndDeletions) {
                if (md._errorOccurred) {
                    // Undo changes in sync state
                    DataObject original = md._deletion.createCopy(false);
                    original.rollbackChanges();
                    _newStateToSave.remove(md._creation.getUUID());
                    _newStateToSave.add(original);
                } else {
                    DataObject dataObject = _newStateToSave.get(md._deletion.getID());
                    if (dataObject != null)
                        dataObject.setUUID(md._creation.getUUID());
                    for (DataObject serverChange : result.getChanges()) {
                        if (md._deletion.getID().equals(serverChange.getID()))
                            serverChange.setUUID(md._creation.getUUID());
                    }
                }
            }
            _newStateToSaveChanged = true;
        }
        return super.saveNewSyncState(result, folder, syncId);
    }

    @Override
    protected void addError(UUID uuid, DataObject object, USMException error, ErrorStatusCode errorStatus) {
        if (uuid == null)
            uuid = object.getUUID();
        if (uuid != null) {
            if (!_mergedCreationsAndDeletions.isEmpty()) {
                for (MergeData md : _mergedCreationsAndDeletions) {
                    if (uuid.equals(md._deletion.getUUID()) || uuid.equals(md._creation.getUUID())) {
                        md._errorOccurred = true;
                        addError(md._deletion.getUUID().toString(), object, error, errorStatus);
                        addError(md._creation.getUUID().toString(), object, error, errorStatus);
                        return;
                    }
                }
            }
            addError(uuid.toString(), object, error, errorStatus);
        }
    }
}
