/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.*;
import static com.openexchange.usm.json.ConnectorBundleErrorCodes.*;
import static com.openexchange.usm.json.response.ResponseStatusCode.*;
import java.util.BitSet;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.FolderConstants;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;
import com.openexchange.usm.api.contenttypes.transferhandlers.FolderContentTypeTransferHandler;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.datatypes.PIMAttachments;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.api.session.assets.FolderType;
import com.openexchange.usm.datatypes.contacts.ContactDistributionMember;
import com.openexchange.usm.datatypes.contacts.Image;
import com.openexchange.usm.datatypes.tasks.calendar.CommonCalendarTasksFieldNames;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.HttpSessionUtil;
import com.openexchange.usm.json.JSONSessionInitializer;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.USMJSONVersion;
import com.openexchange.usm.json.USMSessionCredentials;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.json.streaming.ServerTempId;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.util.Toolkit;

public abstract class NormalCommandHandler extends BaseCommandHandler {

    protected static final String GLOBAL_ADDRESS_BOOK_ID = "6";

    private static final String DELETE_EXCEPTIONS = "delete_exceptions";
    
    private static final String OWN_RIGHTS = "own_rights";

    protected DataObject[] _allFolders;
    
    // Used for MeetingRequests, to only search once for the default calendar folder if there are multiple MeetingRequests to sync
    protected Folder _defaultCalendarFolder = null;
    
    protected Folder _defaultConcatcsFolder = null;

    protected NormalCommandHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
        String sessionID = getStringParameter(SESSIONID);
        USMSessionCredentials usmSessionID = HttpSessionUtil.getSession(getHttpSession(), sessionID);
        setSession(retrieveUSMSession(servlet, usmSessionID));
    }

    protected Session retrieveUSMSession(USMJSONServlet servlet, USMSessionCredentials creds) throws USMJSONAPIException {
        return retrieveUSMSession(servlet, _currentHttpRequest, creds);
    }

    protected FolderContentType getFolderContentType() {
        return (FolderContentType) _servlet.getContentTypeManager().getContentType(DefaultContentTypes.FOLDER_ID);
    }

	protected void storeExtraFields(ContentType type, BitSet extraFields) throws StorageAccessException,
			USMStorageException {
        JSONSessionInitializer.storeExtraFields(_session, type, extraFields);
    }

    protected JSONObject storeCompleteDataObjectInJSONObject(DataObject o) throws JSONException, USMJSONAPIException {
        return storeCompleteDataObjectInJSONObject(o, null);
    }

	protected JSONObject storeCompleteDataObjectInJSONObject(DataObject o,
			Map<DataObject, List<DataObject>> exceptionsMap) throws JSONException, USMJSONAPIException {
        List<DataObject> exceptions = getExceptionsOfDataObject(o, exceptionsMap);
        JSONObject newObject = storeDataObjectFieldsInJSONObject(o, exceptions);
        if (exceptions != null)
            storeExceptionsInJSONObject(newObject, exceptions);
        newObject.put(OBJECT_TYPE, o.getContentType().getID());
        if(o.getUUID() != null) newObject.put(UUID_KEY, o.getUUID().toString());
        return newObject;
    }

    private List<DataObject> getExceptionsOfDataObject(DataObject o, Map<DataObject, List<DataObject>> exceptionsMap) {
        if (exceptionsMap != null) {
            for (Map.Entry<DataObject, List<DataObject>> entry : exceptionsMap.entrySet()) {
                DataObject object = entry.getKey();
                if (object.getID().equals(o.getID()))
                    return entry.getValue();
            }
        }
        return null;
    }

	private void storeExceptionsInJSONObject(JSONObject jo, List<DataObject> exceptions) throws USMJSONAPIException,
			JSONException {
        JSONArray exceptionsJSONArray = new JSONArray();

        JSONArray deleteExceptions = jo.optJSONArray(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS);
        for (DataObject changeException : exceptions) {
            boolean existsInDeleteExceptions = false;
            for (int i = 0; deleteExceptions != null && i < deleteExceptions.length(); i++) {
				if (changeException.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION).equals(
						deleteExceptions.get(i)))
                    existsInDeleteExceptions = true;
            }
            if (!existsInDeleteExceptions) {
                JSONObject exceptionJSONObject = storeCompleteDataObjectInJSONObject(changeException);
                exceptionsJSONArray.put(exceptionJSONObject);
            }
        }
        jo.put(CHANGE_EXCEPTIONS, exceptionsJSONArray);
    }

    protected JSONObject storeDataObjectFieldsInJSONObject(DataObject o, List<DataObject> exceptions) throws JSONException, USMJSONAPIException {
        JSONObject newObject = new JSONObject();
        ContentTypeField[] fields = o.getContentType().getFields();
        for (int i = 0; i < fields.length; i++) {
            DataType<?> fieldType = fields[i].getFieldType();
            Object v = o.getFieldContent(i);
            String fieldName = fields[i].getFieldName();
            if (shouldIncludeFieldInClientResult(o, i, fieldType, v, fieldName)) {
                if (DefaultContentTypes.MAIL_ID.equals(o.getContentType().getID()) && JSONSessionInitializer.isInternalMailField(fieldName))
                    continue;
                if (IMAGE1.equals(fieldType.getName())) {
                    if (!(v instanceof Image)) {
                        newObject.put(IMAGE1, JSONObject.NULL);
                    } else {
                        Image image = (Image) v;
                        if (image.getTempId() != null) {
                            newObject.put(TEMPID, image.getTempId());
                        } else if (image.getData() != null) {
                            newObject.put(IMAGE1, Toolkit.encodeBase64(image.getData()));
                        } else {
                            newObject.put(IMAGE1, JSONObject.NULL);
                        }
                    }
                } else if (ATTACHMENTS_LAST_MODIFIED.equals(fieldName)) {
                    storeAttachmentsInJSONObject(newObject, v, convertToPIMAttachmentDataType(fieldType), o);
                } else if (NUMBER_OF_ATTACHMENTS.equals(fieldName)) {
                    fieldType.storeInJSONObject(o.getSession(), newObject, fieldName, v);
                    if (!o.isFieldModified(ATTACHMENTS_LAST_MODIFIED)) {
                        for (int j = 0; j < fields.length; j++) {
                            if (ATTACHMENTS_LAST_MODIFIED.equals(fields[j].getFieldName())) {
                                fieldType = fields[j].getFieldType();
                                v = o.getFieldContent(j);
                                storeAttachmentsInJSONObject(newObject, v, convertToPIMAttachmentDataType(fieldType), o);
                            }
                        }
                    }
                } else if (CommandConstants.DISTRIBUTION_LIST.equals(fieldName)) {
                    JSONArray newList = createDistributionListJSONArrayForResult(o, newObject, fieldType, v, fieldName);
                    newObject.put(CommandConstants.DISTRIBUTION_LIST, newList);
                } else if (o instanceof Folder && TITLE.equals(fieldName)) {
                    String folderTitleForClient = generateFolderTitleForClient((Folder) o, v);
                    fieldType.storeInJSONObject(o.getSession(), newObject, fieldName, folderTitleForClient);
                } else {
                    fieldType.storeInJSONObject(o.getSession(), newObject, fieldName, v);
                }
            }
        }

        if (o.getContentType().getCode() == DefaultContentTypes.FOLDER_CODE) {
            addOptionalFolderUUID(o, newObject);
		} else if (o.getContentType().getCode() == DefaultContentTypes.CALENDAR_CODE && !isSimpleAppointment(o)
				&& !isAppException(o)) {
            JSONArray deleteExceptions = new JSONArray();
            boolean foundChange = false;
            Object v = o.getFieldContent(DELETE_EXCEPTIONS);
            if (v instanceof Object[]) {
                for (Object e : ((Object[]) v))
                    deleteExceptions.put(e);
                v = o.getFieldContent(CHANGE_EXCEPTIONS);
                if (v instanceof Object[]) {
                    for (Object e : ((Object[]) v)) {
						if (isChangeExceptionOutOfSyncRange(e)
								|| (!isParentFolderShared(o) && !isCurrentUserParticipantToChangeException(exceptions,
                            e)))
                            deleteExceptions.put(e);
                        else
                            foundChange = true;
                    }
                }
            }
            // We need to always report delete_exceptions because change_exceptions that are outside of the
            // sync interval have to be reported as delete exceptions. This means that we can not rely on the
            // modification state of the property because even if the property hasn't changed we will report
            // a different content because a change_exception moved into or out of the sync interval.
            // Only for new creations and an empty delete_exceptions list we may omit the field.
            if (deleteExceptions.length() > 0 || o.getChangeState() != ChangeState.CREATED) {
                newObject.put(DELETE_EXCEPTIONS, deleteExceptions);
            }

            if (!foundChange) {
                if (o.getChangeState() == ChangeState.MODIFIED && o.isFieldModified(CHANGE_EXCEPTIONS)) {
                    v = o.getFieldContent(CHANGE_EXCEPTIONS);
                    if (!(v instanceof Object[]) || ((Object[]) v).length == 0)
                        newObject.put(CHANGE_EXCEPTIONS, new JSONArray());
                }
            }
        }
        return newObject;
    }

    private JSONArray createDistributionListJSONArrayForResult(DataObject o, JSONObject newObject, DataType<?> fieldType, Object v, String fieldName) throws JSONException {
        fieldType.storeInJSONObject(o.getSession(), newObject, fieldName, v);
        JSONArray list = newObject.getJSONArray(CommandConstants.DISTRIBUTION_LIST);
        JSONArray newList = new JSONArray();
        int length = list.length();
        for (int j = 0; j < length; j++) {
            ContactDistributionMember dMember = new ContactDistributionMember(list.getJSONObject(j));
            JSONObject jObj = dMember.toJSONObject();
            String user_id = dMember.getUser_id();
            if (user_id != null) {
                try {
                    DataObject cachedListMember = getDefaultContacsFolder() != null ? o.getSession().getCachedFolderElement(getDefaultContacsFolder().getID(), o.getContentType(), user_id) : null;
                    if (cachedListMember == null) cachedListMember = o.getSession().getCachedFolderElement(GLOBAL_ADDRESS_BOOK_ID, o.getContentType(), user_id);
                    UUID uuid = cachedListMember != null ? cachedListMember.getUUID() : null;
                    if (uuid != null) jObj.put(UUID_KEY, uuid.toString());
                } catch (NumberFormatException ignored) {
                    // do nothing
                } catch (USMException ignored) {
                    // do nothing
                }
            }
            newList.put(jObj);
        }
        return newList;
    }
    
    protected Folder getDefaultCalendarFolder() throws StorageAccessException, USMStorageException {
        if (_defaultCalendarFolder == null)
            _defaultCalendarFolder = determineDefaultCalendarFolder();
        return _defaultCalendarFolder;
    }

    private Folder determineDefaultCalendarFolder() throws StorageAccessException, USMStorageException {
        Folder[] folders = _session.getCachedFolders();
        for (Folder f : folders) {
            if (DefaultContentTypes.CALENDAR_ID.equals(f.getElementsContentTypeID())) {
                Object v = f.getFieldContent(FolderConstants.STANDARD_FOLDER);
                if (v instanceof Boolean) {
                    if (((Boolean) v).booleanValue())
                        return f;
                } else {
                    v = f.getFieldContent(FolderConstants.STANDARD_FOLDER_TYPE);
                    if ((v instanceof Number) && ((Number) v).intValue() == FolderConstants.DEFAULT_CALENDAR_FOLDER)
                        return f;
                }
            }
        }
        return null;
    }
    
    protected Folder getDefaultContacsFolder() throws StorageAccessException, USMStorageException {
        if (_defaultConcatcsFolder == null)
            _defaultConcatcsFolder = determineDefaultContacsFolder();
        return _defaultConcatcsFolder;
    }

    private Folder determineDefaultContacsFolder() throws StorageAccessException, USMStorageException {
        Folder[] folders = _session.getCachedFolders();
        for (Folder f : folders) {
            if (DefaultContentTypes.CONTACTS_ID.equals(f.getElementsContentTypeID())) {
                Object v = f.getFieldContent(FolderConstants.STANDARD_FOLDER);
                if (v instanceof Boolean) {
                    if (((Boolean) v).booleanValue())
                        return f;
                } else {
                    v = f.getFieldContent(FolderConstants.STANDARD_FOLDER_TYPE);
                    if ((v instanceof Number) && ((Number) v).intValue() == FolderConstants.DEFAULT_CONTACTS_FOLDER)
                        return f;
                }
            }
        }
        return null;
    }
    
    private boolean isParentFolderShared(DataObject o) throws USMJSONAPIException {
        Folder parentFolder = o.getParentFolder();
        if (parentFolder == null)
            try {
                parentFolder = _session.getCachedFolder(o.getParentFolderID());
            } catch (StorageAccessException e) {
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_DB_ERROR_ON_READ_SHARED_FOLDER,
						ResponseStatusCode.DATABASE_ERROR, "DB Access Exception on reading shared folder");
            } catch (USMStorageException e) {
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_DB_ERROR_ON_READ_SHARED_FOLDER_1,
						ResponseStatusCode.DATABASE_ERROR, "SQL Exception on reading shared folder");
            }
        if (parentFolder == null)
            return false;
        else
            return FolderType.SHARED.matches(parentFolder.getFieldContent(FolderConstants.TYPE));
    }

    private boolean isCurrentUserParticipantToChangeException(List<DataObject> exceptions, Object e) {
        DataObject exceptionObject = findDataObjectByRecurrenceDatePosition(exceptions, e);
        int currentUser = _session.getUserIdentifier();
        if (exceptionObject != null) {
            int attendeeStatus = SyncCommandHandler.getAttendeeStatus(exceptionObject, currentUser);
            return attendeeStatus != -1 && attendeeStatus != 2;
        }
        return false;
    }

    private DataObject findDataObjectByRecurrenceDatePosition(Collection<DataObject> list, Object position) {
        if (list != null && position != null) {
            for (DataObject o : list) {
                if (position.equals(o.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION)))
                    return o;
            }
        }
        return null;
    }

    private boolean isChangeExceptionOutOfSyncRange(Object e) {
        return ((Number) e).longValue() < _session.getCalendarStartDate() || ((Number) e).longValue() > _session.getCalendarEndDate();
    }

	protected boolean shouldIncludeFieldInClientResult(DataObject o, int i, DataType<?> fieldType, Object v,
			String fieldName) {
		return (o.getChangeState() != ChangeState.MODIFIED) ? !fieldType.isDefaultValue(v)
				&& !shouldFilterOutField(fieldName) : o.isFieldModified(i) && isFieldSynchronized(o, i)
				&& !shouldFilterOutField(fieldName);
    }

    protected boolean shouldFilterOutField(String fieldName) {
        return CHANGE_EXCEPTIONS.equals(fieldName) || RECURRENCE_ID.equals(fieldName);
    }

    protected boolean isFieldSynchronized(DataObject o, int index) {
        BitSet contentTypeFilter = _session.getFieldFilter(o.getContentType());
        return contentTypeFilter.get(index);
    }

    protected boolean isSimpleAppointment(DataObject dataObject) {
        if (dataObject.getContentType().getCode() != DefaultContentTypes.CALENDAR_CODE)
            return false;
        Object o = dataObject.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_TYPE);
        return (o instanceof Number) && ((Number) o).intValue() == 0;
    }

    public static boolean isAppointmentException(DataObject dataObject) {
        if(dataObject == null)
            return false;
        ContentType contentType = dataObject.getContentType();
        if (contentType == null || contentType.getCode() != DefaultContentTypes.CALENDAR_CODE)
            return false;
        String id = dataObject.getID();
        if (!Toolkit.provided(id))
            return false;
        Object recurrenceId = dataObject.getFieldContent(RECURRENCE_ID);
        if (!(recurrenceId instanceof Number))
            return false;
        return ((Number) recurrenceId).intValue() != 0 && !id.equals(recurrenceId.toString());
    }

    protected boolean isAppException(DataObject dataObject) {
        return isAppointmentException(dataObject);
    }

    protected void addOptionalFolderUUID(DataObject o, JSONObject object) throws USMJSONAPIException, JSONException {
        if (object.has(FOLDER_ID)) {
            initAllFolders();
            DataObject parent = DataObjectUtil.findDataObject(o.getParentFolderID(), _allFolders);
            object.put(FOLDER_UUID, (parent != null) ? parent.getUUID().toString() : "");
        }
    }

	private void initAllFolders() throws USMJSONAPIException {
		if (_allFolders != null)
			return;
		try {
			_allFolders = _session.getCachedFolders();
		} catch (StorageAccessException e) {
			throw generateCantDetermineFolderIDException();
		} catch (USMStorageException e) {
			throw generateCantDetermineFolderIDException();
		}
	}

	@SuppressWarnings("unchecked")
	private DataType<PIMAttachments> convertToPIMAttachmentDataType(DataType<?> fieldType) {
		return (DataType<PIMAttachments>) fieldType;
	}

	private USMJSONAPIException generateCantDetermineFolderIDException() throws USMJSONAPIException {
		return new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_CANNOT_DETERMINE_FOLDER_UUID,
				ResponseStatusCode.INTERNAL_ERROR, "Can not determine folder_uuid, list of folders not available");
    }

    private void storeAttachmentsInJSONObject(JSONObject jObj, Object value, DataType<PIMAttachments> fieldType, DataObject dataObject) throws JSONException {
        PIMAttachments attachments = fieldType.checkValue(value);
        JSONArray array = new JSONArray();
        if (attachments == null) // || attachments.size() == 0
            return;
        boolean supportsAttachmentStreaming = USMJSONVersion.supportsAttachmentStreaming(_session);
        for (PIMAttachment att : attachments.getAttachments()) {
            if (supportsAttachmentStreaming && att.getFileSize() > _servlet.getMaxInlineAttachmentSize()) {
                JSONObject object = att.toJSONObject();
                object.put(UUID_KEY, att.getUUID().toString());
                object.put(CommandConstants.TEMPID, ServerTempId.forPIMAttachment(dataObject, att));
                array.put(object);
            } else {
                try {
                    byte[] data = dataObject.getContentType().getTransferHandler().getAttachmentData(dataObject, att.getOxId());
                    if (data != null) {
                        JSONObject object = att.toJSONObject();
                        object.put(UUID_KEY, att.getUUID().toString());
                        object.put(CommandConstants.DATA_KEY, Toolkit.encodeBase64(data));
                        array.put(object);
                    }
                } catch (USMException e) {
                    _servlet.getJournal().warn(
                        _session + " Couldn't read attachment " + att.getOxId() + " of object " + dataObject.getID() + '(' + dataObject.getContentType().getID() + ')',
                        e);
                }
            }
        }
        if (array.length() >= 0)
            jObj.put(ATTACHMENTS, array);
    }

    /**
     * Bug 17240: when the folder belongs to another user, then that user's name must be used as a prefix for the folder title.
     * @throws USMJSONAPIException
     */
    private String generateFolderTitleForClient(Folder folder, Object oxFolderTitleUntyped) throws USMJSONAPIException {
        String oxFolderTitle = (oxFolderTitleUntyped == null) ? "" : oxFolderTitleUntyped.toString();

        String module = folder.getElementsContentTypeID();
		if (!FolderType.SHARED.matches(folder.getFieldContent(TYPE))
				|| FolderContentTypeTransferHandler.MODULE_SYSTEM.equals(module))
            return oxFolderTitle;

        Folder parent = getParentFolder(folder);
        while (parent != null) {
            if (FolderType.SYSTEM.matches(parent.getFieldContent(TYPE))) {
                String folderOwnerName = parent.getFieldContent(TITLE).toString();
                // Bug 18160: prevent double prefixing
                if (oxFolderTitle.startsWith(folderOwnerName))
                    return oxFolderTitle;
                return folderOwnerName + " - " + oxFolderTitle;
            }
            parent = getParentFolder(parent);
        }
        _servlet.getJournal().debug(folder.getSession() + " No user name found for shared folder " + folder.getID());
        return oxFolderTitle;
    }

    private Folder getParentFolder(Folder folder) throws USMJSONAPIException {
        Folder parent = folder.getParentFolder();
        if (parent != null)
            return parent;
        initAllFolders();
        DataObject parentUntyped = DataObjectUtil.findDataObject(folder.getParentFolderID(), _allFolders);
        if (parentUntyped instanceof Folder)
            return (Folder) parentUntyped;
        return null;
    }

    protected void logSyncData(String command, String uuid, long oldSyncID, long newSyncID, long tempSyncID) {
        if (uuid == null)
            uuid = "<hierarchy>";
        if (tempSyncID == 0L || tempSyncID == newSyncID)
            _servlet.getJournal().debug(_session + command + " modified SyncID(" + uuid + "):" + oldSyncID + "->" + newSyncID);
        else
            _servlet.getJournal().debug(
                _session + command + " modified SyncID(" + uuid + "):" + oldSyncID + "->" + newSyncID + "(" + tempSyncID + ")");
    }
    
    protected ResponseObject createCannotLockFolderResponse(String command, String folderType, String folderUUID, String previousAcquirer) {
        String errorMessage = "Can not gain exclusive access to " + folderType + "folder " + folderUUID + " for " + command + ": locked by " + previousAcquirer;
        _servlet.getJournal().warn(_session.toString() + ' ' + errorMessage);
        return new ResponseObject(TEMPORARY_NOT_AVAILABLE, COMMAND_FAILED_TO_ACQUIRE_LOCK_ON_FOLDER, errorMessage, null, null);
    }
    
    protected boolean checkCreatePermissionsAllObjects(int own_rights) {
        return (own_rights & 0x7F) >= 2;
    }

    protected boolean checkDeletePermissionsAllObjects(int own_rights) {
        return (own_rights & 0xFE00000) >= 0x400000;
    }

    protected boolean checkDeletePermissionsOwnObjects(int own_rights) {
        return (own_rights & 0xFE00000) >= 0x200000;
    }

    protected boolean checkDeletePermissions(int own_rights, DataObject object) {
        return isOwnObject(object) ? checkDeletePermissionsOwnObjects(own_rights) : checkDeletePermissionsAllObjects(own_rights);
    }

    protected boolean checkModifyPermissions(int ownRights, DataObject object) {
        return isOwnObject(object) ? checkModifyPermissionsOwnObjects(ownRights) : checkModifyPermissionsAllObjects(ownRights);
    }

    private boolean checkModifyPermissionsAllObjects(int ownRights) {
        return (ownRights & 0x1FC000) >= 0x8000;
    }

    private boolean checkModifyPermissionsOwnObjects(int ownRights) {
        return (ownRights & 0x1FC000) >= 0x4000;
    }

    protected boolean isOwnObject(DataObject object) {
        if (DefaultContentTypes.MAIL_CODE == object.getContentType().getCode())
            return true; // mails can be only own objects
        int userId = object.getSession().getUserIdentifier();
        Object createdBy = object.getFieldContent(CommonConstants.CREATED_BY);
        return (createdBy instanceof String) && createdBy.equals(String.valueOf(userId));
    }
    
    protected int getOwnRights(Folder folder) {
        Object o = folder.getFieldContent(OWN_RIGHTS);
        if (o instanceof Number)
            return ((Number) o).intValue();
        try {
            BitSet requestedFields = new BitSet();
            int fieldIndex = folder.getFieldIndex(OWN_RIGHTS);
            requestedFields.set(fieldIndex);
            folder.getContentType().getTransferHandler().readFolder(folder, requestedFields);
            o = folder.getFieldContent(fieldIndex);
            if (o instanceof Number)
                return ((Number) o).intValue();
        } catch (USMException ignored) {
            // If we could not read the folder, e.g. if the folder has been deleted, we assume that the user has no rights for elements in
            // that folder
        }
        return 0;
    }
}
