/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.OBJECTIDS;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.InvalidUUIDException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.UUIDToolkit;

/**
 * Handler for the USM-locateObjects command.
 * 
 * @author afe
 */

public class LocateObjectsHandler extends NormalCommandHandler {

    private static final String[] REQUIRED_PARAMETERS = { SESSIONID, OBJECTIDS };

    private static final String[] OPTIONAL_PARAMETERS = {};

    private static final UUID FOLDER_HIERARCHY_UUID = new UUID(0L, 0L);

    private static class CachedParentData {

        private long _timestamp;

        private UUID _uuid;

        public CachedParentData() {
            _timestamp = 0L;
            _uuid = null;
        }

        public void update(long timestamp, UUID uuid) {
            if (_timestamp < timestamp) {
                _timestamp = timestamp;
                _uuid = uuid;
            }
        }

        public UUID getUUID() {
            return _uuid;
        }
    }

    public LocateObjectsHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        JSONArray objectUUIDs = getJSONArray(_parameters, OBJECTIDS);
        Map<UUID, CachedParentData> parentMap = new HashMap<UUID, CachedParentData>();
        int length = objectUUIDs.length();
        try {
            for (int i = 0; i < length; i++)
                parentMap.put(UUIDToolkit.extractUUIDFromString(getString(objectUUIDs, i)), new CachedParentData());
            Folder[] folders = _session.getCachedFolders();
            if (folders != null) {
                for (Folder f : folders) {
                    updateInMap(parentMap, f, FOLDER_HIERARCHY_UUID);
                    DataObject[] elements = _session.getCachedFolderElements(f.getID(), f.getElementsContentType());
                    if (elements != null) {
                        for (DataObject e : elements)
                            updateInMap(parentMap, e, f.getUUID());
                    }
                }
            }
            JSONObject response = new JSONObject();
            for (Map.Entry<UUID, CachedParentData> e : parentMap.entrySet()) {
                UUID uuid = e.getKey();
                UUID parent = e.getValue().getUUID();
				response.put(uuid.toString(), (parent == null) ? "not found"
						: (parent == FOLDER_HIERARCHY_UUID) ? "folder" : parent.toString());
            }
            return new ResponseObject(ResponseStatusCode.SUCCESS, response);
        } catch (StorageAccessException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.LOCATE_OBJECTS_DB_ERROR, e);
        } catch (USMStorageException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.LOCATE_OBJECTS_SQL_ERROR, e);
        } catch (JSONException e) {
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.LOCATE_OBJECTS_JSON_ERROR, e);
        } catch (InvalidUUIDException e) {
            throw USMJSONAPIException.createInvalidUUIDException(e);
        }
    }

    private void updateInMap(Map<UUID, CachedParentData> parentMap, DataObject cachedObject, UUID parentUUID) {
        CachedParentData data = parentMap.get(cachedObject.getUUID());
        if (data != null)
            data.update(cachedObject.getTimestamp(), parentUUID);
    }

    @Override
    protected String[] getOptionalParameters() {
        return OPTIONAL_PARAMETERS;
    }

    @Override
    protected String[] getRequiredParameters() {
        return REQUIRED_PARAMETERS;
    }

}
