/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
/*
 * Copyright (C) 2009 MicroDoc GmbH, Munich, Germany
 */

package com.openexchange.usm.contenttypes.manager.impl;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.osgi.framework.BundleContext;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.ContentTypeListener;
import com.openexchange.usm.api.contenttypes.common.ContentTypeManager;
import com.openexchange.usm.api.exceptions.USMIllegalArgumentException;

/**
 * @author afe
 */
public class ContentTypeManagerImpl implements ContentTypeManager {

    private static final ContentType[] EMPTY_CONTENT_TYPE_ARRAY = new ContentType[0];

    // TOOD: do we need listeners??
    private final List<ContentTypeListener> _listeners = new CopyOnWriteArrayList<ContentTypeListener>();

    // private final List<ContentType> _contentTypes = new CopyOnWriteArrayList<ContentType>();
    private static final Log _journal = LogFactory.getLog(ContentTypeManagerImpl.class);

    private ContentTypeServiceTracker _tracker;

    private BundleContext _bundleContext;

    @Override
    public void addContentTypeListener(ContentTypeListener listener) {
        _listeners.add(listener);
    }

    @Override
    public void removeContentTypeListener(ContentTypeListener listener) {
        _listeners.remove(listener);
    }

    @Override
    public ContentType[] getRegisteredContentTypes() {
        if (_tracker == null) {
            _tracker = new ContentTypeServiceTracker(this);
            _tracker.open();
        }
        Object[] services = _tracker.getServices();
        if (services == null)
            return EMPTY_CONTENT_TYPE_ARRAY;
        ContentType[] result = new ContentType[services.length];
        int i = 0;
        for (Object object : services) {
            if (object instanceof ContentType) {
                result[i] = (ContentType) object;
                i++;
            }
        }

        return result;
    }

    public void registerContentType(ContentType type) {
        checkTypeStructure(type);
        synchronized (_tracker) {
            for (ContentType t : getRegisteredContentTypes()) {
                if (t.getID().equals(type.getID())) {
                    String errorMsg = "ContentType with ID '" + t.getID() + "' already exists";
                    _journal.error(errorMsg);
                    throw new USMIllegalArgumentException(USMContentTypeManagerErrorCodes.DUPLICATE_REGISTRATION, errorMsg);
                }
            }
        }
        for (ContentTypeListener listener : _listeners) {
            try {
                listener.addedContentType(type);
            } catch (Exception e) {
                _journal.error("Exception while registering " + type + " in listener " + listener, e);
            }
        }
        _journal.info("ContentType " + type.getID() + " registered");
    }

    public void unregisterContentType(ContentType type) {
        for (ContentTypeListener listener : _listeners) {
            try {
                listener.removedContentType(type);
            } catch (Exception e) {
                _journal.error("Exception while unregistering " + type + " in listener " + listener, e);
            }
        }
        _journal.info("ContentType " + type.getID() + " unregistered");
    }

    public ContentTypeManagerImpl(BundleContext context) {
        _bundleContext = context;
        _journal.info("USM ContentTypeManager activated");
    }

    public void deactivate() {
        _listeners.clear();
        if (_tracker != null)
            _tracker.close();
        _journal.info("USM ContentTypeManager deactivated");
    }

    @Override
    public ContentType getContentType(String id) {
        for (ContentType t : getRegisteredContentTypes()) {
            if (t.getID().equals(id))
                return t;
        }
        return null;
    }

    public BundleContext getBundleContext() {
        return _bundleContext;
    }

    private void checkTypeStructure(ContentType type) {
        ContentTypeField[] fields = type.getFields();
        if (fields.length < 2)
            throw new USMIllegalArgumentException(
                USMContentTypeManagerErrorCodes.INVALID_CONTENT_TYPE_NUMBER1,
                "Not enough fields in " + type.getID() + ", at least ID and parent folder ID are required");
        for (int i = 0; i < fields.length; i++) {
            if ((i == 0) != fields[i].isIDField())
                throw new USMIllegalArgumentException(
                    USMContentTypeManagerErrorCodes.INVALID_CONTENT_TYPE_NUMBER2,
                    ((i == 0) ? "Field 0 is no ID field" : " Extra ID field(s)") + " in ContentType " + type.getID());
            if ((i == 1) != fields[i].isParentFolderIDField())
                throw new USMIllegalArgumentException(
                    USMContentTypeManagerErrorCodes.INVALID_CONTENT_TYPE_NUMBER3,
                    ((i == 1) ? "Field 1 is no parent folder ID field" : " Extra parent folder ID field(s)") + " in ContentType " + type.getID());
        }
    }
}
