/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.filemanagement.ManagedFileFilter;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.usm.api.configuration.ConfigurationProperties;
import com.openexchange.usm.api.session.Session;

/**
 * {@link TempFileStorage} Storage facility that provides session-based access to temp-files (creating, accessing, delete all of a Session)
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class TempFileStorage {

    private static class PrefixManagedFileFilter implements ManagedFileFilter {

        private final String _prefix;

        public PrefixManagedFileFilter(String prefix) {
            _prefix = prefix;
        }

        @Override
        public boolean accept(ManagedFile managedFile) {
            return managedFile.getID().startsWith(_prefix);
        }
    }

    private static final InputStream _EMPTY_INPUT_STREAM = new InputStream() {

        @Override
        public int read() {
            return -1;
        }
    };

    private static int _tempFileTimeout = ConfigurationProperties.TEMP_FILES_TIMEOUT_DEFAULT;
    
    private static ManagedFileManagement _fileManagement;

    private static volatile TempFileStorage _localFileStorage;

    public static void setManagedFileManagement(ManagedFileManagement fileManagement) {
        _fileManagement = fileManagement;
    }

    public static void setTempFileTimeout(int timeout) {
        _tempFileTimeout = timeout;
    }
    
    public static int getTempFileTimeout() {
        return _tempFileTimeout;
    }

    /**
     * If available, accesses the ManagedFileManagement to get a temporary file for write operations, and assigns it an USM-session specific
     * ID. Any caller must make sure that the returned TempFile is closed eventually. If no ManagedFileManagement is registered, uses a
     * simple backup mechanism that directly creates Java temp-files.
     * 
     * @param session USMSession for which the TempFile is created / accessed
     * @param tempid unique tempid as specified by the client
     * @param offset byte offset at which write operations will start (must be >= 0)
     * @return TempFile to which data can be written
     * @throws IOException If an I/O error occurs during opening of or seeking in the file
     */
    public static TempFile getTempFileForWrite(Session session, String tempid, long offset) throws IOException {
        String id = getTempFileID(session, tempid);
        ManagedFileManagement fileManagement = _fileManagement;
        if (fileManagement != null) {
            ManagedFile file = retrieveFileFromManagement(fileManagement, id);
            return new TempFile(file, offset);
        }
        return getLocalFileStorage().getTempFileForWrite(id, offset);
    }

    /**
     * If available, accesses the ManagedFileManagement to get an already existing file for read operations. Any caller must make sure that
     * the returned TempFile is closed eventually. If no ManagedFileManagement is registered, uses a simple backup mechanism that uses
     * directly created Java temp-files.
     * 
     * @param session
     * @param tempid
     * @return
     * @throws IOException
     */
    public static TempFile getTempFileForRead(Session session, String tempid) throws IOException {
        String id = getTempFileID(session, tempid);
        ManagedFileManagement fileManagement = _fileManagement;
        if (fileManagement != null) {
            try {
                ManagedFile file = fileManagement.getByID(id);
                return new TempFile(file);
            } catch (OXException e) {
                throw new IOException("File not found for tempid '" + tempid + '\'', e);
            }
        }
        return getLocalFileStorage().getTempFileForRead(tempid, id);
    }

    /**
     * Deleted all temp files associated with a given USM session. Returns the number of temp files that have been deleted.
     * 
     * @param session
     * @return number of temp files that have been deleted
     */
    public static int deleteTempFiles(Session session) {
        String prefix = getTempFilePrefix(session);
        ManagedFileManagement fileManagement = _fileManagement;
        if (fileManagement != null) {
            List<ManagedFile> filesToDelete = getManagedFilesToDelete(fileManagement, prefix);
            for (ManagedFile file : filesToDelete)
                file.delete();
            return filesToDelete.size();
        }
        return getLocalFileStorage().deleteTempFiles(prefix);
    }

    /**
     * Tries to delete the temp file with the given client tempid for the specified USM session.
     * 
     * @param session
     * @param tempid
     * @return true if file was deleted, false if file no longer exists or if file could not be deleted
     * @throws IOException If an OXException was thrown by the OX ManagedFileManagement
     */
    public static boolean deleteTempFile(Session session, String tempid) throws IOException {
        String id = getTempFileID(session, tempid);
        ManagedFileManagement fileManagement = _fileManagement;
        if (fileManagement != null) {
            if (!fileManagement.contains(id))
                return false;
            try {
                fileManagement.removeByID(id);
                return true;
            } catch (OXException e) {
                throw new IOException("Could not delete temp file " + tempid, e);
            }
        }
        return getLocalFileStorage().deleteTempFile(id);
    }

    private static List<ManagedFile> getManagedFilesToDelete(ManagedFileManagement fileManagement, String prefix) {
        try {
            return fileManagement.getManagedFiles(new PrefixManagedFileFilter(prefix));
        } catch (OXException e) {
            return Collections.emptyList();
        }
    }

    private static ManagedFile retrieveFileFromManagement(ManagedFileManagement fileManagement, String id) throws IOException {
        try {
            if (fileManagement.containsLocal(id))
                return fileManagement.getByID(id);
            return fileManagement.createManagedFile(id, _EMPTY_INPUT_STREAM, _tempFileTimeout);
        } catch (OXException e) {
            throw new IOException("Error accessing temp file " + id, e);
        }
    }

    private static String getTempFileID(Session session, String tempId) {
        return getTempFilePrefix(session) + tempId;
    }

    private static String getTempFilePrefix(Session session) {
        return "USM_" + session + '_';
    }

    private static TempFileStorage getLocalFileStorage() {
        if (_localFileStorage == null) {
            synchronized (_EMPTY_INPUT_STREAM) {
                if (_localFileStorage == null)
                    _localFileStorage = new TempFileStorage();
            }
        }
        return _localFileStorage;
    }

    private final Map<String, File> _localTempFiles = new ConcurrentHashMap<String, File>(16, 0.75f, 1);

    private TempFileStorage() {
    }

    private TempFile getTempFileForWrite(String id, long offset) throws IOException {
        File f = _localTempFiles.get(id);
        if (f == null) {
            f = File.createTempFile("USM_TempFile", ".temp");
            f.deleteOnExit();
            _localTempFiles.put(id, f);
        }
        return new TempFile(f, offset);
    }

    private TempFile getTempFileForRead(String localId, String completeId) throws IOException {
        File f = _localTempFiles.get(completeId);
        if (f == null)
            throw new IOException("File not found for tempid '" + localId + '\'');
        return new TempFile(f);
    }

    private int deleteTempFiles(String prefix) {
        int count = 0;
        for (Iterator<Map.Entry<String, File>> i = _localTempFiles.entrySet().iterator(); i.hasNext();) {
            Entry<String, File> entry = i.next();
            if (entry.getKey().startsWith(prefix)) {
                if (entry.getValue().delete())
                    count++;
                i.remove();
            }
        }
        return count;
    }

    private boolean deleteTempFile(String id) {
        File f = _localTempFiles.remove(id);
        return f != null && f.delete();
    }
}
