/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.util;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.usm.api.io.InputStreamProvider;

/**
 * {@link TempFile} wrapper that updates associated ManagedFile on every write to a temporary file.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class TempFile implements Closeable, InputStreamProvider {

    private final ManagedFile _managedFile;

    private final RandomAccessFile _randomAccessFile;

    public TempFile(ManagedFile managedFile, long offset) throws IOException {
        _managedFile = managedFile;
        File f = managedFile.getFile();
        if (f == null)
            throw new IOException("Temp file no longer available");
        _randomAccessFile = new RandomAccessFile(f, "rw");
        seek(offset);
    }

    public TempFile(ManagedFile managedFile) {
        _managedFile = managedFile;
        _randomAccessFile = null;
    }

    public TempFile(File file, long offset) throws IOException {
        _managedFile = null;
        _randomAccessFile = new RandomAccessFile(file, "rw");
        seek(offset);
    }

    public TempFile(File file) throws FileNotFoundException {
        _managedFile = null;
        _randomAccessFile = new RandomAccessFile(file, "r");
    }

    private void seek(long offset) throws IOException {
        try {
            _randomAccessFile.seek(offset);
        } catch (IOException e) {
            Toolkit.close(_randomAccessFile);
            throw e;
        }
    }

    public void write(byte[] data) throws IOException {
        if (_randomAccessFile == null)
            throw new IOException("Can not access TempFile: Not open for write access");
        _randomAccessFile.write(data);
        if (_managedFile != null)
            _managedFile.touch();
    }

    @Override
    public void close() throws IOException {
        if (_randomAccessFile != null) {
            try {
                if (_managedFile != null)
                    _managedFile.setSize(_randomAccessFile.length());
            } finally {
                _randomAccessFile.close();
            }
        }
    }

    public long getSize() throws IOException {
        if (_managedFile != null)
            return _managedFile.getSize();
        if (_randomAccessFile != null)
            return _randomAccessFile.length();
        return 0L;
    }

    @Override
    public InputStream getInputStream() throws IOException {
        if (_managedFile != null) {
            try {
                return _managedFile.getInputStream();
            } catch (OXException e) {
                throw new IOException("Can't access InputStream of ManagedFile", e);
            }
        }
        if (_randomAccessFile == null)
            throw new IOException("File is not accessible");
        _randomAccessFile.seek(0L);
        return new FileInputStream(_randomAccessFile.getFD());
    }
}
