/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.contenttypes.util;

import java.util.Arrays;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;

/**
 * {@link OXObjectIDList} List of OX Object IDs and their timestamps as parsed from a JSONArray result, sorted by OX object ID (numerically or lexically)
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class OXObjectIDList {

    private static class OXTimestamp implements Comparable<OXTimestamp> {

        private final static int parseIdNumber(String id) {
            try {
                return Integer.parseInt(id);
            } catch (NumberFormatException e) {
                return 0;
            }
        }

        public final String _id;

        public final int _idNumber;

        public final long _lastModified;

        public OXTimestamp(String id, long lastModified) {
            _id = id;
            _lastModified = lastModified;
            _idNumber = parseIdNumber(_id);
        }

        @Override
        public int compareTo(OXTimestamp o) {
            if (_idNumber == 0 || o._idNumber == 0)
                return _id.compareTo(o._id);
            return _idNumber - o._idNumber;
        }
    }

    private final static OXTimestamp[] _EMPTY_ARRAY = {};

    private final long _timestamp;

    private final OXTimestamp[] _data;

    public OXObjectIDList(long defaultTimestamp, JSONObject oxObjectIDdata) throws JSONException {
        _timestamp = oxObjectIDdata.optLong(CommonConstants.RESULT_TIMESTAMP, defaultTimestamp);
        JSONArray jsonar = oxObjectIDdata.getJSONArray(CommonConstants.RESULT_DATA);
        int length = jsonar.length();
        if (length == 0) {
            _data = _EMPTY_ARRAY;
        } else {
            _data = new OXTimestamp[length];
            for (int i = 0; i < length; i++) {
                JSONArray columnsArray = jsonar.getJSONArray(i);
                _data[i] = new OXTimestamp(columnsArray.getString(0), columnsArray.getLong(1));
            }
        }
        Arrays.sort(_data); // Make sure that data is sorted by id so that series will be before any exceptions of the series
    }

    public long getTimestamp() {
        return _timestamp;
    }

    public int size() {
        return _data.length;
    }

    public String getID(int i) {
        return _data[i]._id;
    }

    public long getLastModified(int i) {
        return _data[i]._lastModified;
    }
}
