/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.contenttypes.attachments;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;
import com.openexchange.usm.api.contenttypes.resource.ResourceInputStream;
import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.datatypes.PIMAttachments;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.PIMAttachmentCountLimitExceededException;
import com.openexchange.usm.api.exceptions.PIMAttachmentSizeLimitExceededException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.ox.json.JSONResult;
import com.openexchange.usm.api.ox.json.JSONResultType;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.api.ox.json.OXResource;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.util.TempFile;
import com.openexchange.usm.util.TempFileStorage;
import com.openexchange.usm.util.Toolkit;

/**
 * Handler for the interaction with the OX server for PIM Attachments.
 * 
 * @author ldo
 */
public class PIMAttachmentTransferHandler {

    private static final PIMAttachments UNSET_PIMATTACHMENTS = new PIMAttachments(0, true, new PIMAttachment[0]);

    private static final String APPLICATION_OCTET_STREAM = "application/octet-stream";

    private static final String CONTENT_TYPE = "content_type";

    private static final String FILE_PART_NAME = "file_0";

    private static final String JSON_PART_NAME = "json_0";

    private static final String ACTION_DOCUMENT = "document";

    private static final String ACTION_DETACH = "detach";

    private static final String ACTION_ATTACH = "attach";

    private static final String OX_AJAX_PATH = "attachment";

    private static String createColumnsParameter() {
        StringBuilder sb = new StringBuilder(100);
        for (Integer fieldId : PIMAttachment.FIELDS.keySet()) {
            sb.append(fieldId);
            sb.append(',');
        }
        return sb.toString();
    }

    private static final String ATTACHMENT_COLUMNS = createColumnsParameter();

    private static final Log LOG = LogFactory.getLog(PIMAttachmentTransferHandler.class);

    private final OXJSONAccess _ajaxAccess;

    public PIMAttachmentTransferHandler(OXJSONAccess ajaxAccess) {
        _ajaxAccess = ajaxAccess;
    }

    public long createNewAttachment(DataObject object, PIMAttachment attachment) throws USMException {
        if (attachment.getData() != null)
            return createNewAttachmentFromData(object, attachment);
        if (attachment.getTempId() != null)
            return createNewAttachmentFromLocalTempfile(object, attachment);
        throw new USMException(
            PIMAttachmentErrorCodes.NO_DATA_PROVIDED_FOR_ATTACHMENT,
            "No data provided for new attachment '" + attachment + '\'');
    }

    private long createNewAttachmentFromData(DataObject object, PIMAttachment attachment) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Creating attachment " + attachment + " from data for object" + object.getID());
        checkAttachmentSizeLimit(object, attachment.getData().length);
        JSONObject requestBody = buildCreateAttachmentJSONRequestBody(object, attachment);
        JSONResult result = _ajaxAccess.storeResource(
            OX_AJAX_PATH,
            ACTION_ATTACH,
            object.getSession(),
            new HashMap<String, String>(),
            requestBody,
            attachment.getData(),
            attachment.getMimeType(),
            JSON_PART_NAME,
            FILE_PART_NAME);
        return processCreateAttachmentResponse(object, attachment, result);
    }

    private long createNewAttachmentFromLocalTempfile(DataObject object, PIMAttachment attachment) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Creating attachment " + attachment + " from tempfile for object" + object.getID());
        TempFile tempFile = null;
        try {
            tempFile = TempFileStorage.getTempFileForRead(object.getSession(), attachment.getTempId());
            long fileSize = tempFile.getSize();
            checkAttachmentSizeLimit(object, fileSize);
            JSONObject requestBody = buildCreateAttachmentJSONRequestBody(object, attachment);
            JSONResult result = _ajaxAccess.storeResourceFromStream(
                OX_AJAX_PATH,
                ACTION_ATTACH,
                object.getSession(),
                new HashMap<String, String>(),
                requestBody,
                attachment.getMimeType(),
                JSON_PART_NAME,
                FILE_PART_NAME,
                fileSize,
                tempFile);
            return processCreateAttachmentResponse(object, attachment, result);
        } catch (IOException e) {
            throw new USMException(
                PIMAttachmentErrorCodes.CAN_NOT_ACCESS_TEMP_FILE_FOR_ATTACHMENT_DATA,
                "Can't access temp file for creating attachment with tempid '" + attachment.getTempId() + '\'',
                e);
        } finally {
            Toolkit.close(tempFile);
        }
    }

    private void checkAttachmentSizeLimit(DataObject object, long attachmentSize) throws USMException {
        PIMAttachments completeList = getCompleteAttachmentsList(object, null, null);
        long sizeLimit = object.getSession().getPIMAttachmentsSizeLimit();
        long currentSize = attachmentSize;
        if (completeList != null) {
            int countLimit = object.getSession().getPIMAttachmentsCountLimit();
            if (countLimit >= 0 && completeList.size() >= countLimit)
                throw new PIMAttachmentCountLimitExceededException(
                    PIMAttachmentErrorCodes.PIM_ATTACHMENT_COUNT_LIMIT_EXCEEDED,
                    "Number of attachments would exceed limit of " + countLimit);
            for (PIMAttachment a : completeList.getAttachments())
                currentSize += a.getFileSize();
        }
        if (sizeLimit >= 0 && currentSize > sizeLimit)
            throw new PIMAttachmentSizeLimitExceededException(
                PIMAttachmentErrorCodes.PIM_ATTACHMENT_SIZE_LIMIT_EXCEEDED,
                "Overall size of all attachments would exceed limit of " + sizeLimit);
    }

    private static JSONObject buildCreateAttachmentJSONRequestBody(DataObject object, PIMAttachment attachment) throws USMException {
        JSONObject jsonAtt = attachment.toJSONObject();
        String[] names = PIMAttachment.FIELDS.values().toArray(new String[PIMAttachment.FIELDS.size()]);
        JSONObject requestBody;
        try {
            requestBody = new JSONObject(jsonAtt, names);
            requestBody.put(CommonConstants.ATTACHED, object.getID());
            requestBody.put(CommonConstants.FOLDER, object.getParentFolderID());
            requestBody.put(CommonConstants.MODULE, object.getContentType().getCode());
        } catch (JSONException e1) {
            throw new USMException(
                PIMAttachmentErrorCodes.CAN_NOT_CREATE_REQUEST_BODY,
                "Create New Attachment: Can not create request body.");
        }
        return requestBody;
    }

    private static long processCreateAttachmentResponse(DataObject object, PIMAttachment attachment, JSONResult result) throws OXCommunicationException, USMException {
        JSONArray array = extractArrayResult(result);
        try {
            attachment.setOxId(array.getInt(0));
        } catch (JSONException e) {
            throw new OXCommunicationException(
                PIMAttachmentErrorCodes.ID_NOT_EXISTING,
                "The OX Server did not return the id of the new attachment",
                result.toString());
        }
        long timestamp = getLastModifiedOfNewestAttachment(object);
        return timestamp;
    }

    public PIMAttachments getAllAttachments(DataObject object) throws USMException {
        if (object.getID() == null) // this means that the object was created by client and still doesn't have ox id
            return UNSET_PIMATTACHMENTS;
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Read all attachments for object" + object.getID());
        PIMAttachments completeList = getCompleteAttachmentsList(object, null, null);
        if (completeList == null || completeList.size() == 0)
            return completeList;
        int countLimit = object.getSession().getPIMAttachmentsCountLimit();
        int length = (countLimit >= 0 && completeList.size() > countLimit) ? countLimit : completeList.size();
        if (length == 0)
            return null;
        long availableSize = object.getSession().getPIMAttachmentsSizeLimit();
        List<PIMAttachment> visibleAttachments = new ArrayList<PIMAttachment>(length);
        for (int i = 0; i < length; i++) {
            PIMAttachment a = completeList.getAttachments()[i];
            if (availableSize >= 0) {
                availableSize -= a.getFileSize();
                if (availableSize < 0)
                    break;
            }
            visibleAttachments.add(a);
        }
        PIMAttachment[] attachments = visibleAttachments.toArray(new PIMAttachment[visibleAttachments.size()]);
        setSavedUUIDs(object, attachments);
        return new PIMAttachments(completeList.getTimestamp(), true, attachments);
    }

    private PIMAttachments getCompleteAttachmentsList(DataObject object, String sort, String order) throws USMException {
        if (object.getID() == null) // this means that the object was created by client and still doesn't have ox id
            return UNSET_PIMATTACHMENTS;
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.ATTACHED, object.getID());
        parameters.put(CommonConstants.FOLDER, object.getParentFolderID());
        parameters.put(CommonConstants.MODULE, String.valueOf(object.getContentType().getCode()));
        parameters.put(CommonConstants.COLUMNS, ATTACHMENT_COLUMNS);
        parameters.put(CommonConstants.SORT, (sort != null) ? sort : "1");
        parameters.put(CommonConstants.ORDER, (order != null) ? order : "asc");

        JSONResult result = _ajaxAccess.doGet(OX_AJAX_PATH, CommonConstants.ACTION_ALL, object.getSession(), parameters);
        JSONArray array = extractArrayResult(result);
        int length = array.length();
        if (length == 0)
            return null;
        PIMAttachment[] attachments = new PIMAttachment[length];
        try {
            for (int i = 0; i < length; i++) {
                JSONArray attachmentFields = array.getJSONArray(i);
                attachments[i] = new PIMAttachment(UUID.randomUUID(), attachmentFields);
            }
            long timestamp = extractTimestamp(result);
            return new PIMAttachments(timestamp, true, attachments);
        } catch (JSONException e) {
            throw new OXCommunicationException(
                PIMAttachmentErrorCodes.INVALID_OX_RESULT_1,
                "PIM attachments list reported from OX in unexpected format",
                result.toString());
        }
    }

    private static void setSavedUUIDs(DataObject object, PIMAttachment[] pimAtts) {
        PIMAttachments oldAttachments = (PIMAttachments) object.getFieldContent(CommonConstants.ATTACHMENTS_LAST_MODIFIED);
        if (oldAttachments == null)
            return;
        for (PIMAttachment newAtt : pimAtts) {
            for (PIMAttachment oldAtt : oldAttachments.getAttachments()) {
                if (newAtt.getOxId() == oldAtt.getOxId()) {
                    newAtt.setUUID(oldAtt.getUUID());
                    break;
                }
            }
        }
    }

    private static JSONArray extractArrayResult(JSONResult result) throws OXCommunicationException {
        if (result.getResultType() == JSONResultType.Error) {
            throw new OXCommunicationException(
                PIMAttachmentErrorCodes.ERROR_ON_STORE_RESOURCE_1,
                "OX server returned error",
                result.getJSONObject());
        }
        if (result.getResultType() != JSONResultType.JSONObject)
            throw new OXCommunicationException(
                PIMAttachmentErrorCodes.INVALID_OX_RESULT_2,
                "OX server didn't send expected JSONObject",
                result.toString());
        try {
            return result.getJSONObject().getJSONArray(CommonConstants.RESULT_DATA);
        } catch (JSONException e) {
            throw new OXCommunicationException(
                PIMAttachmentErrorCodes.INVALID_OX_RESULT_3,
                "OX server did not send data array",
                e,
                result.toString());
        }
    }

    private static long extractTimestamp(JSONResult result) throws OXCommunicationException {
        if (result.getResultType() == JSONResultType.Error) {
            throw new OXCommunicationException(
                PIMAttachmentErrorCodes.ERROR_ON_STORE_RESOURCE_2,
                "OX server returned error",
                result.getJSONObject());
        }
        if (result.getResultType() != JSONResultType.JSONObject)
            throw new OXCommunicationException(
                PIMAttachmentErrorCodes.INVALID_OX_RESULT_5,
                "OX server didn't send expected JSONObject",
                result.toString());
        try {
            return result.getJSONObject().getLong(CommonConstants.TIMESTAMP);
        } catch (JSONException e) {
            throw new OXCommunicationException(
                PIMAttachmentErrorCodes.INVALID_OX_RESULT_6,
                "OX server didn't send timestamp",
                e,
                result.toString());
        }
    }

    public long deleteAttachments(DataObject object, PIMAttachment[] attachmentsToDelete) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Delete attachments for object" + object.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.ATTACHED, object.getID());
        parameters.put(CommonConstants.FOLDER, object.getParentFolderID());
        parameters.put(CommonConstants.MODULE, String.valueOf(object.getContentType().getCode()));
        JSONArray requestBody = new JSONArray();
        for (int i = 0; i < attachmentsToDelete.length; i++) {
            requestBody.put(attachmentsToDelete[i].getOxId());
        }
        JSONResult result = _ajaxAccess.doPut(OX_AJAX_PATH, ACTION_DETACH, object.getSession(), parameters, requestBody);
        if (result.getResultType() == JSONResultType.Error) {
            throw new OXCommunicationException(
                PIMAttachmentErrorCodes.INVALID_OX_RESULT_4,
                "OX returned error on deleting attachments",
                result.getJSONObject());
        }
        // long timestamp = extractTimestamp(result);
        // TODO: remove this call when OX changes the response in create attachment!!!
        long timestamp = getLastModifiedOfNewestAttachment(object);
        return timestamp;
    }

    private static long getLastModifiedOfNewestAttachment(DataObject object) throws USMException {
        DataObject objectCopy = object.createCopy(false);
        BitSet fields = new BitSet();
        fields.set(object.getFieldIndex(CommonConstants.ATTACHMENTS_LAST_MODIFIED));
        objectCopy.getContentType().getTransferHandler().readDataObject(objectCopy, fields);
        PIMAttachments attachments = (PIMAttachments) objectCopy.getFieldContent(CommonConstants.ATTACHMENTS_LAST_MODIFIED);
        long timestamp = (attachments == null) ? 0 : attachments.getTimestamp();
        return timestamp;
    }

    public byte[] getAttachmentData(DataObject object, int attachmentId) throws USMException {
        Map<String, String> parameters = createAttachmentParameters(object, attachmentId);
        OXResource result = _ajaxAccess.getResource(object.getSession(), OX_AJAX_PATH, parameters);
        return result.getData();
    }

    public ResourceInputStream getAttachmentDataStream(DataObject object, int attachmentId, long offset, long length) throws USMException {
        Map<String, String> parameters = createAttachmentParameters(object, attachmentId);
        if (offset > 0L)
            parameters.put(CommonConstants.OFF, Long.toString(offset));
        if (length >= 0L && length != Long.MAX_VALUE)
            parameters.put(CommonConstants.LEN, Long.toString(length));
        return _ajaxAccess.getResourceAsStream(object.getSession(), OX_AJAX_PATH, parameters);
    }

    private static Map<String, String> createAttachmentParameters(DataObject object, int attachmentId) {
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.FOLDER, object.getParentFolderID());
        parameters.put(CommonConstants.ATTACHED, object.getID());
        parameters.put(CommonConstants.MODULE, String.valueOf(object.getContentType().getCode()));
        parameters.put(CommonConstants.ID, String.valueOf(attachmentId));
        parameters.put(CommonConstants.ACTION, ACTION_DOCUMENT);
        parameters.put(CONTENT_TYPE, APPLICATION_OCTET_STREAM);
        return parameters;
    }
}
