/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.contenttypes.mail.impl;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.OXFolderContent;
import com.openexchange.usm.api.contenttypes.mail.MailConstants;
import com.openexchange.usm.api.contenttypes.mail.MailContentType;
import com.openexchange.usm.api.contenttypes.resource.ResourceInputStream;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMIllegalArgumentException;
import com.openexchange.usm.api.exceptions.USMStartupException;
import com.openexchange.usm.api.exceptions.UnsupportedContentOperationException;
import com.openexchange.usm.api.io.InputStreamProvider;
import com.openexchange.usm.api.ox.json.JSONResult;
import com.openexchange.usm.api.ox.json.JSONResultType;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.api.ox.json.OXResource;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.contenttypes.util.AbstractTransferHandler;
import com.openexchange.usm.contenttypes.util.USMContentTypesUtilErrorCodes;
import com.openexchange.usm.datatypes.mail.Vacation;
import com.openexchange.usm.util.Toolkit;

public class MailContentTypeTransferHandlerImpl extends AbstractTransferHandler {

    private static final String ATTACH_SRC = "attach_src";

    private static final String _RECEIVED_DATE_COLUMN = "610";

    private static final String DROP_PREFIX = "dropPrefix";

    private static final String COM_OPENEXCHANGE_MAIL_CONVERSION_SEQUENCEID = "com.openexchange.mail.conversion.sequenceid";

    private static final String COM_OPENEXCHANGE_MAIL_CONVERSION_MAILID = "com.openexchange.mail.conversion.mailid";

    private static final String COM_OPENEXCHANGE_MAIL_CONVERSION_FULLNAME = "com.openexchange.mail.conversion.fullname";

    private static final String COM_OPENEXCHANGE_MAIL_ICAL = "com.openexchange.mail.ical";

    private static final String FORCE = "force";

    private static final String FOLDER = "folder";

    private static final String ONE = "1";

    private static final String HARDDELETE = "harddelete";

    private static final String RAW = "raw";

    private static final String VIEW = "view";

    private static final String FLAGS = "flags";

    private static final String UNSEEN = "unseen";

    private static final String SRC = "src";

    private static final String SAVE = "save";

    private static final String FLAG = "flag";

    private static final String MAILFILTER_AJAX_PATH = "mailfilter";

    private final BitSet _getAllBitSet = new BitSet();

    private final int _idIndex;

    private final int _receivedDateIndex;
    
    private final int _sourceFieldIndex;

    public MailContentTypeTransferHandlerImpl(MailContentType contentType, OXJSONAccess ajaxAccess) {
        super(contentType, ajaxAccess);
        _idIndex = getFieldIndex(MailConstants.FIELD_ID);
        _receivedDateIndex = getFieldIndex(MailConstants.RECEIVED_DATE);
        _sourceFieldIndex = getFieldIndex(MailConstants.SOURCE);
        _getAllBitSet.set(_idIndex);
        _getAllBitSet.set(_receivedDateIndex);
    }

    private int getFieldIndex(String fieldId) {
        ContentTypeField[] fields = _contentType.getFields();
        for (int i = 0; i < fields.length; i++) {
            if (fields[i].getFieldName().equals(fieldId))
                return i;
        }
        throw new USMStartupException(
            USMContentTypesMailErrorCodes.REQUIRED_FIELDS_MISSING_ERROR,
            "Required field is missing from ContentType: " + fieldId);
    }

    public void sendMail(String mail, Session session) throws USMException {
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(SRC, ONE);
        JSONResult result = _ajaxAccess.doPut(CommonConstants.MAIL_PATH, CommonConstants.ACTION_NEW, session, parameters, mail);
        checkResult(result, JSONResultType.JSONObject);
    }

    public void sendMail(InputStreamProvider streamProvider, Session session) throws USMException {
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(SRC, ONE);
        JSONResult result = _ajaxAccess.doPut(CommonConstants.MAIL_PATH, CommonConstants.ACTION_NEW, session, parameters, streamProvider);
        checkResult(result, JSONResultType.JSONObject);
    }

    public Object getMailAttachment(Session session, String folderId, String mailId, String attachmentId) throws USMException {
        Map<String, String> parameters = createMailAttachmentParameters(folderId, mailId, attachmentId);
        OXResource result = _ajaxAccess.getResource(session, CommonConstants.MAIL_PATH, parameters);
        return result;
    }

    public ResourceInputStream getMailAttachmentAsStream(Session session, String folderId, String mailId, String attachmentId) throws USMException {
        Map<String, String> parameters = createMailAttachmentParameters(folderId, mailId, attachmentId);
        ResourceInputStream result = _ajaxAccess.getResourceAsStream(session, CommonConstants.MAIL_PATH, parameters);
        return result;
    }

    private static Map<String, String> createMailAttachmentParameters(String folderId, String mailId, String attachmentId) {
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.FOLDER, folderId);
        parameters.put(CommonConstants.ID, mailId);
        parameters.put(MailConstants.ATTACHMENT, attachmentId);
        parameters.put(SAVE, ONE);
        parameters.put(CommonConstants.ACTION, MailConstants.ATTACHMENT);
        return parameters;
    }

    public String createNewMail(String folderId, int flags, String mailMIMEdataBlock, Session session) throws USMException {
        // TODO: use import instead of new (see svn history!)
        Map<String, String> parameters = buildNewMailParameters(folderId, flags);
        JSONResult result = _ajaxAccess.doPut(CommonConstants.MAIL_PATH, CommonConstants.ACTION_NEW, session, parameters, mailMIMEdataBlock);
        return verifyNewMailResult(result);
    }
    
    public String createNewMailFromStream(String folderId, int flags, InputStreamProvider streamProvider, Session session) throws USMException {
        Map<String, String> parameters = buildNewMailParameters(folderId, flags);
        JSONResult result = _ajaxAccess.doPut(CommonConstants.MAIL_PATH, CommonConstants.ACTION_NEW, session, parameters, streamProvider);
        return verifyNewMailResult(result);
    }

    private String verifyNewMailResult(JSONResult result) throws OXCommunicationException, InternalUSMException {
        checkResult(result, JSONResultType.JSONObject);
        JSONObject o = result.getJSONObject();
        try {
            return o.getJSONObject(CommonConstants.RESULT_DATA).getString(CommonConstants.RESULT_ID);
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER4,
                null,
                "Missing result ID on new " + _contentType.getID(),
                e);
        }
    }

    private static Map<String, String> buildNewMailParameters(String folderId, int flag) {
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(SRC, ONE);
        parameters.put(CommonConstants.FOLDER, folderId);
        parameters.put(FLAGS, String.valueOf(flag));
        parameters.put(FORCE, ONE);
        return parameters;
    }

    public DataObject getReplyForward(Session session, String messageID, String folderID, String action, String view, boolean dropPrefix) throws USMException {
        DataObject mail = _contentType.newDataObject(session);
        mail.setID(messageID);
        mail.setParentFolderID(folderID);
        mail.commitChanges();
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.ID, messageID);
        parameters.put(CommonConstants.FOLDER, folderID);
        parameters.put(UNSEEN, ONE);
        parameters.put(VIEW, view);
        parameters.put(DROP_PREFIX, String.valueOf(dropPrefix));

        JSONResult result = _ajaxAccess.doGet(CommonConstants.MAIL_PATH, action, session, parameters);
        if (result.getResultType() == JSONResultType.Error)
            throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER2, result.getJSONObject());
        BitSet reqFields = createRequestedFields(_contentType);
        reqFields.set(18); // attachments
        reqFields.set(17); // headers
        updateDataObjectFromJSONResult(mail, reqFields, result);
        mail.setID(messageID);
        mail.setParentFolderID(folderID);
        return mail;
    }

    private BitSet createRequestedFields(ContentType type) {
        ContentTypeField[] fields = type.getFields();
        BitSet requestedFields = new BitSet();
        for (int i = 0; i < fields.length; i++) {
            if (fields[i].getFieldID() >= 0) {
                requestedFields.set(i);
            }
        }
        return requestedFields;
    }

    @Override
    public void writeDeletedDataObject(DataObject object) throws USMException {
        writeDeletedDataObject(object, Boolean.TRUE.equals(object.getSession().getCustomProperty(Session.MAIL_HARDDELETE_PROPERTY)));
    }

    private void writeDeletedDataObject(DataObject object, boolean hardDelete) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Delete mail " + object.getID());
        checkTypeToWrite(object, DefaultContentTypes.MAIL_ID);
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.TIMESTAMP, String.valueOf(object.getTimestamp()));
        if (hardDelete)
            parameters.put(HARDDELETE, ONE);
        JSONObject jObj = new JSONObject();
        JSONArray array = new JSONArray();
        try {
            jObj.put(CommonConstants.ID, object.getID());
            jObj.put(CommonConstants.FOLDER, object.getParentFolderID());

            array.put(jObj);
        } catch (JSONException e) {
            throw new USMIllegalArgumentException(
                USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER2,
                "Duplicate or null entry in provided key/value pairs");
        }
        JSONResult result = _ajaxAccess.doPut(CommonConstants.MAIL_PATH, CommonConstants.ACTION_DELETE, object.getSession(), parameters, array);
        checkForNotDeletedObjects(object, result);
    }

    @Override
    public DataObject[] readUpdatedFolderContent(Folder folder, BitSet requestedFields, long timestamp) throws USMException {
        Map<String, String> parameters = new HashMap<String, String>();
        // XXX Not IMAP - Response: Just an empty JSON array is going to be returned since this action cannot be applied to IMAP.

        // parameters.put(UtilConstants.FOLDER, folder.getID());
        // parameters.put(UtilConstants.COLUMNS, createColumnsParameter(folder, requestedFields));
        // parameters.put(UtilConstants.TIMESTAMP, String.valueOf(timestamp));

        if (LOG.isDebugEnabled())
            LOG.debug(folder.getSession() + " Read updates in " + folder.getID());
        JSONResult result = _ajaxAccess.doGet(CommonConstants.MAIL_PATH, CommonConstants.ACTION_UPDATES, folder.getSession(), parameters);
        checkResult(result, JSONResultType.JSONObject);
        return getListOfUpdates(folder, requestedFields, result);
    }

    @Override
    public void writeNewDataObject(DataObject object) throws USMException {
        if (LOG.isErrorEnabled())
            LOG.error(object.getSession() + " Creating a mail isn't support");
        throw new UnsupportedContentOperationException(USMContentTypesMailErrorCodes.CREATE_ERROR, "Creating a mail is not supported");
    }

    @Override
    public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
        checkTypeToWrite(object, DefaultContentTypes.MAIL_ID);
        Map<String, String> parameters = new HashMap<String, String>();

        parameters.put(CommonConstants.ID, object.getID());
        parameters.put(CommonConstants.FOLDER, object.getOriginalParentFolderID());
        // XXX is it to use? In HTTP Ajax API timestamp isn't use for mail update
        parameters.put(CommonConstants.TIMESTAMP, String.valueOf(timestamp));

        JSONObject requestBody = new JSONObject();
        if (object.isFieldModified(MailConstants.MAIL_UPDATE_FLAGS)) {
            Object oldVal = object.getFieldContent(MailConstants.MAIL_UPDATE_FLAGS);
            int newFlagsValue = (oldVal instanceof Number) ? ((Number) oldVal).intValue() : 0;
            Object newVal = object.getOriginalFieldContent(MailConstants.MAIL_UPDATE_FLAGS);
            int oldFlagsValue = (newVal instanceof Number) ? ((Number) newVal).intValue() : 0;
            int flagsToSet = newFlagsValue & ~oldFlagsValue;
            int flagsToRemove = oldFlagsValue & ~newFlagsValue;
            if (flagsToSet > 0)
                putInJSONObject(requestBody, "set_flags", flagsToSet);
            if (flagsToRemove > 0)
                putInJSONObject(requestBody, "clear_flags", flagsToRemove);
        }
        if (object.isFieldModified(MailConstants.MAIL_UPDATE_COLOR_LABEL)) {
            Object colorValue = object.getFieldContent(MailConstants.MAIL_UPDATE_COLOR_LABEL);
            int colorLabel = (colorValue instanceof Number) ? ((Number) colorValue).intValue() : 0;
            putInJSONObject(requestBody, MailConstants.MAIL_UPDATE_COLOR_LABEL, colorLabel);
        }
        if (requestBody.length() > 0) {
            JSONResult result = _ajaxAccess.doPut(
                CommonConstants.MAIL_PATH,
                CommonConstants.ACTION_UPDATE,
                object.getSession(),
                parameters,
                requestBody);
            if (result.getResultType() == JSONResultType.Error) {
                checkConflictingChange(result.getJSONObject());
                throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER4, result.getJSONObject());
            }
        }
        if (!object.isFieldModified(CommonConstants.PARENTFOLDER_ID))
            return;
        requestBody = new JSONObject();
        putInJSONObject(requestBody, CommonConstants.PARENTFOLDER_ID, object.getParentFolderID());
        JSONResult result = _ajaxAccess.doPut(
            CommonConstants.MAIL_PATH,
            CommonConstants.ACTION_UPDATE,
            object.getSession(),
            parameters,
            requestBody);
        if (result.getResultType() == JSONResultType.Error) {
            checkConflictingChange(result.getJSONObject());
            throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER5, result.getJSONObject());
        } else if (result.getResultType() == JSONResultType.JSONObject && result.getJSONObject().has("data")) {
            // if the email is moved to another folder, update the ID of the object with the new id
            JSONObject data = result.getJSONObject().optJSONObject("data");
            if (data != null) {
                String id = data.optString(CommonConstants.ID, null);
                if (id != null)
                    object.setID(id);
            }
        }
    }

    private static void putInJSONObject(JSONObject object, String key, Object value) {
        try {
            object.put(key, value);
        } catch (JSONException ignored) {
            // can not happen for our constant field names
        }
    }

    @Override
    public void readDataObject(DataObject object, BitSet requestedFields) throws USMException {
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(UNSEEN, ONE);
        parameters.put(VIEW, RAW);
        if("EAS".equals(object.getSession().getProtocol())){
            parameters.put(ATTACH_SRC, ONE);
            BitSet extendedFields = new BitSet();
            extendedFields.or(requestedFields);
            extendedFields.set(_sourceFieldIndex);
            requestedFields = extendedFields;
        }
        readDataObject(object, requestedFields, parameters);
    }

    public JSONObject readStructuredMail(DataObject object, int maxAttachmentSize) throws USMException {
        if (object == null || Toolkit.notProvided(object.getID()) || Toolkit.notProvided(object.getParentFolderID()))
            return new JSONObject();
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Read structured mail with id " + object.getID());
        checkTypeToWrite(object, _contentType.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(UNSEEN, ONE);
        parameters.put(CommonConstants.ID, object.getID());
        parameters.put(CommonConstants.FOLDER, object.getParentFolderID());
        if (maxAttachmentSize >= 0)
            parameters.put(MailConstants.MAX_SIZE, Integer.toString(maxAttachmentSize));
        JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), CommonConstants.ACTION_GET_STRUCTURE, object.getSession(), parameters);
        checkResult(result, JSONResultType.JSONObject);
        JSONObject resultObject = null;
        try {
            resultObject = result.getJSONObject().getJSONObject("data");
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER6,
                object,
                "read structured mail: data  not present",
                e);
        }
        return resultObject;
    }

    public byte[] readMailSource(DataObject object, BitSet requestedFields, boolean ignoreICalAttachments) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Read mail with id " + object.getID() + " with ignore ical attachments param: " + ignoreICalAttachments);
        checkTypeToWrite(object, _contentType.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(UNSEEN, ONE);
        parameters.put(SRC, ONE);
        parameters.put(SAVE, ONE);
        if (ignoreICalAttachments)
            parameters.put("ignorable", "ics");
        parameters.put(CommonConstants.ID, object.getID());
        parameters.put(CommonConstants.FOLDER, object.getParentFolderID());
        parameters.put(CommonConstants.ACTION, CommonConstants.ACTION_GET);
        return _ajaxAccess.getResource(object.getSession(), getOXAjaxAccessPath(), parameters).getData();
    }

    @Override
    protected String getOXAjaxAccessPath() {
        return CommonConstants.MAIL_PATH;
    }

    @Override
    public OXFolderContent readFolderContent(Folder folder, int limit, DataObjectFilter filter) throws USMException {
        return new OXFolderContent(true, readFolderContent(folder, folder.getSession().getFieldFilter(folder.getElementsContentType()), limit));
    }

    @Override
    public DataObject[] readFolderContent(Folder folder, BitSet requestedFields) throws USMException {
        return readFolderContent(folder, requestedFields, Integer.MAX_VALUE);
    }

    private DataObject[] readFolderContent(Folder folder, BitSet requestedFields, int limit) throws USMException {
        if (AbstractTransferHandler.CHECK_ACCESS_RIGHTS_BEFORE_FOLDER_READ && !hasFolderReadRights(folder))
            return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
        Session session = folder.getSession();
        long startDate = session.getStartDate();
        long endDate = session.getEndDate();
        try {
            if (startDate == 0L && endDate == Long.MAX_VALUE)
                return super.readCompleteFolderContent(folder, requestedFields);
            DataObject[] idList = super.readCompleteFolderContent(folder, _getAllBitSet);
            List<DataObject> filteredList = new ArrayList<DataObject>();
            for (DataObject o : idList) {
                Long t = (Long) o.getFieldContent(_receivedDateIndex);
                if (t == null || (t >= startDate && t <= endDate))
                    filteredList.add(o);
            }
            if (filteredList.isEmpty())
                return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
            if (filteredList.size() > limit) {
                // We could add a check here to limit the number of emails read, but we would have to extract the latest sync state, which
                // also costs some time. For now we do nothing special.
            }
            JSONArray ids = new JSONArray();
            for (DataObject o : filteredList) {
                JSONObject element = new JSONObject();
                try {
                    element.put(FOLDER, folder.getID());
                    element.put(CommonConstants.ID, o.getID());
                } catch (JSONException ignored) {
                    // Ignored because it can not happen
                }
                ids.put(element);
            }
            Map<String, String> parameters = new HashMap<String, String>();
            parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
            JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_LIST, session, parameters, ids);
            DataObject[] listOfResults = getListOfResults(folder, requestedFields, result);
            return extractDeletedMails(session, listOfResults);
        } catch (OXCommunicationException e) {
            // catch the OXCommunication exception and than get the access rights
            if (hasFolderReadRights(folder))
                throw e;
            return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
        }
    }

    private DataObject[] extractDeletedMails(Session session, DataObject[] listOfResults) {
        Object o = session.getCustomProperty(MailContentType.FILTER_MARKED_AS_DELETED_MAILS);
        boolean filterMails = (o instanceof Boolean) ? ((Boolean) o) : false;
        if (filterMails) {
            List<DataObject> filteredMails = new ArrayList<DataObject>();
            for (int i = 0; i < listOfResults.length; i++) {
                if ((((Integer) listOfResults[i].getFieldContent(MailConstants.FLAGS)) & 0x2) == 0) {
                    filteredMails.add(listOfResults[i]);
                }
            }
            listOfResults = filteredMails.toArray(new DataObject[filteredMails.size()]);
        }
        return listOfResults;
    }

    public JSONObject convertICalToJSON(DataObject mailObject, String attSeqId, String timezone) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug("Converting iCal attachment to JSONObject for mail: " + mailObject.getID());

        JSONObject conversionObject = createConversionDataSource(mailObject, attSeqId);

        JSONObject dataHandler = new JSONObject();
        try {
            dataHandler.put("identifier", "com.openexchange.ical.json");
            if (timezone != null) {
                JSONArray args = new JSONArray();
                JSONObject timezoneObj = new JSONObject();
                timezoneObj.put("com.openexchange.groupware.calendar.timezone", timezone);
                args.put(timezoneObj);
                dataHandler.put("args", args);
            }
            conversionObject.put("datahandler", dataHandler);
        } catch (JSONException e1) {
            throw generateException(
                USMContentTypesMailErrorCodes.ERROR_ON_DATA_HANDLER_CREATE,
                mailObject,
                "convert iCal to JSON: can not create data handler",
                e1);
        }

        JSONResult result = _ajaxAccess.doPut(
            CommonConstants.CONVERSION_PATH,
            CommonConstants.ACTION_CONVERT,
            mailObject.getSession(),
            new HashMap<String, String>(),
            conversionObject);
        checkResult(result, JSONResultType.JSONObject);
        JSONObject resultObject = null;
        try {
            resultObject = result.getJSONObject().getJSONArray("data").getJSONObject(0);
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER8,
                mailObject,
                "convert iCal to JSON: data  not present",
                e);
        }
        return resultObject;
    }

    public JSONObject analyzeICalAttachment(DataObject mailObject, String attSeqId) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug("Analyzing iCal attachment for mail: " + mailObject.getID());

        JSONObject conversionObject = createITIPDataSource(mailObject, attSeqId);

        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put("dataSource", COM_OPENEXCHANGE_MAIL_ICAL);
        JSONResult result = _ajaxAccess.doPut(
            CommonConstants.ITIP_PATH,
            CommonConstants.ACTION_ANALYZE,
            mailObject.getSession(),
            parameters,
            conversionObject);
        checkResult(result, JSONResultType.JSONObject);
        JSONObject resultObject = null;
        try {
            resultObject = result.getJSONObject().getJSONArray("data").getJSONObject(0);
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER9,
                mailObject,
                "analyze iCal: data  not present",
                e);
        }
        return resultObject;
    }

    public JSONObject updateObjectBasedOnMeetingRequest(DataObject mailObject, String attSeqId, String action) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug("Updating object based on iCal attachment for mail: " + mailObject.getID() + ", action: " + action);

        JSONObject conversionObject = createITIPDataSource(mailObject, attSeqId);

        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put("dataSource", COM_OPENEXCHANGE_MAIL_ICAL);
        JSONResult result = _ajaxAccess.doPut(CommonConstants.ITIP_PATH, action, mailObject.getSession(), parameters, conversionObject);
        checkResult(result, JSONResultType.JSONObject);
        try {
            JSONArray dataArray = result.getJSONObject().getJSONArray("data");
            return dataArray.length() > 0 ? dataArray.getJSONObject(0) : null;
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER9,
                mailObject,
                "analyze iCal: data  not present",
                e);
        }
    }

    public JSONObject respondToMeetingRequest(DataObject mailObject, String attSeqId, int confirmation) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug("Accepting meeting request for mail: " + mailObject.getID());

        JSONObject conversionObject = createITIPDataSource(mailObject, attSeqId);

        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put("dataSource", COM_OPENEXCHANGE_MAIL_ICAL);

        String action = null;
        switch (confirmation) {
        case 1:
            action = CommonConstants.ACTION_ACCEPT;
            break;
        case 2:
            action = CommonConstants.ACTION_DECLINE;
            break;
        case 3:
            action = CommonConstants.ACTION_TENTATIVE;
            break;
        }
        JSONResult result = _ajaxAccess.doPut(CommonConstants.ITIP_PATH, action, mailObject.getSession(), parameters, conversionObject);

        checkResult(result, JSONResultType.JSONObject);
        JSONObject resultObject = null;
        try {
            resultObject = result.getJSONObject().getJSONArray("data").getJSONObject(0);
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER9,
                mailObject,
                "analyze iCal: data  not present",
                e);
        }

        return resultObject;
    }

    private JSONObject createConversionDataSource(DataObject mailObject, String attSeqId) throws OXCommunicationException {
        JSONObject conversionObject = new JSONObject();
        JSONObject dataSource = new JSONObject();
        try {
            dataSource.put("identifier", COM_OPENEXCHANGE_MAIL_ICAL);
            JSONArray args = new JSONArray();
            JSONObject folder = new JSONObject();
            folder.put(COM_OPENEXCHANGE_MAIL_CONVERSION_FULLNAME, mailObject.getParentFolderID());
            args.put(0, folder);
            JSONObject mailid = new JSONObject();
            mailid.put(COM_OPENEXCHANGE_MAIL_CONVERSION_MAILID, mailObject.getID());
            args.put(1, mailid);
            JSONObject seqid = new JSONObject();
            seqid.put(COM_OPENEXCHANGE_MAIL_CONVERSION_SEQUENCEID, attSeqId);
            args.put(2, seqid);
            dataSource.put("args", args);
            conversionObject.put("datasource", dataSource);
        } catch (JSONException e1) {
            throw generateException(
                USMContentTypesMailErrorCodes.ERROR_ON_DATA_SOURCE_CREATE,
                mailObject,
                "convert iCal to JSON: can not create data source",
                e1);
        }
        return conversionObject;
    }

    private JSONObject createITIPDataSource(DataObject mailObject, String attSeqId) throws OXCommunicationException {
        JSONObject conversionObject = new JSONObject();
        try {
            conversionObject.put(COM_OPENEXCHANGE_MAIL_CONVERSION_FULLNAME, mailObject.getParentFolderID());
            conversionObject.put(COM_OPENEXCHANGE_MAIL_CONVERSION_MAILID, mailObject.getID());
            conversionObject.put(COM_OPENEXCHANGE_MAIL_CONVERSION_SEQUENCEID, attSeqId);
        } catch (JSONException e1) {
            throw generateException(
                USMContentTypesMailErrorCodes.ERROR_ON_DATA_SOURCE_CREATE_2,
                mailObject,
                "analyze iCal: can not create data source",
                e1);
        }
        return conversionObject;
    }

    public List<DataObject> searchMails(JSONObject searchCriteria, String folderId, Session session, BitSet requestedFields) throws USMException {
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
        parameters.put(CommonConstants.FOLDER, folderId);
        JSONResult jResult = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_ALL, session, parameters, searchCriteria);
        return buildSearchResultList(session, requestedFields, jResult);
    }

    public List<DataObject> searchMails(String searchText, String folderId, Session session, BitSet requestedFields) throws USMException {
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
        parameters.put(CommonConstants.FOLDER, folderId);
        JSONObject searchCriteria = new JSONObject();
        JSONArray searchArray = new JSONArray();
        try {
            searchCriteria.put("col", -1);
            searchCriteria.put("pattern", searchText);
            searchArray.put(searchCriteria);
        } catch (JSONException ignored) {
            // can not happen
        }
        JSONResult jResult = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_SEARCH, session, parameters, searchArray);
        return buildSearchResultList(session, requestedFields, jResult);
    }

    protected List<DataObject> buildSearchResultList(Session session, BitSet requestedFields, JSONResult jResult) throws OXCommunicationException, InternalUSMException {
        JSONArray jsonar;
        try {
            checkResult(jResult, JSONResultType.JSONObject);
            jsonar = jResult.getJSONObject().getJSONArray(CommonConstants.RESULT_DATA);
        } catch (JSONException e1) {
            LOG.error(session + " Can not read the result data array for search mails", e1);
            return Collections.emptyList();
        }
        int length = jsonar.length();
        if (length == 0)
            return Collections.emptyList();
        List<DataObject> resultMails = new ArrayList<DataObject>();
        for (int j = 0; j < length; j++) {
            JSONArray columnsArray;
            try {
                columnsArray = jsonar.getJSONArray(j);
                DataObject newDataObject = _contentType.newDataObject(session);
                updateDataObjectFromJSONArray(newDataObject, requestedFields, columnsArray);
                resultMails.add(newDataObject);
            } catch (JSONException e) {
                LOG.error(session + " Can not read the result data for search mails", e);
            }
        }
        return resultMails;
    }

    public JSONObject getVacationRule(Session session) throws USMException {
        // GET /ajax/mailfilter?action=list&flag=vacation&session=4108cc5c4bee4d309e10b013ad280a90 HTTP/1.1\r\n
        // RESULT:
        // {"data":[{"active":true,"id":0,"test":{"id":"true"},"rulename":"Abwesenheit","position":0,"actioncmds":[{"id":"vacation","days":"7","text":"This is a test Out Of Office email.","addresses":["user5@ox-p5.microdoc.com","user5@ox.
        Vacation vacation = new Vacation();
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(FLAG, Vacation.RULE_ID);
        JSONResult jResult = _ajaxAccess.doGet(MAILFILTER_AJAX_PATH, CommonConstants.ACTION_LIST, session, parameters);
        checkResult(jResult, JSONResultType.JSONObject);
        try {
            JSONArray jsonar = jResult.getJSONObject().getJSONArray(CommonConstants.RESULT_DATA);
            if(jsonar != null && jsonar.length() > 0)  
                vacation = new Vacation(jsonar.getJSONObject(0));
            return vacation.toJSONObject();
        } catch (JSONException e) {
            LOG.error(session + " Can not read the vacation rule", e);
        }
        return null;
    }

    public void setVacationRule(Session session, JSONObject rule) throws USMException {
        // PUT /ajax/mailfilter?action=new&session=ae231c985599470bb7d55bc393ba3779 HTTP/1.1\r\n
        // {"rulename":"Abwesenheit","active":true,"flags":["vacation"],"test":{"id":"true"},"actioncmds":[{"id":"vacation","days":10,"addresses":["user4@ox-p5.microdoc.com"],"text":"This is a test","subject":"Test Out of Office"}]}
        // RESULT : {"data":0}
        Map<String, String> parameters = new HashMap<String, String>();
        JSONObject requestBody = createVacationRequestBody(rule);
        JSONResult jResult = _ajaxAccess.doPut(MAILFILTER_AJAX_PATH, CommonConstants.ACTION_NEW, session, parameters, requestBody);
        checkResult(jResult, JSONResultType.JSONObject);
    }

    public void updateVacationRule(Session session, JSONObject rule) throws USMException {
        // PUT /ajax/mailfilter?action=update&session=ae231c985599470bb7d55bc393ba3779 HTTP/1.1\r\n
        // [truncated]
        // {"rulename":"Abwesenheit","active":true,"flags":["vacation"],"test":{"id":"true"},"actioncmds":[{"id":"vacation","days":10,"addresses":["user4@ox-p5.microdoc.com","user4@ox.microdoc.de"],"text":"This is a test","subject":"Test
        // RESULT: {"data":null}
        Map<String, String> parameters = new HashMap<String, String>();
        JSONObject requestBody = createVacationRequestBody(rule);
        JSONResult jResult = _ajaxAccess.doPut(MAILFILTER_AJAX_PATH, CommonConstants.ACTION_UPDATE, session, parameters, requestBody);
        checkResult(jResult, JSONResultType.JSONObject);
    }

    public void deleteVacationRule(Session session, JSONObject rule) throws USMException {
        // PUT /ajax/mailfilter?action=delete
        // Request body: An object with the field id.

        Map<String, String> parameters = new HashMap<String, String>();
        JSONObject requestBody = new JSONObject();
        try {
            requestBody.put(Vacation.ID, (new Vacation(rule).getId()));
            JSONResult jResult = _ajaxAccess.doPut(MAILFILTER_AJAX_PATH, CommonConstants.ACTION_DELETE, session, parameters, requestBody);
            checkResult(jResult, JSONResultType.JSONObject);
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesMailErrorCodes.ERROR_ON_VACATION_REQUEST_CREATE_1,
                null,
                "delete Vacation Rule: can not create request Body",
                e);
        }
    }

    private JSONObject createVacationRequestBody(JSONObject rule) throws OXCommunicationException {
        Vacation vacation = new Vacation(rule);
        JSONObject requestBody = new JSONObject();
        try {
            requestBody.put("rulename", Vacation.RULENAME);
            requestBody.put(Vacation.ACTIVE, vacation.isActive());
            requestBody.put("flags", (new JSONArray()).put(Vacation.RULE_ID));
            requestBody.put("id", vacation.getId());
            requestBody.put("test", new JSONObject("{\"id\":\"true\"}"));
            requestBody.put("actioncmds", (new JSONArray()).put(vacation.toActionCmdObject()));
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesMailErrorCodes.ERROR_ON_VACATION_REQUEST_CREATE_2,
                null,
                "Vacation Rule: can not create request Body",
                e);
        }
        return requestBody;
    }

    @Override
    protected void addObjectsLimitParameters(Session session, Map<String, String> parameters) {
        int mailLimit = session.getMailLimit();
        int objectLimit = session.getObjectsLimit();
        int limit = (mailLimit <= 0 || (objectLimit > 0 && objectLimit < mailLimit)) ? objectLimit : mailLimit;
        if (limit > 0)
            addObjectsLimitParameters(limit, _RECEIVED_DATE_COLUMN, parameters);
    }
}
