/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.util;

import java.io.Closeable;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.BitSet;
import org.apache.commons.codec.binary.Base64;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMIllegalStateException;

public class Toolkit {

    /**
     * Makes sure that the given Closeable is closed. Safely handles null parameter, additional IOExceptions thrown while closing the
     * Closeable are silently ignored.
     * 
     * @param c Closeable to be closed (may be null)
     */
    public static void close(Closeable c) {
        if (c != null) {
            try {
                c.close();
            } catch (IOException ignored) {
                // do nothing special on failed close
            }
        }
    }

    public static void close(EncapsulatedConnection c) {
        if (c != null) {
            try {
                c.close();
            } catch (SQLException ignored) {
                // do nothing special on failed close
            } catch (StorageAccessException ignored) {
                // do nothing special on failed close
            }
        }
    }

    public static void close(Connection c) {
        if (c != null) {
            try {
                c.close();
            } catch (SQLException ignored) {
                // do nothing special on failed close
            }
        }
    }

    public static void close(Statement s) {
        if (s != null) {
            try {
                s.close();
            } catch (SQLException ignored) {
                // do nothing special on failed close
            }
        }
    }

    public static void close(ResultSet resultSet) {
        if (resultSet != null) {
            try {
                resultSet.close();
            } catch (SQLException ignored) {
                // do nothing special on failed close
            }
        }
    }

    /**
     * Compares two Objects for equality, safely handles null value for objects
     * 
     * @param o1 first Object of comparison
     * @param o2 second Object of comparison
     * @return true if o1 == null ? o2 == null : o1.equals(o2)
     */
    public static boolean equals(Object o1, Object o2) {
        return (o1 == null) ? o2 == null : o1.equals(o2);
    }

    public static byte[] decodeBase64(String encodedString, String encoding) {
        if (encodedString == null)
            return null;
        try {
            return Base64.decodeBase64(encodedString.getBytes(encoding));
        } catch (UnsupportedEncodingException e) {
            throw new USMIllegalStateException(USMUtilErrorCode.TOOLKIT_UTF8_NOT_SUPPORTED_NUMBER_0, encoding + " not supported");
        }
    }

    public static byte[] decodeBase64(String encodedString) {
        return decodeBase64(encodedString, "UTF-8");
    }

    public static String encodeBase64(byte[] data, String encoding) {
        try {
            return new String(Base64.encodeBase64(data), encoding);
        } catch (UnsupportedEncodingException e) {
            throw new USMIllegalStateException(USMUtilErrorCode.TOOLKIT_UTF8_NOT_SUPPORTED_NUMBER_1, encoding + " not supported");
        }
    }

    public static boolean isBase64Encoded(String queryString) {
        // TODO Newer versions of Base64 directly support isBase64(String), maybe we should update our libraries ?
        try {
            return Base64.isArrayByteBase64(queryString.getBytes("UTF-8"));
        } catch (UnsupportedEncodingException e) {
            throw new USMIllegalStateException(USMUtilErrorCode.TOOLKIT_UTF8_NOT_SUPPORTED_NUMBER_2, "UTF-8 not supported");
        }
    }

    public static String encodeBase64(byte[] data) {
        return encodeBase64(data, "UTF-8");
    }

    public static boolean isNullOrEmpty(String s) {
        if (s != null) {
            for (int i = s.length() - 1; i >= 0; i--) {
                if (!Character.isWhitespace(s.charAt(i)))
                    return false;
            }
        }
        return true;
    }

    /**
     * Answer true if array is not null and not empty
     */
    public static boolean provided(Object[] array) {
        if (array == null) {
            return false;
        }
        return array.length > 0;
    }

    /**
     * Answer true if array is null or empty
     */
    public static boolean notProvided(Object[] array) {
        if (array == null) {
            return true;
        }
        return array.length == 0;
    }

    /**
     * Answer true if the string is not null and not empty
     */
    public static boolean provided(String s) {
        if (s == null) {
            return false;
        }
        return s.length() > 0;
    }

    /**
     * Answer true if the string is null or empty
     */
    public static boolean notProvided(String s) {
        if (s == null) {
            return true;
        }
        return s.length() == 0;
    }

    /**
     * @param <T> the type of the elements which are to be stored in the ArrayList
     * @return a newly created ArrayList which is capable of holding elements of the required type
     */
    public static <T> ArrayList<T> newArrayList() {
        return new ArrayList<T>();
    }

    /**
     * @param <T> the type of the elements which are to be stored in the ArrayList
     * @param initialCapacity the initial capacity of the list
     * @return a newly created ArrayList which is capable of holding elements of the required type
     */
    public static <T> ArrayList<T> newArrayList(int initialCapacity) {
        return new ArrayList<T>(initialCapacity);
    }

    /**
     * @param array Array of objects
     * @param object Object to search for
     * @return true if at least 1 (non-null) element of the array is equal to the given object
     */
    public static boolean contains(Object[] array, Object object) {
        for (Object o : array) {
            if (o != null && o.equals(object))
                return true;
        }
        return false;
    }

    public static BitSet buildFieldBitSet(ContentType contentType, String... fields) {
        BitSet set = new BitSet();
        ContentTypeField[] contentTypeFields = contentType.getFields();
        for (int i = 0; i < contentTypeFields.length; i++) {
            if (contains(fields, contentTypeFields[i].getFieldName()))
                set.set(i);
        }
        return set;
    }
}
