/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.util;

import java.util.Iterator;
import java.util.List;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class JSONToolkit {

    /**
     * Convenience method that searches in a complex JSONObject for a specific sub-JSONObject. If at any point during the traversal an
     * element (including the root) does not exist, this method returns null.
     * 
     * @param root
     * @param path
     * @return
     */
    public static JSONObject getJSONObject(JSONObject root, String... path) {
        return getJSONObject(root, path.length, path);
    }

    public static String getString(JSONObject root, String... path) {
        int limit = path.length - 1;
        JSONObject parent = getJSONObject(root, limit, path);
        if (parent == null || parent.isNull(path[limit]))
            return null;
        return parent.optString(path[limit]);
    }

    private static JSONObject getJSONObject(JSONObject root, int limit, String... path) {
        for (int i = 0; root != null && i < limit; i++)
            root = root.optJSONObject(path[i]);
        return root;
    }

    public static Iterable<String> keys(final JSONObject o) {
        return new Iterable<String>() {

            @Override
            public Iterator<String> iterator() {
                return o.keys();
            }
        };
    }

    public static String[] copyOfKeys(final JSONObject o) {
        List<String> keys = Toolkit.newArrayList(o.length());
        for (String key : keys(o)) {
            keys.add(key);
        }
        return keys.toArray(new String[keys.size()]);
    }

    public static boolean equals(JSONObject o1, JSONObject o2) {
        try {
            if (o1.length() != o2.length())
                return false;
            for (String prop : JSONToolkit.keys(o1)) {
                if (!o2.has(prop))
                    return false;
                Object o1Value = o1.get(prop);
                Object o2Value = o2.get(prop);
                if (o1Value instanceof JSONObject) {
                    if (o2Value instanceof JSONObject) {
                        if (!JSONToolkit.equals((JSONObject) o1Value, (JSONObject) o2Value))
                            return false;
                    } else {
                        return false;
                    }
                } else if (o1Value instanceof JSONArray && o2Value instanceof JSONArray) {
                    if (!JSONToolkit.equals((JSONArray) o1Value, (JSONArray) o2Value))
                        return false;
                } else {
                    if (!o1Value.equals(o2Value))
                        return false;
                }
            }
            return true;
        } catch (JSONException e) {
            return false;
        }
    }

    public static boolean equals(JSONArray a1, JSONArray a2) {
        if (a1.length() != a2.length())
            return false;
        try {
            int length = a1.length();
            for (int i = 0; i < length; i++) {
                Object a1Value = a1.get(i);
                if (!contains(a2, a1Value))
                    return false;
            }
            return true;
        } catch (JSONException e) {
            return false;
        }
    }

    public static boolean contains(JSONArray a, Object o) {
        try {
            int length = a.length();
            for (int i = 0; i < length; i++) {
                Object value = a.get(i);
                if (o instanceof JSONObject && value instanceof JSONObject) {
                    if (JSONToolkit.equals((JSONObject) value, (JSONObject) o))
                        return true;
                } else if (o instanceof JSONArray && value instanceof JSONArray) {
                    if (JSONToolkit.equals((JSONArray) value, (JSONArray) o))
                        return true;
                } else {
                    if (o.equals(value))
                        return true;
                }
            }
            return false;
        } catch (JSONException e) {
            return false;
        }
    }

}
