/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.sync;

import java.util.BitSet;
import com.openexchange.usm.api.contenttypes.folder.OXFolderContent;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.impl.SessionImpl;

public class FolderElementsStorage implements ContentSyncerStorage {

    private final SessionImpl _session;

    private final Folder _folder;

    private final DataObjectFilter _filter;

    private final boolean _storeResult;

    private final long _oldTimestamp;

    private final OXDataCacheID _cacheID;

    private final int _limit;

    public FolderElementsStorage(SessionImpl session, Folder folder, DataObjectFilter filter, boolean storeResult, long oldTimestamp, int limit) {
        _session = session;
        _folder = folder;
        _filter = filter;
        _storeResult = storeResult;
        _oldTimestamp = (filter != null && filter.getOriginalSyncID() != 0L) ? filter.getOriginalSyncID() : oldTimestamp;
        _cacheID = new OXDataCacheID(session, _folder.getID());
        _limit = limit;
    }

    @Override
    public boolean getCurrentServerData(DataObjectSet resultSet) throws USMException {
        try {
            OXFolderContent data = readCurrentFolderContent();
            for (DataObject o : data.getObjects()) {
                o.setParentFolderOwnerID(_folder.getOwnerID());
                o.commitChanges();
                if (_filter == null || _filter.accept(o))
                    resultSet.add(o);
            }
            return data.isComplete();
        } catch (OXCommunicationException e) {
            // Check if folder itself can be read (will throw FolderNotFoundException if not), throw original exception if folder is still
            // accessible
            _folder.getContentType().getTransferHandler().readFolder(_folder, new BitSet());
            throw e;
        }
    }

    private OXFolderContent readCurrentFolderContent() throws USMException {
        if (_limit == Session.NO_LIMIT)
            return new OXFolderContent(true, _folder.getElementsContentType().getTransferHandler().readFolderContent(_folder));
        return _folder.getElementsContentType().getTransferHandler().readFolderContent(_folder, Math.max(_limit, 0) + 10, _filter);
    }

    @Override
    public long storeServerData(long timestamp, DataObjectSet data, boolean updatesPending) throws USMException {
        return _storeResult ? _session.storeCachedDataObjects(_folder.getID(), _oldTimestamp, timestamp, data, updatesPending) : 0L;
    }

    @Override
    public OXDataCacheID getCacheID() {
        return _cacheID;
    }

    @Override
    public String toString() {
        return String.valueOf(_session) + ' ' + _folder.getID() + '(' + _storeResult + ')';
    }

    @Override
    public DataObject findMatchingClientObjectForServerObject(DataObjectSet clientDataObjects, DataObject serverObject) {
        return null;
    }
}
