
package com.openexchange.usm.mapping.storage.sql.impl;

/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

import gnu.trove.map.TIntObjectMap;
import gnu.trove.map.hash.TIntObjectHashMap;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.usm.api.database.DatabaseAccess;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.mapping.storage.FolderIDMappingStorage;
import com.openexchange.usm.api.session.assets.ContextSessionID;
import com.openexchange.usm.util.Toolkit;

/**
 * This is a reduced version of the old Folder id short to long mapping. It will be used to read the mappings which already exist in the db
 * and are reported to clients.
 * 
 * @author <a href="mailto:ldo @microdoc.de">Ljupka Grabsch</a>
 */
public class FolderIDMappingSQL implements FolderIDMappingStorage {

    private static class SessionIdMapping {

        private final Map<String, Integer> _longToShortMap = new HashMap<String, Integer>();

        private final TIntObjectMap<String> _shortToLongMap = new TIntObjectHashMap<String>();

        public SessionIdMapping() {
            super();
        }

        public void addMapping(String longID, int shortID) {
            _longToShortMap.put(longID, shortID);
            _shortToLongMap.put(shortID, longID);
        }

        public Integer getShortID(String longID) {
            return _longToShortMap.get(longID);
        }

        public String getLongID(int shortID) {
            return _shortToLongMap.get(shortID);
        }
    }

    public static final int DB_ACCESS_FAILED = 0x610006;

    public static final int SELECT_FAILED_NUMBER1 = 0x610002;

    private static final String UNIQUE_SESSION_IDENTIFIER = " cid = ? AND usmSessionId = ?";

    private final Log _journal = LogFactory.getLog(FolderIDMappingSQL.class);

    // TODO Best solution would be to move the mapping to the USM-session itself, instead of using a separate storage per session here
    private final Map<ContextSessionID, SessionIdMapping> _sessionMappings = new ConcurrentHashMap<ContextSessionID, SessionIdMapping>();

    private final DatabaseAccess _dbAccess;

    public FolderIDMappingSQL(DatabaseAccess dbAccess) {
        _dbAccess = dbAccess;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.mapping.storage.FolderIDMappingStorage#getLongID(int, int, int)
     */
    @Override
    public String getLongID(int sessionID, int contextID, int shortID) throws USMStorageException, StorageAccessException {
        String longID = getSessionIdMapping(sessionID, contextID).getLongID(shortID);
        if (longID == null) {
            _journal.warn(sessionID + " No long ID for short ID " + shortID);
        }
        return longID;
    }

    private SessionIdMapping readCacheFromDB(int contextID, int sessionID) throws StorageAccessException, USMStorageException {
        SessionIdMapping mapping = new SessionIdMapping();
        EncapsulatedConnection con = _dbAccess.getReadOnly(contextID);
        PreparedStatement statement = null;
        ResultSet result = null;
        try {
            statement = con.prepareStatement("select shortID,LongID from usmIdMapping where " + UNIQUE_SESSION_IDENTIFIER);
            addSessionIdentifierParamsToStmt(statement, contextID, sessionID);
            result = statement.executeQuery();
            while (result.next()) {
                int shortID = result.getInt(1);
                String longID = result.getString(2);
                mapping.addMapping(longID, shortID);
            }
        } catch (SQLException e) {
            String errorMessage = sessionID + " SQL error reading cache from DB";
            _journal.error(errorMessage, e);
            throw new USMStorageException(SELECT_FAILED_NUMBER1, errorMessage, e);
        } finally {
            Toolkit.close(result);
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        return mapping;
    }

    private void addSessionIdentifierParamsToStmt(PreparedStatement statement, int contextID, int sessionID) throws SQLException {
        statement.setInt(1, contextID);
        statement.setInt(2, sessionID);
    }

    @Override
    public Integer findShortID(int sessionID, int contextID, String longID) throws StorageAccessException, USMStorageException {
        // Since v.7.4.1 USM doesn't save any more mappings in a storage system,
        // we just read the cache once:
        return getSessionIdMapping(sessionID, contextID).getShortID(longID);
    }

    private SessionIdMapping getSessionIdMapping(int sessionID, int contextID) throws StorageAccessException, USMStorageException {
        ContextSessionID id = new ContextSessionID(contextID, sessionID);
        SessionIdMapping mapping = _sessionMappings.get(id);
        if (mapping == null) {
            mapping = readCacheFromDB(contextID, sessionID);
            _sessionMappings.put(id, mapping);
        }
        return mapping;
    }
}
