/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.OBJECTIDS;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.startup.ThreadControlService;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.InvalidUUIDException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.ObjectChanges;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.api.session.exceptions.SlowSyncRequiredException;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.json.util.ThreadControlReference;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.UUIDToolkit;

/**
 * Handler for the USM-Ping Command.
 *
 * @author ldo
 */
public class PingHandler extends NormalCommandHandler {

    private static final String _DEFAULT_INBOX_FOLDER_KEY = "json.DefaultInboxFolder";

    private static final String _LAST_INBOX_CHECK_KEY = "json.LastInboxCheck";

    private static final String INTERVAL = "interval";

    private static final String[] REQUIRED_PARAMETERS = { SESSIONID };

    private static final String[] OPTIONAL_PARAMETERS = { OBJECTIDS };

    private DataObjectSet _allCachedFolders;

    private boolean _watchFolderHierarchyChanges = true;

    public PingHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
        _servlet.newPingRequestArrived();
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        ThreadControlService threadControl = ThreadControlReference.getThreadControlService();
        Thread currentThread = Thread.currentThread();
        JSONArray objectUUIDs = _parameters.has(OBJECTIDS) ? getJSONArray(_parameters, OBJECTIDS) : null;
        boolean threadAdded = false;
        try {
            String[] oxIDs = extractOXIdsFromUUIDArray(objectUUIDs);
            threadAdded = null == threadControl ? false : threadControl.addThread(currentThread);
            ObjectChanges changes = getSession().waitForChanges(_watchFolderHierarchyChanges, oxIDs, 0);
            JSONArray resultArray = null;
            if (changes != null) {
                resultArray = toUUIDsJSONArray(changes.getContentChangeParentIDs());
                if (changes.hasFolderStructureChanged()) {
                    resultArray.put(String.valueOf(getSession().getContextUUID()));
                }
            }
            if (shouldPollInbox()) {
                try {
                    resultArray = pollInbox(changes, resultArray);
                } catch (USMException e) {
                    _servlet.getJournal().warn(
                        _session + " Error polling Inbox in ping-request, OLOX2-client and/or user mailbox may be misconfigured",
                        e);
                }
                _session.setCustomProperty(_LAST_INBOX_CHECK_KEY, System.currentTimeMillis());
            }
            JSONObject data = new JSONObject();
            if (resultArray != null && resultArray.length() > 0) {
                data.put(OBJECTIDS, resultArray);
            }
            data.put(INTERVAL, _servlet.getPingInterval());
            return new ResponseObject(ResponseStatusCode.SUCCESS, data);
        } catch (SlowSyncRequiredException e) {
            throw new USMJSONAPIException(ConnectorBundleErrorCodes.PING_SLOW_SYNC_REQUIRED_ERROR,
                    ResponseStatusCode.UNKNOWN_SYNCID, "Unknown SyncID");
        } catch (InvalidUUIDException e) {
            throw USMJSONAPIException.createInvalidUUIDException(e);
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.PING_INTERNAL_ERROR, e);
        } catch (InterruptedException e) {
            throw new USMJSONAPIException(ConnectorBundleErrorCodes.PING_WAIT_FOR_CHANGES_INTERRUPTED_ERROR,
                    ResponseStatusCode.INTERNAL_ERROR, "waiting for changes interrupted", e);
        } catch (JSONException e) {
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.PING_JSON_ERROR, e);
        } finally {
            if (threadAdded && null != threadControl) {
                threadControl.removeThread(currentThread);
            }
        }
    }

    private boolean shouldPollInbox() {
        Object o = _session.getCustomProperty(_LAST_INBOX_CHECK_KEY);
        if (!(o instanceof Long)) {
            return true;
        }
        Long l = (Long) o;
        return System.currentTimeMillis() > (l + _servlet.getInboxPollingInterval());
    }

    public JSONArray pollInbox(ObjectChanges changes, JSONArray resultArray) throws USMException {
        String inboxId = getDefaultInboxId();
        if (inboxId == null || hasDefaultInboxChanges(changes, inboxId)) {
            return resultArray;
        }
        long newestTimestamp = getSession().getNewestTimestamp(inboxId);
        if (newestTimestamp <= 0L || System.currentTimeMillis() - newestTimestamp <= _servlet.getInboxPollingInterval()) {
            return resultArray;
        }
        // clear cached value so that it will be updated from OX configuration
        getSession().setCustomProperty(_DEFAULT_INBOX_FOLDER_KEY, null);
        Folder defaultInbox = (Folder) getAllCachedFolders().get(inboxId);
        if (defaultInbox == null || defaultInbox.getUUID() == null) {
            return resultArray;
        }
        SyncResult syncResult = getSession().syncChangesWithServer(inboxId, newestTimestamp, Session.NO_LIMIT, null, false, null);
        if (syncResult.getChanges().length == 0) {
            return resultArray;
        }
        if (resultArray == null) {
            resultArray = new JSONArray();
        }
        resultArray.put(defaultInbox.getUUID());
        return resultArray;
    }

    private DataObjectSet getAllCachedFolders() throws StorageAccessException, USMStorageException {
        if (_allCachedFolders == null) {
            _allCachedFolders = new DataObjectSet(getSession().getCachedFolders());
        }
        return _allCachedFolders;
    }

    private String getDefaultInboxId() throws USMException {
        Object o = getSession().getCustomProperty(_DEFAULT_INBOX_FOLDER_KEY);
        if (o instanceof String) {
            return (String) o;
        }
        String defaultInboxID = JSONToolkit.getString(getSession().getOXUserConfiguration("modules", "mail", "defaultFolder"), "inbox");
        getSession().setCustomProperty(_DEFAULT_INBOX_FOLDER_KEY, defaultInboxID);
        return defaultInboxID;
    }

    private static boolean hasDefaultInboxChanges(ObjectChanges changes, String inboxId) {
        if (changes == null) {
            return false;
        }

        for (String folderID : changes.getContentChangeParentIDs()) {
            if (folderID.equals(inboxId)) {
                return true;
            }
        }
        return false;
    }

    private UUID getContextUUID() {
        try {
            return getSession().getContextUUID();
        } catch (StorageAccessException e) {
            USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.PING_NO_DB_ACCESS, e);
        } catch (USMStorageException e) {
            USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.PING_SQL_ERROR, e);
        }
        return null; // not reached
    }

    private String[] extractOXIdsFromUUIDArray(JSONArray jsonArray) throws USMJSONAPIException, InvalidUUIDException, StorageAccessException, USMStorageException {
        if (jsonArray == null) {
            return null;
        }
        int length = jsonArray.length();
        List<String> oxIDs = new ArrayList<String>(length);
        UUID contextUUID = getContextUUID();
        for (int i = 0; i < length; i++) {
            String id = getString(jsonArray, i);
            if (isSpecialFolderType(id)) {
                Folder f = (Folder) getAllCachedFolders().get(id);
                if (f != null) {
                    oxIDs.add(f.getID());
                }
            } else {
                UUID uuid = UUIDToolkit.extractUUIDFromString(id);
                if (uuid.equals(contextUUID)) {
                    _watchFolderHierarchyChanges = true;
                } else {
                    Folder f = (Folder) getAllCachedFolders().get(uuid);
                    if (f != null) {
                        oxIDs.add(f.getID());
                    }
                }
            }
        }
        return oxIDs.toArray(new String[oxIDs.size()]);
    }

    private JSONArray toUUIDsJSONArray(String[] idsArray) throws StorageAccessException, USMStorageException {
        DataObjectSet allFolders = getAllCachedFolders();
        JSONArray result = new JSONArray();
        for (int i = 0; i < idsArray.length; i++) {
            if (isSpecialFolderType(idsArray[i])) {
                result.put(idsArray[i]);
            } else {
                Folder cachedFolder = (Folder) allFolders.get(idsArray[i]);
                if (cachedFolder != null && cachedFolder.getUUID() != null && isOfDefaultContentType(cachedFolder.getElementsContentTypeID())) {
                    result.put(cachedFolder.getUUID().toString());
                }
            }
        }
        return result;
    }

    private static boolean isSpecialFolderType(String id) {
        return DefaultContentTypes.GROUPS_ID.equals(id) || DefaultContentTypes.RESOURCES_ID.equals(id);
    }

    private static boolean isOfDefaultContentType(String contentTypeID) {
        return DefaultContentTypes.FOLDER_ID.equals(contentTypeID) || DefaultContentTypes.CONTACTS_ID.equals(contentTypeID) || DefaultContentTypes.CALENDAR_ID.equals(contentTypeID) || DefaultContentTypes.TASK_ID.equals(contentTypeID) || DefaultContentTypes.MAIL_ID.equals(contentTypeID);
    }

    @Override
    protected String[] getOptionalParameters() {
        return OPTIONAL_PARAMETERS;
    }

    @Override
    protected String[] getRequiredParameters() {
        return REQUIRED_PARAMETERS;
    }
}
