/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.database.ox;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.database.CreateTableService;
import com.openexchange.database.DBPoolingExceptionCodes;
import com.openexchange.exception.OXException;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.util.Toolkit;

public class CreateUSMTablesService implements CreateTableService {

    private final static String[] REQUIRED_TABLES = {};

    private static final Log LOG = LogFactory.getLog(CreateUSMTablesService.class);

    @Override
    public void perform(Connection con) throws OXException {
        LOG.debug("Performing CreateTableService 'CreateUSMTablesService'");
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            for (int i = 0; i < TableDefinition.SQL_STATEMENTS.length; i++) {
                stmt.execute(TableDefinition.SQL_STATEMENTS[i]);
            }
            for (int i = 0; i < TableDefinition.UUID_SQL_STATEMENTS.length; i++) {
                stmt.execute(TableDefinition.UUID_SQL_STATEMENTS[i]);
            }
            for (int i = 0; i < TableDefinition.SEQUENCE_TABLES_NAMES.length; i++) {
                TableDefinition.generateSequence(TableDefinition.SEQUENCE_TABLES_NAMES[i], con, false, true);
            }
            LOG.info("CreateTableService 'CreateUSMTablesService' successfully performed!");
        } catch (SQLException e) {
            DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } catch (USMStorageException e) {
            DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } catch (StorageAccessException e) {
            DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            Toolkit.close(stmt);
        }
    }

    @Override
    public String[] requiredTables() {
        return REQUIRED_TABLES;
    }

    @Override
    public String[] tablesToCreate() {
        String[] tablesToCreate = new String[TableDefinition.TABLE_NAMES.length + TableDefinition.SEQUENCE_TABLES_NAMES.length];
        int i = 0;
        for (String table : TableDefinition.TABLE_NAMES) {
            tablesToCreate[i] = table;
            i++;
        }
        for (String table : TableDefinition.SEQUENCE_TABLES_NAMES) {
            tablesToCreate[i] = table;
            i++;
        }
        return tablesToCreate;
    }
}
