/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.datatypes.mail;

import java.io.Serializable;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * 
 * @author ibr
 *
 *    Name 	        Type 	 Value
      id 	        String 	 Object ID (unique only inside the same message)
      content_type 	String 	 MIME type
      content 	    String 	 Content as text. Present only if easily convertible to text.
      filename 	    String 	 Displayed filename (mutually exclusive with content).
      size 	        Number 	 Size of the attachment in bytes.
      disp 	        String 	 Attachment's disposition: null, inline, attachment or alternative. 
 */
public class MailAttachment implements Serializable {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	public static final String OBJECTID = "id";
	public static final String CONTENTTYPE = "content_type";
	public static final String CONTENT = "content";
	public static final String FILENAME = "filename";
	public static final String SIZE = "size";
	public static final String DISP = "disp";
    public static final String CID = "cid";

	//Object ID (unique only inside the same message) 
	private final String _id;
	//MIME type
	private final String _contentType;
	//Content as text. Present only if easily convertible to text. 
	private final String _content;
	//Displayed filename (mutually exclusive with content). 
	private final String _filename;
	//Size of the attachment in bytes. 
	private final int _size;
	//Attachment's disposition: null, inline, attachment or alternative. 
	private final String _disp;
    //Attachment's content-ID. 
    private final String _cid;

	public MailAttachment(String id, String contentType, String content, String filename, int size, String disp) {
		_id = id;
		_contentType = contentType;
		_content = content;
		_filename = filename;
		_size = size;
		_disp = disp;
		_cid = "";
	}

	public MailAttachment() {
		_id = "";
		_contentType = "";
		_content = "";
		_filename = "";
		_size = 0;
		_disp = "";
        _cid = "";
   }

	public MailAttachment(JSONObject jSONObject) throws JSONException {
		_id = jSONObject.has(OBJECTID) ? jSONObject.getString(OBJECTID) : null;
		_contentType = jSONObject.has(CONTENTTYPE) ? jSONObject.getString(CONTENTTYPE) : "";
		_content = jSONObject.has(CONTENT) ? jSONObject.getString(CONTENT) : "";
		_filename = jSONObject.has(FILENAME) ? jSONObject.getString(FILENAME) : "";
		_size = jSONObject.has(SIZE) ? jSONObject.getInt(SIZE) : -1;
		_disp = jSONObject.has(DISP) ? jSONObject.getString(DISP) : "";
        _cid = jSONObject.has(CID) ? jSONObject.getString(CID) : "";
	}

	public String getId() {
		return _id;
	}

	public String getContent_type() {
		return _contentType;
	}

	public String getContent() {
		return _content;
	}

	public String getFilename() {
		return _filename;
	}

	public int getSize() {
		return _size;
	}

	public String getDisp() {
		return _disp;
	}

    public String getCid() {
        return _cid;
    }
	
	@Override
	public int hashCode() {
		final int prime = 37;
		int result = 1;
		result = prime * result + _id.hashCode();
		result = prime * result + _contentType.hashCode();
		result = prime * result + _content.hashCode();
		result = prime * result + _filename.hashCode();
		result = prime * result + _size;
		result = prime * result + _disp.hashCode();
        result = prime * result + _cid.hashCode();
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		MailAttachment other = (MailAttachment) obj;

		if (_id == null && other._id != null)
			return false;
		if (_contentType == null && other._contentType != null)
			return false;
		if (_content == null && other._content != null)
			return false;
		if (_filename == null && other._filename != null)
			return false;
		if (_disp == null && other._disp != null)
			return false;

		if (_id != null && !_id.equals(other._id))
			return false;
		if (_contentType != null && !_contentType.equals(other._contentType))
			return false;
		if (_content != null && !_content.equals(other._content))
			return false;
		if (_filename != null && !_filename.equals(other._filename))
			return false;
		if (_size != other._size)
			return false;
		if (_disp != null && !_disp.equals(other._disp))
			return false;
        if (_cid != null && !_cid.equals(other._cid))
            return false;
		return true;
	}

	@Override
	public String toString() {
		return '{' + OBJECTID + ':' + _id + ';' + CONTENTTYPE + ':' + _contentType + ';' + CONTENT + ':' + _content
				+ ';' + FILENAME + ':' + _filename + ';' + SIZE + ':' + _size + ';' + DISP + ':' + _disp + ';' + CID + ':' + _cid + '}';
	}

	public JSONObject toJSONObject() throws JSONException {
		JSONObject js = new JSONObject();
		js.put(OBJECTID, _id);
		js.put(CONTENTTYPE, _contentType);
		js.put(CONTENT, _content);
		js.put(FILENAME, _filename);
		js.put(SIZE, _size);
		js.put(DISP, _disp);
        js.put(CID, _cid);
		return js;
	}
}
