/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.datatypes.contacts;

import java.io.Serializable;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * This class contains data for distribution list member from HTTP Ajax API (Module contacts)
 * @author ibr
 *
 */
public class ContactDistributionMember implements Serializable {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	public static final String USERID = "id";
	public static final String DISPLAYNAME = "display_name";
	public static final String MAIL = "mail";
	public static final String MAILFIELD = "mail_field";

	// User ID of the member if the member is an existing contact. 
	private final String _userID;
	private final String _displayName;
	// Email address 
	private final String _mail;
	/*
	 * Which email field of an existing contact (if any) is used for the mail field.
	 *  0 	independent contact
	 *  1 	default email field (email1)
	 *  2 	second email field (email2)
	 *  3 	third email field (email3) 
	 */
	private final int _mailField;

	public ContactDistributionMember(String userID, String displayName, String mail, int mailField) {
		_userID = userID;
		_displayName = displayName;
		_mail = mail;
		_mailField = mailField;
	}

	public ContactDistributionMember() {
		_userID = null;
		_displayName = "";
		_mail = "";
		_mailField = 0;
	}

	public ContactDistributionMember(JSONObject jSONObject) throws JSONException {
		_userID = jSONObject.has(USERID) ? jSONObject.getString(USERID) : null;
		_displayName = jSONObject.has(DISPLAYNAME) ? jSONObject.getString(DISPLAYNAME) : "";
		_mail = jSONObject.has(MAIL) ? jSONObject.getString(MAIL) : "";
		_mailField = jSONObject.has(MAILFIELD) ? jSONObject.getInt(MAILFIELD) : -1;
	}

	public String getUser_id() {
		return _userID;
	}

	public String getDisplay_name() {
		return _displayName;
	}

	public int getMail_field() {
		return _mailField;
	}

	public String getMail() {
		return _mail;
	}

	@Override
	public int hashCode() {
		final int prime = 37;
		int result = 1;
		result = prime * result + ((_userID == null) ? 0 : _userID.hashCode());
		result = prime * result + _displayName.hashCode();
		result = prime * result + _mail.hashCode();
		result = prime * result + _mailField;

		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ContactDistributionMember other = (ContactDistributionMember) obj;
		if (_userID == null) {
			if (other._userID != null)
				return false;
		} else if (!_userID.equals(other._userID))
			return false;
		if (!_displayName.equals(other._displayName))
			return false;
		if (!_mail.equals(other._mail))
			return false;
		if (_mailField != other._mailField)
			return false;

		return true;
	}

	public JSONObject toJSONObject() throws JSONException {
		JSONObject result = new JSONObject();
		result.put(USERID, _userID);
		result.put(DISPLAYNAME, _displayName);
		result.put(MAIL, _mail);
		result.put(MAILFIELD, _mailField);
		return result;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(100);
		sb.append('{');
		boolean appendComma = false;
		appendComma = appendField(sb, USERID, _userID, appendComma);
		appendComma = appendField(sb, DISPLAYNAME, _displayName, appendComma);
		appendComma = appendField(sb, MAIL, _mail, appendComma);
		if (appendComma)
			sb.append(',');
		sb.append("\"" + MAILFIELD + "\":\"").append(_mailField).append("\"}");
		return sb.toString();
	}

	private boolean appendField(StringBuilder sb, String name, String value, boolean appendComma) {
		if (value == null || value.length() > 0)
			return appendComma;
		if (appendComma)
			sb.append(',');
		sb.append('"').append(name).append("\":\"").append(value).append('"');
		return true;
	}
}
