/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.session;

import java.util.Map;
import com.openexchange.annotation.NonNull;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.ox.event.OXEventListener;

/**
 * Service that provides access to a Session. A Session is uniquely
 * identified by the user and the device.
 * 
 * @author afe
 *
 */
public interface SessionManager extends OXEventListener {
	/**
	 * Request a specific Session. When the Session is returned, it
	 * is guaranteed that the Session either is correctly authenticated
	 * on the OX server or still in the Cache from a previous access
	 * (and was therefore previously authenticated).<br>
	 * The Session will be initialized so that all currently available
	 * ContentTypes with all of their fields are synchronized.
	 * 
	 * @param user user name / login
	 * @param password password for the user
	 * @param protocol the name of the Sync protocol (ex. EAS)
	 * @param device unique device ID
	 * @return unique Session (regarding user and device)
	 * @throws USMException if session can not be retrieved (e.g. login fails or a communication error occurred)
	 */
    @Deprecated
	Session getSession(String user, String password, String protocol, String device) throws USMException;

	/**
	 * Request a specific Session. When the Session is returned, it
	 * is guaranteed that the Session either is correctly authenticated
	 * on the OX server or still in the Cache from a previous access
	 * (and was therefore previously authenticated). A SessionInitializer
	 * is called after it is determined that the session has been created
	 * for the first time (i.e. no persistent information was found in the
	 * database). This SessionInitializer is called before the Session is
	 * returned and can be used to configure the Session, e.g. to set used
	 * ContentTypes and their relevant fields.<br>
	 * If a SessionInitializer is provided, a new Session will be initialized
	 * with no ContentType set (i.e. all ContentTypes are completely filtered
	 * out) and the SessionInitializer has to provide the correct
	 * initialization.<br>
	 * If no SessionInitializer is provided (<code>null</code>), the Session
	 * will be initialized so that all currently available ContentTypes with
	 * all of their fields are synchronized.
	 * 
	 * @param user
	 * @param password
	 * @param protocol 
	 * @param device
	 * @param initializer
	 * @param clientIP
	 * @param xHeaders
	 * @return
	 * @throws USMException
	 */
	@NonNull Session getSession(String user, String password, String protocol, String device, SessionInitializer initializer,
			String clientIP, Map<String, String> xHeaders) throws USMException;

	boolean existsSessionInDB(String user, String password, String protocol, String device) throws USMException;

	/**
	 * This method enables waiting for the specified timeout in the Session.waitForChanges() method. If waiting is not enabled, waitForChanges() will
	 * always ignore the timeout and return immediately.
	 * 
	 * @param protocol
	 */
	void enableWaitingForChanges(String protocol);

	/**
	 * This method disables waiting in the Session.waitForChanges() method. Any threads that are currently waiting will get notified and will stop
	 * waiting. Any future calls to waitForChanges() will return immediately (until enableWaitingForChanges is called again).
	 * 
	 * @param protocol
	 */
	void disableWaitForChanges(String protocol);

	/**
	 * This method removes the given session from the cache.
	 * 
	 * @param session
	 */
    void removeSession(Session session);
}
